#include "config.h"
#include "WebPluginPDFImplUnix.h"
#include "npruntime_impl.h" // _NPN_*
#include "kjs/JSLock.h"
#include "kjs_proxy.h"
#include "CString.h"
#include <assert.h>
#include "Document.h"
#include "runtime_root.h"
#include "NP_jsobject.h"
#include "HTMLPlugInElement.h"
#include "Element.h" // why isn't the compiler picking this up from ApolloWebKitPluginImpl.h's includes?
#include "ResourceLoader.h"
#include "Event.h"
#include "EventNames.h"
#include "EventListener.h"
#include "Page.h"
#include "Settings.h"
#include "FrameTree.h"
#include "FrameView.h"
#include "IScrollViewApolloImpl.h"
#include <gdk/gdkx.h>
#include "gtk2xtbin.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dlfcn.h>
#include <dirent.h>

// to help bring similarity with WebNetscapePluginPackage
#define NPP_New             m_pluginFuncs.newp
#define NPP_Destroy         m_pluginFuncs.destroy
#define NPP_SetWindow       m_pluginFuncs.setwindow
#define NPP_NewStream       m_pluginFuncs.newstream
#define NPP_DestroyStream   m_pluginFuncs.destroystream
#define NPP_StreamAsFile    m_pluginFuncs.asfile
#define NPP_WriteReady      m_pluginFuncs.writeready
#define NPP_Write           m_pluginFuncs.write
#define NPP_Print           m_pluginFuncs.print
#define NPP_HandleEvent     m_pluginFuncs.event
#define NPP_URLNotify       m_pluginFuncs.urlnotify
#define NPP_GetValue        m_pluginFuncs.getvalue
#define NPP_SetValue        m_pluginFuncs.setvalue
#define NPP_GetJavaClass    m_pluginFuncs.javaClass

// Adobe Reader Specific
#define MAX_VERSION_LEN     256
#define ADOBE_NPPDF_HACK    1
#define PRINT_WINDOW_SIZES    0

//Windows implementation of PluginPDFImpl

static const char apolloPDFWindowClass[] = "ApolloRuntimePDFContentWindow";

namespace WebKitApollo
{
    //    Statics that need to be initialized only once when we first find and load the plug-in library
    void *             PluginPDFImplUnix::m_hInst = NULL;
    WebCore::String    PluginPDFImplUnix::m_path;
    NPNetscapeFuncs    PluginPDFImplUnix::m_browserFuncs;
    NPPluginFuncs      PluginPDFImplUnix::m_pluginFuncs;

    ApolloEntryParamRec PluginPDFImplUnix::m_apolloEntryFuncs;
    PluginPDFImplUnix*  PluginPDFImplUnix::sPluginPDFBeingInitialized = NULL;

    unsigned int        PluginPDFImplUnix::m_instanceCount = 0;
    NPP_ShutdownProcPtr PluginPDFImplUnix::mNP_Shutdown = NULL;

    //    Misc strings for looking at how we're embedded in the HTML page
    WebCore::QualifiedName* PluginPDFImplUnix::mEmbedTag = NULL;
    WebCore::QualifiedName* PluginPDFImplUnix::mObjectTag = NULL;
    WebCore::AtomicString*  PluginPDFImplUnix::mNullAtom = NULL;

    void CFile::initAsTemp()
    {
        remove();

        char    path[MAX_PATH];
        strcpy(path, "/tmp/pluginPDFXXXXXX");
        if(!mktemp(path))
            return;
        else
            strcpy(m_filePath, path);

        m_file = fopen((char *)m_filePath, "w+b");
        ASSERT(m_file);
    }

    void CFile::close()
    {
        if(m_file)
        {
            int ret = fclose(m_file);
            ASSERT(ret == 0);

            m_file = 0;
        }
    }

    void CFile::remove()
    {
        close();

        if(m_filePath[0])
        {
            int ret = unlink((char*)m_filePath);
            ASSERT(!ret);

            m_filePath[0] = 0;
        }
    }

    void CFile::write(const char* bytes, size_t numBytes)
    {
        if(m_file)
        {
            size_t ret = fwrite(bytes, 1, numBytes, m_file);

            ASSERT(ret == numBytes);
        }
    }

    PluginPDFImplUnix::PluginPDFImplUnix( WebCore::Element* const element,
                                          const WebCore::KURL& url,
                                          const Vector<WebCore::String>& paramNames, 
                                          const Vector<WebCore::String>& paramValues,
                                          const WebCore::String& mimeType,
                                          WebFrameImpl* const pWebFrameOwner )
        :   PluginNP( element, url, paramNames, paramValues, mimeType, pWebFrameOwner ),
            m_isStarted( false ), m_gtkWindow(NULL), m_gtkXtBin(NULL), m_pluginNeedsXEmbed(false), m_isTransparent(false), m_windowScriptNPObject(NULL),
            m_pluginFunctionCallDepth(0), m_shouldStopSoon(false), m_instance(&m_instanceStruct),
            m_specifiedHeight(-1), m_specifiedWidth(-1), m_cAttributes(NULL), m_cValues(NULL),
            m_docStream(NULL),
            m_sType(NP_SEEK),
            m_instanceIsDead(false),
            m_delayedPostTimer(this, &PluginPDFImplUnix::delayedPostTimerFired),
            m_inClientReceivedResponse(false), 
            m_clientReceivedResponseResourceHandle(NULL),
            m_useDoublePOSTRequest(false),
            m_delayedGetTimer(this, &PluginPDFImplUnix::delayedGetTimerFired),
            m_notifyData(NULL)
    {
        memset(&m_window, 0, sizeof(m_window));
        memset(&m_instanceStruct, 0, sizeof(m_instanceStruct));
        m_instanceStruct.ndata = this;

        // XXX - is this correct?  looks weird
        WebCore::KURL urlAgent;
        m_userAgent = mWebFrameOwner->frame()->loader()->userAgent(urlAgent).latin1();

        if (mWebFrameOwner->frame()->document()->isPluginDocument())
            m_mode = NP_FULL;
        else
            m_mode = NP_EMBED;

        if(!mNullAtom)
            mNullAtom = new WebCore::AtomicString;

        if(!mEmbedTag)
            mEmbedTag = new WebCore::QualifiedName(*mNullAtom, "embed", "http://www.w3.org/1999/xhtml");

        if(!mObjectTag)
            mObjectTag = new WebCore::QualifiedName(*mNullAtom, "object", "http://www.w3.org/1999/xhtml");

        start();
    }

    void PluginPDFImplUnix::updatePluginWindow(bool showPlugins, bool showWindowedPlugins)
    {

        // TODO_AIRLinux : See if this is required
        // if(!m_gtkWindow)
        //   updateWindow();


        if(!m_gtkWindow)
            return;

        if (showWindowedPlugins) {
            if(m_gtkWindow && GTK_IS_WIDGET(m_gtkWindow))
                gtk_widget_show(m_gtkWindow);
        }
        else {
            if(m_gtkWindow && GTK_IS_WIDGET(m_gtkWindow))
                gtk_widget_hide(m_gtkWindow);
        }

        //    All events, including paint, may indicate that we have to
        //    resize.  This is because our drawing model (nested HWNDs)
        //    doesn't match the Apollo drawing model (one HWND).  Setting
        //    us to the same geometry will make this call do the window-
        //    relative math and all the clipping stuff, and call back
        //    frameGeometryHasChanged if it really has.
        setFrameGeometry( frameGeometry() );
    }

    NPError PluginPDFImplUnix::PopulateWindowAndWSInfoForNPWindow(NPWindow* window) 
    {
        GdkWindow *gdkWindow = m_gtkWindow->window;
        if (!gdkWindow)
            return NPERR_GENERIC_ERROR;

        window->ws_info = (NPSetWindowCallbackStruct *)malloc(sizeof(NPSetWindowCallbackStruct));
        
        if(!m_pluginNeedsXEmbed)
        {
            if(!m_gtkXtBin)
                m_gtkXtBin = gtk_xtbin_new(gdkWindow, 0);
            if(!m_gtkXtBin)
                return NPERR_GENERIC_ERROR;
            gtk_widget_set_usize(m_gtkXtBin, window->width, window->height);
            gtk_widget_show(m_gtkXtBin);
        }

        NPSetWindowCallbackStruct *ws;
        ws = (NPSetWindowCallbackStruct *)window->ws_info;

        // fill in window info structure
        ws->type = 0; 
        ws->depth = gdk_window_get_visual(gdkWindow)->depth;

        if (!m_pluginNeedsXEmbed)
            ws->display = GTK_XTBIN(m_gtkXtBin)->xtdisplay;
        else
            ws->display = GDK_WINDOW_XDISPLAY(gdkWindow);

        ws->visual = GDK_VISUAL_XVISUAL(gdk_window_get_visual(gdkWindow));
        ws->colormap = GDK_COLORMAP_XCOLORMAP(gdk_window_get_colormap(gdkWindow));

        XFlush(ws->display);


        if (!m_pluginNeedsXEmbed) {
            // And now point the NPWindow structures window to the actual X window
            window->window = (void*)GTK_XTBIN(m_gtkXtBin)->xtwindow;
            gtk_xtbin_resize(m_gtkXtBin, window->width, window->height);
        }

        return NPERR_NO_ERROR;
    }

    static GtkWidget* GetChildWindow(GtkWidget* parentWindow)
    {
        GtkWidget *child;
        GList *children = gtk_container_get_children(GTK_CONTAINER(parentWindow));
        if(children)
        {
            GList *firstChild = children;
            if(firstChild)
            {
                child = (GtkWidget*)(firstChild->data);
                g_list_free(children);
                return child;
            }
        }
        return NULL;
    } 

    void PluginPDFImplUnix::CallNPPSetWindowIfAppropriate()
    {
        //    We don't want to create a child window for the plug-in until
        //    we get a URL.  In the case that an OBJECT element was created
        //    for us but didn't have a "data" property, we'll let our
        //    m_gtkWindow WM_PAINT routines draw grey.
        ASSERT(m_gtkWindow);

        if (m_gtkWindow != NULL && !getUrl().isEmpty() && !m_instanceIsDead )
        {
            GtkWidget *gtkChild = NULL; //GetChildWindow(m_gtkWindow);
            if (gtkChild == NULL)
            {
                m_window.type = NPWindowTypeWindow;

                if(m_pluginNeedsXEmbed)
                {
                    m_window.window = (void*)gtk_socket_get_id(GTK_SOCKET(m_gtkWindow));
                }

                PopulateWindowAndWSInfoForNPWindow(&m_window);

                willCallPlugInFunction();
                NPError npErr = NPP_SetWindow(m_instance, &m_window);
                didCallPlugInFunction();

                ASSERT(npErr == NPERR_NO_ERROR);
            }
        }
    }

    void PluginPDFImplUnix::updateWindow()
    {
        // If we never loaded the plug-in, or this instance is dead, bail
        if (m_hInst == NULL || m_instanceIsDead)
            return; 

        if(!m_gtkWindow && m_window.height == 0 && m_window.width == 0) // XXX - probably don't want to CreateWindow with 0x0 rect
            return;

        // Is this the first time we have been called?
        if (m_gtkWindow == NULL)
        {
            // Unix-specific code to create a window here that is like the window provided by Firefox.
            // This is because the current integratation with Reader uses the browser code path through the 
            // Netscape/Firefox plug-in, all of which expects a platform specific window handle that is
            // appropriately sized. 
            // XXX ToDo in future version:  We'd like to get rid of windows completely out of this integration
            // path and have PDF render in the flash pipeline.  

            GtkContainer* gtkContainer = GTK_CONTAINER(mWebFrameOwner->frame()->view()->containingWindow());

            // Any windows call can dispatch events such that by the return, our instance could be dead.
            if (gtkContainer == NULL)
                return;

            if(m_pluginNeedsXEmbed)
            {
                m_gtkWindow = gtk_socket_new();

                gtk_container_add(gtkContainer, m_gtkWindow);

                gtk_widget_realize(m_gtkWindow);
                gtk_widget_show(m_gtkWindow);

                g_object_set_data(G_OBJECT(m_gtkWindow), "kWebPluginViewProperty", this);
            }
            else
            {
                m_gtkWindow = gtk_layout_new(NULL, NULL);

                gtk_container_add(gtkContainer, m_gtkWindow);

               /* We must set this window's size requisition.  Otherwise when a
                  resize is queued (when gtk_widget_queue_resize is called) the
                  window will snap to its default requisition of 0x0.  If we omit
                  this call, Frames and Dialogs shrink to degenerate 1x1 windows
                  when their resizable property changes. */
                gtk_widget_set_size_request(m_gtkWindow, m_window.width, m_window.height);
                gtk_widget_set_uposition (m_gtkWindow, m_window.x, m_window.y);

                gtk_widget_realize(m_gtkWindow);
                gtk_widget_show(m_gtkWindow);
                g_object_set_data(G_OBJECT(m_gtkWindow), "kWebPluginViewProperty", this);
            }

            CallNPPSetWindowIfAppropriate();
        }
        else
        {
#if PRINT_WINDOW_SIZES
            printf("m_window    %d   %d   %d   %d\n", m_window.x, m_window.y, m_window.width, m_window.height);
#endif

            const WebCore::IntRect clipRect(m_window.clipRect.left, 
                    m_window.clipRect.top, 
                    m_window.clipRect.right - m_window.clipRect.left,
                    m_window.clipRect.bottom - m_window.clipRect.top);

            int leftClip = clipRect.x() - m_window.x;
            int topClip = clipRect.y() - m_window.y;

            CallNPPSetWindowIfAppropriate();
            // move the top of the Acrobat window by the scroll amount
            GtkWidget* gtkChild = GetChildWindow(m_gtkWindow);
            if(gtkChild)
            {
#if PRINT_WINDOW_SIZES
                printf("acrobat window    %d   %d   %d   %d\n", -leftClip, -topClip, m_window.width, m_window.height);
#endif
                gdk_window_move_resize(gtkChild->window, -leftClip, -topClip, m_window.width, m_window.height);
            }

            // move the clipped window
#if PRINT_WINDOW_SIZES
            printf("clipRect (same as intersect)  %d   %d   %d   %d\n", clipRect.x(), clipRect.y(), clipRect.width(), clipRect.height());
#endif
            // TODO_AirLinux : gtk_widget_set_uposition is deprecated. Change this.
            gtk_widget_set_size_request(m_gtkWindow, clipRect.width(), clipRect.height());
            if(clipRect.width() && clipRect.height())
                gtk_widget_set_uposition (m_gtkWindow, clipRect.x(), clipRect.y());
        }
    }

    void PluginPDFImplUnix::frameGeometryHasChanged( const WebCore::IntRect &oldRect, const WebCore::IntRect& windowRect, const WebCore::IntRect& windowClipRect )
    {
        // Notes:
        // All coordinates are with respect to the top left of the outermost window (0,0)
        // windowRect - Complete PDF area
        // windowClipRect - Complete visible window area in which PDF areas can be seen (so excludes the URL specifying area). 
        //                  It does not change till the window is moved.
        // clipRect - Visible PDF area
        // m_window.clipRect - same as the clipRect
#if PRINT_WINDOW_SIZES
        printf("windowRect    %d   %d   %d   %d\n", windowRect.x(), windowRect.y(), windowRect.width(), windowRect.height());
        printf("windowClipRect    %d   %d   %d   %d\n", windowClipRect.x(), windowClipRect.y(), windowClipRect.width(), windowClipRect.height());
#endif
        m_window.x = windowRect.x();
        m_window.y = windowRect.y();
        m_window.width = windowRect.width();
        m_window.height = windowRect.height();
        
        WebCore::IntRect clipRect = windowRect;
        clipRect.intersect(windowClipRect);
#if PRINT_WINDOW_SIZES
        printf("clipRect (intersect)    %d   %d   %d   %d\n", clipRect.x(), clipRect.y(), clipRect.width(), clipRect.height());
#endif

        // m_window.clipRect only accepts positive values :(
        m_window.clipRect.left = clipRect.x();
        m_window.clipRect.top = clipRect.y();
        m_window.clipRect.right = m_window.clipRect.left + clipRect.width();
        m_window.clipRect.bottom = m_window.clipRect.top + clipRect.height();
#if PRINT_WINDOW_SIZES
        printf("m_window.clipRect   %d   %d   %d   %d\n", m_window.clipRect.left, m_window.clipRect.top, 
                                                                    m_window.clipRect.right, m_window.clipRect.bottom);
#endif

        updateWindow();
    }

    void PluginPDFImplUnix::initStructs()
    {
        memset(&m_pluginFuncs, 0, sizeof(m_pluginFuncs));
        m_pluginFuncs.size = sizeof(m_pluginFuncs);

        memset(&m_browserFuncs, 0, sizeof(m_browserFuncs));
        m_browserFuncs.version = NP_VERSION_MINOR;
        m_browserFuncs.size = sizeof(m_browserFuncs);
        m_browserFuncs.geturl = NPN_GetURL;
        m_browserFuncs.posturl = NPN_PostURL;
        m_browserFuncs.requestread = NPN_RequestRead;
        m_browserFuncs.newstream = NPN_NewStream;
        m_browserFuncs.write = NPN_Write;
        m_browserFuncs.destroystream = NPN_DestroyStream;
        m_browserFuncs.status = NPN_Status;
        m_browserFuncs.uagent = NPN_UserAgent;
        m_browserFuncs.memalloc = NPN_MemAlloc;
        m_browserFuncs.memfree = NPN_MemFree;
        m_browserFuncs.memflush = NPN_MemFlush;
        m_browserFuncs.reloadplugins = NPN_ReloadPlugins;
        m_browserFuncs.geturlnotify = NPN_GetURLNotify;
        m_browserFuncs.posturlnotify = NPN_PostURLNotify;
        m_browserFuncs.getvalue = NPN_GetValue;
        m_browserFuncs.setvalue = NPN_SetValue;
        m_browserFuncs.invalidaterect = NPN_InvalidateRect;
        m_browserFuncs.invalidateregion = NPN_InvalidateRegion;
        m_browserFuncs.forceredraw = NPN_ForceRedraw;
        m_browserFuncs.getJavaEnv = NPN_GetJavaEnv;
        m_browserFuncs.getJavaPeer = NPN_GetJavaPeer;

        m_browserFuncs.releasevariantvalue = _NPN_ReleaseVariantValue;
        m_browserFuncs.getstringidentifier = _NPN_GetStringIdentifier;
        m_browserFuncs.getstringidentifiers = _NPN_GetStringIdentifiers;
        m_browserFuncs.getintidentifier = _NPN_GetIntIdentifier;
        m_browserFuncs.identifierisstring = _NPN_IdentifierIsString;
        m_browserFuncs.utf8fromidentifier = _NPN_UTF8FromIdentifier;
        m_browserFuncs.createobject = _NPN_CreateObject;
        m_browserFuncs.retainobject = _NPN_RetainObject;
        m_browserFuncs.releaseobject = _NPN_ReleaseObject;
        m_browserFuncs.invoke = _NPN_Invoke;
        m_browserFuncs.invokeDefault = _NPN_InvokeDefault;
        m_browserFuncs.evaluate = _NPN_Evaluate;
        m_browserFuncs.getproperty = _NPN_GetProperty;
        m_browserFuncs.hasproperty = _NPN_HasProperty;
        m_browserFuncs.setproperty = _NPN_SetProperty;
        m_browserFuncs.removeproperty = _NPN_RemoveProperty;
        m_browserFuncs.setexception = _NPN_SetException;

        memset(&m_apolloEntryFuncs, 0, sizeof(m_apolloEntryFuncs));
        m_apolloEntryFuncs.size = sizeof(m_apolloEntryFuncs);
        m_apolloEntryFuncs.apolloVersion = APOLLO_ENTRY_PARAM_VERSION;
        m_apolloEntryFuncs.beginModal = NPA_BeginModal;
        m_apolloEntryFuncs.endModal = NPA_EndModal;
    }

    void PluginPDFImplUnix::freeAttributeKeysAndValues()
    {
        ASSERT((m_cAttributes && m_cValues) || (!m_cAttributes && !m_cValues));

        if(m_cAttributes && m_cValues)
        {
            unsigned int i;

            for (i = 0; i < mParamNames.size(); i++) 
            {
                free(m_cAttributes[i]);
                free(m_cValues[i]);
            }

            free(m_cAttributes);
            m_cAttributes = NULL;

            free(m_cValues);
            m_cValues = NULL;
        }
    }

    void PluginPDFImplUnix::dealloc()
    {
        freeAttributeKeysAndValues();
    }

    PluginPDFImplUnix::~PluginPDFImplUnix()
    {

        // Clean up from delayed posting
        if (m_delayedPostTimer.isActive())
            m_delayedPostTimer.stop();
        if (m_delayedGetTimer.isActive())
            m_delayedGetTimer.stop();

        // Fix for bug 1827864 - Application crashes when a page is reloaded,  where some bidirectional javascript 
        // has been executed.
        // When the PDF object is created, script objects are created in which some are created by Acrobat while 
        // others are created by us. In all objects, the allocaters and deallocaters are provided by the creators.
        // So the objects which are created by Acrobat can be deallocated only till Acrobat is alive. Earlier, we
        // were not deleting those objects here and as a result they were tried to be deleted later in the cleanup
        // using function WebCore::Frame::clearScriptObjects. Because Acrobat did not exist till then, it resulted
        // into a crash. 
        // Now we are explicitly cleaning up all the objects created by this plugin before Acrobat goes off.

        mWebFrameOwner->frame()->cleanupScriptObjectsForPlugin(this);

        stop(NPRES_DONE, true);

        if (m_windowScriptNPObject) 
        {
            // Call _NPN_DeallocateObject() instead of _NPN_ReleaseObject() so that we don't leak if a plugin fails to release the window
            // script object properly.
            // This shouldn't cause any problems for plugins since they should have already been stopped and destroyed at this point.
            _NPN_DeallocateObject(m_windowScriptNPObject);
            m_windowScriptNPObject = 0;
        }

        dealloc();
    }

    static inline bool FileExists(const char *fileName)
    {
        struct stat buf;
        if(stat(fileName, &buf) == -1)
            return false;
        else
            return true;
    }

    // this was originally based on firefox's plug-in finding code.  But that doesn't work
    // correctly for minor releases :(
    static WebCore::String FindAcrobat(const char *baseFolder)
    {
        char    *minVersionRequired = "8.1.1";
        WebCore::String latestFind;

        // Check all the directories in the base folder provided
        // TODO: locale specific changes for localized input
        DIR* baseDir = opendir(baseFolder);
        struct dirent* dirEntry;
        if (baseDir)
        {
            while ( (dirEntry = readdir(baseDir)) != NULL )
            {
                // Each dirEntry here would correspond to a different version of Reader

                // First check the AcroVersion file and see if the version is proper
                char readerVersionFile[MAX_PATH];
                strncpy(readerVersionFile, baseFolder, MAX_PATH);
                strncat(readerVersionFile, "/", MAX_PATH);
                strncat(readerVersionFile, dirEntry->d_name, MAX_PATH);
                strncat(readerVersionFile, "/Reader/AcroVersion", MAX_PATH);

                FILE *versionFD = fopen(readerVersionFile, "r");
                if(versionFD)    // File exists
                {
                    char versionString[MAX_VERSION_LEN];
                    fread(&versionString, MAX_VERSION_LEN, 1, versionFD);
                    fclose(versionFD);
                    // TODO : numeric comparison
                    if(strcmp(versionString, minVersionRequired) < 0)    
                        continue;
                }
                else
                    continue;

                // Now create the plugin path
                char pluginFilePath[MAX_PATH];
                strncpy(pluginFilePath, baseFolder, MAX_PATH);
                strncat(pluginFilePath, "/", MAX_PATH);
                strncat(pluginFilePath, dirEntry->d_name, MAX_PATH);
                strncat(pluginFilePath, "/Browser/intellinux/nppdf.so", MAX_PATH);
                // TODO : TBD :  Later, give AIR instead of Browser in the folder path

                latestFind = WebCore::String(pluginFilePath);
                break;
            }
        }
        closedir(baseDir);
    
        return latestFind;
    }


    static bool FindAcrobatInUserPath (char *readerPath)
    {
        bool    found = false;

        // We would go and get the PATH set by the user and see if there exists acroread 
        gchar    *absPath = g_find_program_in_path ("acroread");
    
        if(!absPath)
            return found;

        gchar    linkPath[MAX_PATH];
        strncpy(linkPath, absPath, MAX_PATH);
        while(1) 
        {
            struct stat    fileStat;
            char         buf[MAX_PATH];
    
            lstat(linkPath, &fileStat);
            if((fileStat.st_mode & S_IFLNK) == S_IFLNK) 
            {
                // If the link is invalid
                if(readlink(linkPath, buf, MAX_PATH) == -1)
                    break;
    
                // If the link is valid, go further
                strncpy(linkPath, buf, MAX_PATH);
            } 
            else 
            {
                // If the path is not a link, go 3 levels up to get to the base dir
                char *till_bin = g_path_get_dirname (linkPath);
                char *till_reader = g_path_get_dirname (till_bin);
                char *till_adobe = g_path_get_dirname (till_reader);
    
                strncpy(readerPath, till_adobe, MAX_PATH);    
                found = true;
    
                free(till_bin);
                free(till_reader);
                free(till_adobe);

                break;
            }
        }
        
        free(absPath);
        return found;
    }


    static void FindAcrobats(Vector<WebCore::String> &paths)
    {
        WebCore::String path;

        char readerPathBase[MAX_PATH];
        if(FindAcrobatInUserPath(readerPathBase))
        {
            path = FindAcrobat(readerPathBase);
            if(!path.isEmpty())
                paths.append(path);
        }

        path = FindAcrobat("/opt/Adobe");
        if(!path.isEmpty())
            paths.append(path);
    }

    bool PluginPDFImplUnix::PluginExists()
    {
        if(!m_path.isEmpty())
            return true;

        static Vector<WebCore::String> s_paths;
        static bool sb_Initialized = false;

        if(!sb_Initialized)
        {
            sb_Initialized = true;

#if _DEBUG
            s_paths.append("/home/romil/reader_rel5_new/Plugins/EWH/Netscape/Build/l86_dev/nppdf.so");
            s_paths.append("/ashutosh/nppdf.so");
#endif

            FindAcrobats(s_paths);
        }
        
        for(Vector<WebCore::String>::iterator i = s_paths.begin(); i != s_paths.end(); ++i)
        {
            if(FileExists(i->utf8().data()))
            {
                m_path = WebCore::String(i->utf8().data());
                break;
            }
        }

        return !m_path.isEmpty();
    }


    bool PluginPDFImplUnix::load( PluginPDFImplUnix *pdfImplUnix )
    {
        //    If we already have loaded the plug-in, no need to do it again.
        if(m_hInst)
            return true;

        initStructs();

        //    Load the plug-in
        int pdfStatus = pdfImplUnix->getWebFrameOwner()->loadPDFPlugin( reinterpret_cast<void **>( &m_hInst ) );
        if ( pdfStatus != 0 )
        {
            pdfImplUnix->getWebFrameOwner()->handlePDFError( pdfStatus );
            unload( pdfImplUnix );
            return false;
        }

        ASSERT(m_hInst);
        if(!m_hInst)
            return false;

        // Note: NP_GetEntryPoints and NP_Initialize
        // NP_GetEntryPoints function is used to get all the function of plugin which can be called by
        // the browser. On Linux, it is not defined in the plugin and instead is done by NP_Initialize only. 

        NP_InitializeFuncPtr NP_Initialize =  (NP_InitializeFuncPtr)dlsym(m_hInst, "NP_Initialize");
        mNP_Shutdown =  (NPP_ShutdownProcPtr)dlsym(m_hInst, "NP_Shutdown");
        NP_ApolloEntryFuncPtr NP_ApolloEntry =  (NP_ApolloEntryFuncPtr)dlsym(m_hInst, "NP_ApolloEntry");

        if(!NP_Initialize || !mNP_Shutdown || !NP_ApolloEntry)
        {
            mNP_Shutdown = NULL; // prevent shutdown from being called
            unload( pdfImplUnix );
            return false;
        }

        //    If "NP_ApolloEntry" is a local, the push of the parameter causes the call to crash
        //    because the call (call ebp-0x0ch) gets the wrong local variable!
        if (NP_ApolloEntry(&m_apolloEntryFuncs) != NPERR_NO_ERROR )
        {
            mNP_Shutdown = NULL; // prevent shutdown from being called
            unload( pdfImplUnix );
            return false;
        }

        if(NP_Initialize(&m_browserFuncs, &m_pluginFuncs) != NPERR_NO_ERROR)
        {
            mNP_Shutdown = NULL; // prevent shutdown from being called
            unload( pdfImplUnix );
            return false;
        }

        return true;
    }

    bool PluginPDFImplUnix::unload( PluginPDFImplUnix *pdfImplUnix )
    {
        if(!m_hInst)
            return true;

        if(m_instanceCount > 0)
        {
            ASSERT(false);
            return true;
        }

        if (mNP_Shutdown)
            mNP_Shutdown();

        if(m_hInst)                
        {
            pdfImplUnix->getWebFrameOwner()->unloadPDFPlugin( reinterpret_cast<void *>( m_hInst ) );
            m_hInst = NULL;
        }

        return true;
    }

    void PluginPDFImplUnix::open( PluginPDFImplUnix *pdfImplUnix )
    {
        m_instanceCount++;
        
        if (!m_hInst) 
        {
            // Should load when the first instance opens the plug-in package
            ASSERT(m_instanceCount == 1);
            load( pdfImplUnix );
        }
    }

    void PluginPDFImplUnix::close( PluginPDFImplUnix *pdfImplUnix )
    {
        ASSERT(m_instanceCount > 0);

        if (--m_instanceCount == 0)
            unload( pdfImplUnix );
    }

    bool PluginPDFImplUnix::start()
    {
        if(m_isStarted)
            return true;

        // XXX - preferences for arePlugInsEnabled?
        sPluginPDFBeingInitialized = this;
        open( this );
        sPluginPDFBeingInitialized = NULL;

        if(!m_hInst)
        {
            close( this );
            return false;
        }

        setAttributeKeys();

        // Plug-ins are "windowed" by default.
        m_window.type = NPWindowTypeWindow;

        // NPN_New(), which creates the plug-in instance, should never be called while calling a plug-in function for that instance.
        ASSERT(m_pluginFunctionCallDepth == 0);
    
        // Apollo puts a blank space before the mime type.  Strip it off. 
        WebCore::String strippedMime = mMimeType.stripWhiteSpace();
    
        NPError npErr = NPP_New((NPMIMEType)strippedMime.latin1().data(), m_instance, m_mode, mParamNames.size(), m_cAttributes, m_cValues, NULL);

        if (npErr != NPERR_NO_ERROR) 
        {
            ASSERT(false);
            close( this );
            return false;
        }

        NPStream    *p_NPStream = new NPStream;
        if (!p_NPStream)
        {
            close( this );
            return false;
        }

        memset(p_NPStream, 0, sizeof(NPStream));

        p_NPStream->ndata = (void *) this;
        m_docStream = new NPStreamWrapper(p_NPStream,  true);
        if (!m_docStream) 
        {
            delete p_NPStream;
            close( this );
            return false;
        }

        // XXX - amohr, this whole section sounds fishy due to the fact that m_sType isn't populated until PluginPDFImplWin::clientReceivedResponse
        if (m_mode == NP_EMBED) 
        {
            // Start loading the URL if we are embedded
            // If we are NP_FULL, the data will be redirected to us
            if (m_sType == NP_SEEK) 
            {
                loadURLForClient(getUrl(), (void *)m_docStream);
                m_isStarted = true;
                return true;
            }
            else 
            {
                // XXX Todo: Support NP_STREAMASFILE ?..
                ASSERT(false);
                close( this );
                return false;
            }
        }
        else 
        {
            if (m_mode == NP_FULL)
            {
                m_isStarted = true;
                return true;
            }
            else  {
                ASSERT(false); // We shouldn't be here. 
                close( this );
                return false;
            }
        }

        return true;
    }

    bool PluginPDFImplUnix::stop(NPError reason, bool inDestructor)
    {
        // If we're already calling a plug-in function, do not call NPP_Destroy().  The plug-in function we are calling
        // may assume that its instance->pdata, or other memory freed by NPP_Destroy(), is valid and unchanged until said
        // plugin-function returns.
        // See <rdar://problem/4480737>.
        if (m_pluginFunctionCallDepth > 0) 
        {
            m_shouldStopSoon = false;
            return false;
        }
        
        // XXX [self removeTrackingRect];

        if (!m_isStarted)
            return false;
        
        m_isStarted = false;
        
        // Stop any active streams 

        // Delete and stop the Aux streams
        // If they are started, we need to call NPP_DestroyStream on them
        cleanUpAuxStreams();

        if (m_mode == NP_FULL)
        {
            // If the plugin is not embedded, we need to tell the frame to stop sending us data
            //    but if we're in the destructor, the frame is already going away and has set this
            //    to null (and will die if we try to make this call).
            if (!inDestructor)
            {
                mWebFrameOwner->clearResourceHandleRedirect();
            }
        }

        NPError npErr;
        // XXX [streams makeObjectsPerformSelector:@selector(stop)];
        // Then, destroy the doc stream:
        if ((m_docStream->getIsDead() == false ) && m_docStream->getNPStream()) {
            willCallPlugInFunction();
            npErr = NPP_DestroyStream(m_instance, m_docStream->getNPStream(), reason);
            didCallPlugInFunction();
            delete m_docStream;
            m_docStream = NULL;
        }
      

        // Stop the null events
        // XXX [self stopNullEvents];

        // Set cursor back to arrow cursor
        // XXX [[NSCursor arrowCursor] set];
        
        // Stop notifications and callbacks.
        // XXX [self removeWindowObservers];
        // XXX [[pendingFrameLoads allKeys] makeObjectsPerformSelector:@selector(_setInternalLoadDelegate:) withObject:nil];
        // XXX [NSObject cancelPreviousPerformRequestsWithTarget:self];

        // Setting the window type to 0 ensures that NPP_SetWindow will be called if the plug-in is restarted.
        // XXX lastSetWindow.type = 0;
        
        npErr = NPP_Destroy(m_instance, NULL);
        // Set the instance to Dead
        m_instanceIsDead = true;
        m_instance->pdata = NULL;

        // Destroy the window
        if (m_gtkWindow && GTK_IS_WIDGET(m_gtkWindow)) {
            gtk_widget_destroy(m_gtkWindow);
            m_gtkWindow = NULL;
        }
        
        // This instance no longer needs the plug-in package
        close( this );
        
        // We usually remove the key event handler in resignFirstResponder but it is possible that resignFirstResponder 
        // may never get called so we can't completely rely on it.
        // XXX [self removeKeyEventHandler];

        return true;
    }
    void PluginPDFImplUnix::cleanUpAuxStreams()
    {
        for(Vector<NPStreamWrapper *>::iterator i = m_auxStreams.begin(); i != m_auxStreams.end(); ++i)
        {
            NPStreamWrapper *pStream = (NPStreamWrapper *)(*i);
            if (pStream) {
                // If it is started, we need to stop it, and then call NPP_DestroyStream
                if (pStream->getIsStarted())
                {
                    WTF::RefPtr<WebCore::ResourceHandle> resourceHandle = handleFromClient(pStream);
                    if (resourceHandle)
                        resourceHandle->cancel();


                    if (pStream->getNPStream()) 
                    {
                        void *notifyData = pStream->getNotifyData();
                        if (notifyData) 
                        {
                            willCallPlugInFunction();
                            NPP_URLNotify(m_instance, pStream->getNPStream()->url, NPRES_USER_BREAK, notifyData);
                            didCallPlugInFunction();

                            pStream->setNotifyData(NULL); // it's been freed by this point
                        }

                        willCallPlugInFunction();
                        NPP_DestroyStream(m_instance, pStream->getNPStream(), NPRES_USER_BREAK);
                        didCallPlugInFunction();
                    }


                }
                delete pStream;
            }

        }
        m_auxStreams.clear();

    }

    WTF::Vector< NPStreamWrapper * >::iterator PluginPDFImplUnix::findAuxStream( NPStreamWrapper* const pStream )
    {
        return std::find( m_auxStreams.begin(), m_auxStreams.end(), pStream );
    }


    void PluginPDFImplUnix::removeAuxStream(NPStreamWrapper * const pStream )
    {
        size_t const streamPosition = findAuxStream( pStream ) - m_auxStreams.begin();
        m_auxStreams.remove( streamPosition );
    }

    NPError PluginPDFImplUnix::postURLHelper(const WebCore::KURL& kURL, WebCore::Frame *pFrame, const WTF::Vector<char>& aBuf)
    {
        const char *bodyPtr = NULL;
        unsigned int bodyLength = 0;
        WebCore::HTTPHeaderMap headerMap;

        //    Special case: DOUBLE POST.    We may already have a resource request saved
        if ( m_useDoublePOSTRequest )
        {
            m_doublePOSTRequest.httpBody()->flatten(m_buf);
            headerMap = m_doublePOSTRequest.httpHeaderFields();
            bodyPtr = m_buf.data();
            bodyLength = m_buf.size();

            //    Unref all the things inside the saved resource request that are
            //    reeference-counted.
            m_doublePOSTRequest = WebCore::ResourceRequest();
        }
        else
        {
            // First we need to parse the headers out of the buffer.
            // The customer headers to add to the post  are distinguished from the body by a blank line. 
            // If there are no customer headers, the buffer will start with a blank line. 

            WebCore::String line("");
            unsigned int index;
            for(index = 0; index < aBuf.size(); ++index)
            {
                char currChar = aBuf[index];
             
                if(currChar == '\n' && !line.length())
                {
                    index++;
                    break;
                }
                else if(currChar == '\n')
                {
                    int pos = line.find(':');
                    ASSERT(pos >= 0);

                    WebCore::String name = line.substring(0, pos).stripWhiteSpace();
                    WebCore::String value = line.substring(pos+1).stripWhiteSpace();

                    headerMap.add(name, value);
                    line = "";
                }
                else if(currChar != '\r')
                {
                    line.append(currChar);
                }
            }

            if(index < aBuf.size())
                bodyPtr = &aBuf[index];
            bodyLength = aBuf.size() - index;
        }


        NPStreamWrapper *pStreamWrapper = createNewStream(kURL, NULL);
        if (!pStreamWrapper)
            return NPERR_GENERIC_ERROR;

        // XXX ToDo .. add NPStreamWrapper to list in order to delete when destroyed
        // XXX Todo. If there is a frame, then we will need to set a timer to do the 
        // loadURLForClient. 

        loadURLForClient(kURL, "POST", bodyPtr, bodyLength, (void *)pStreamWrapper, headerMap, pFrame);
        
        return NPERR_NO_ERROR;
    }

    void PluginPDFImplUnix::delayedPostTimerFired(WebCore::Timer<PluginPDFImplUnix>*)
    {
        m_delayedPostTimer.stop();
        WebCore::Frame *pFrame =  mWebFrameOwner->frame();
        postURLHelper(m_postURL, pFrame, m_buf);
    }

    NPError PluginPDFImplUnix::postURL(NPP instance, const char* URL, const char* target, uint32 len, const char* buf, NPBool file)
    {
        // We don't support certain URL/Source/Target combinations for security reasons.
        NPError err = allowURL( URL, target );
        if ( err != NPERR_NO_ERROR )
            return err;

        // Posting from a file is only supported on Unix
        if (file)
        {
                // buf contains the filename instead of data
                FILE *dataFD = fopen(buf + 8, "r");   // Move 8 chars ahead to pass "file:///"

                int lFileLen = 0;
                fseek(dataFD, 0L, SEEK_END);  /* Position to end of file */
                lFileLen = ftell(dataFD);     /* Get file length */
                rewind(dataFD);               /* Back to start of file */

                char *dataBuffer = (char*) malloc (lFileLen + 1);
                
                fread(dataBuffer, lFileLen, 1, dataFD);
                dataBuffer[lFileLen] = '\0';
                fclose(dataFD);

                buf = dataBuffer;
                len = lFileLen;
        }

        // Acrobat/Reader only uses a window of null (which means send the results back to the plug-in
        // or a window of "_current", which means send to the current frame. 
        WebCore::String  frameTarget("_current");
        bool bSendToFrame = (target ? frameTarget == target : false);

        int retval = 0;

        WebCore::Frame *pFrame = NULL;
        // If we are going to send this to the frame, then start resource loader from the frame
        // We need to, however delay this because in the process of doing the post which returns
        // a non adobe format like HTML, Apollo will tear down our widget. We are in a callback 
        // at this exact point from the Firefox plug-in and if we were to post now, we would
        // blow up as the stack unwinds. 
        if (bSendToFrame) {
            if (!m_delayedPostTimer.isActive()) {

                //    Special case: DOUBLE POST.  If we are inside NPP_NewStream(), the only way
                //    we got here was a double post.  Due to a bug in NS_PDFX.cpp, we have to
                //    get the body and everything from the existing request rather than from
                //    the plug-in.
                if (m_inClientReceivedResponse && m_clientReceivedResponseResourceHandle != NULL) {
                    m_useDoublePOSTRequest = true;
                    m_doublePOSTRequest = m_clientReceivedResponseResourceHandle->request();
                    m_postURL = m_doublePOSTRequest.url();
                }
                else {
                    m_postURL = WebCore::KURL(URL);
                    m_buf.clear();
                    m_buf.append(buf, len);
                }
                m_delayedPostTimer.startOneShot(0);
                retval = NPERR_NO_ERROR;
            }
            else 
                retval = NPERR_GENERIC_ERROR;
        }
        else  {
            WTF::Vector<char> vBuf;
            vBuf.append(buf, len);
            retval = postURLHelper(WebCore::KURL(URL), pFrame, vBuf);
        }

        if(file)
        {
            if(buf)
                free((void*)buf);
        }

        return retval;
    }

    PluginPDFImplUnix* PluginPDFImplUnix::GetInstancePointer(NPP instance)
    {
        //    We may be called during overall initialization where the
        //    plugin doesn't know the instance; we remember which one
        //    that is.
        if ( instance == NULL )
        {
            ASSERT(sPluginPDFBeingInitialized);
            return sPluginPDFBeingInitialized;
        }
        else
        {
            ASSERT(instance && instance->ndata); // or else we need setCurrentPluginView call
            if(instance && instance->ndata)
                return (PluginPDFImplUnix*)instance->ndata;
        }

        ASSERT(false);
        return NULL;
    }

    // ----- begin netscape functions ---------

    void* PluginPDFImplUnix::NPN_MemAlloc(uint32 size)
    {
        return malloc(size);
    }

    void PluginPDFImplUnix::NPN_MemFree(void* ptr)
    {
        free(ptr);
    }

    uint32 PluginPDFImplUnix::NPN_MemFlush(uint32 size)
    {
        ASSERT(false);
        return size;
    }

    void PluginPDFImplUnix::NPN_ReloadPlugins(NPBool reloadPages)
    {
        ASSERT(false);
    }

    NPError PluginPDFImplUnix::NPN_RequestRead(NPStream* stream, NPByteRange* rangeList)
    {
        // XXX - TBI ... this is causing some examples to fail
        ASSERT(false);
        return NPERR_GENERIC_ERROR;
    }

    NPError PluginPDFImplUnix::NPN_GetURLNotify(NPP instance, const char* URL, const char* target, void* notifyData)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        return ptr->getURLNotify(URL, target, notifyData);
    }

    NPError PluginPDFImplUnix::NPN_GetURL(NPP instance, const char* URL, const char* target)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        return ptr->getURLNotify(URL, target, NULL);
    }

    NPError PluginPDFImplUnix::NPN_PostURLNotify(NPP instance, const char* URL, const char* target, uint32 len, const char* buf, NPBool file, void* notifyData)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        // XXX - TBI
        ASSERT(false);
        return NPERR_GENERIC_ERROR;
    }

    NPError PluginPDFImplUnix::NPN_PostURL(NPP instance, const char* URL, const char* target, uint32 len, const char* buf, NPBool file)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        return ptr->postURL(instance, URL, target, len, buf, file);
    }

    NPError PluginPDFImplUnix::NPN_NewStream(NPP instance, NPMIMEType type, const char* target, NPStream** stream)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        return NPERR_GENERIC_ERROR;
    }

    int32 PluginPDFImplUnix::NPN_Write(NPP instance, NPStream* stream, int32 len, void* buffer)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        return NPERR_GENERIC_ERROR;
    }

    NPError PluginPDFImplUnix::NPN_DestroyStream(NPP instance, NPStream* stream, NPReason reason)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        // XXX - TBI
        ASSERT(false);
        return NPERR_GENERIC_ERROR;
    }

    const char* PluginPDFImplUnix::NPN_UserAgent(NPP instance)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NULL;

        return ptr->m_userAgent.data();
    }

    void PluginPDFImplUnix::NPN_Status(NPP instance, const char* message)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return;

        WebCore::String msg(message);
        WebCore::Frame* frame = ptr->mWebFrameOwner->frame();
        WebCore::Page* page = frame->page();
        WebCore::Chrome* chrome = page ? page->chrome() : 0;
        
        if(chrome)
            chrome->setStatusbarText(frame, msg);
    }

    void PluginPDFImplUnix::NPN_InvalidateRect(NPP instance, NPRect *invalidRect)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return;

        NPRect rect;
        rect.left = invalidRect->left;
        rect.top = invalidRect->top;
        rect.right = invalidRect->right;
        rect.bottom = invalidRect->bottom;

        ptr->invalidateRect(rect);
    }

    void PluginPDFImplUnix::NPN_InvalidateRegion(NPP instance, NPRegion invalidRegion)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return;

        ptr->invalidateRegion(invalidRegion);
    }

    void PluginPDFImplUnix::NPN_ForceRedraw(NPP instance)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return;

        ptr->invalidate();
        ptr->draw();
    }

    NPError PluginPDFImplUnix::NPN_GetValue(NPP instance, NPNVariable variable, void *value)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        return ptr->getVariable(variable, value);    
    }

    NPError PluginPDFImplUnix::NPN_SetValue(NPP instance, NPPVariable variable, void *value)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(!ptr)
            return NPERR_GENERIC_ERROR;

        return ptr->setVariable(variable, value);
    }

    void* PluginPDFImplUnix::NPN_GetJavaEnv(void)
    {
        ASSERT(false);
        return NULL;
    }

    void* PluginPDFImplUnix::NPN_GetJavaPeer(NPP instance)
    {
        ASSERT(false);
        return NULL;
    }

    // Apollo callbacks
    void PluginPDFImplUnix::NPA_BeginModal(NPP instance)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(ptr)
            ptr->beginModal();
    }

    void PluginPDFImplUnix::NPA_EndModal(NPP instance)
    {
        PluginPDFImplUnix* ptr = GetInstancePointer(instance);
        if(ptr)
            ptr->endModal();
    }

    // helper methods for callback handlers

    WebCore::KURL PluginPDFImplUnix::getBaseURL()
    {
        WebCore::Document* doc = mWebFrameOwner->frame()->document();
        return doc->completeURL(doc->baseURL());
    }

    void PluginPDFImplUnix::setAttributeKeys()
    {
        ASSERT( mParamNames.size() == mParamValues.size() );

        // Convert the attributes to 2 C string arrays.
        // These arrays are passed to NPP_New, but the strings need to be
        // modifiable and live the entire life of the plugin.

        m_cAttributes = (char **)malloc(mParamNames.size() * sizeof(char *));
        m_cValues = (char **)malloc(mParamValues.size() * sizeof(char *));

        unsigned int count = mParamNames.size();

        for (unsigned int i = 0; i < count; i++) 
        {
            if (mParamNames[i].foldCase() == "height") 
            {
                m_specifiedHeight = mParamValues[i].toInt();
            } 
            else if (mParamNames[i].foldCase()  == "width") 
            {
                m_specifiedWidth = mParamValues[i].toInt();
            }

            m_cAttributes[i] = strdup(mParamNames[i].utf8().data());
            m_cValues[i] = strdup(mParamValues[i].utf8().data());
        }
    }

    void PluginPDFImplUnix::willCallPlugInFunction()
    {
        // Could try to prevent infinite recursion here, but it's probably not worth the effort.
        m_pluginFunctionCallDepth++;
    }

    void PluginPDFImplUnix::didCallPlugInFunction()
    {
        ASSERT(m_pluginFunctionCallDepth > 0);
        m_pluginFunctionCallDepth--;
        
        // If -stop was called while we were calling into a plug-in function, and we're no longer
        // inside a plug-in function, stop now.
        if (m_pluginFunctionCallDepth == 0 && m_shouldStopSoon) 
        {
            m_shouldStopSoon = false;
            stop();
        }
    }

    NPStreamWrapper* PluginPDFImplUnix::createNewStream(const WebCore::KURL& kURL, void* notifyData)
    {
        NPStream    *p_NPStream = new NPStream;
        if (!p_NPStream)
            return NULL;

        memset(p_NPStream, 0, sizeof(NPStream));
        p_NPStream->ndata = (void*)this;

        NPStreamWrapper *pStreamWrapper  = new NPStreamWrapper(p_NPStream, kURL, false);
        if (!pStreamWrapper)
        {
            delete p_NPStream;
            return NULL;
        }

        // Add this to our list of aux streams
        m_auxStreams.append(pStreamWrapper);
        pStreamWrapper->setNotifyData(notifyData);

        return pStreamWrapper;
    }

    // returns true if docStream
    bool PluginPDFImplUnix::releaseStream(NPStreamWrapper* pStream)
    {
        // If it is not the doc stream, we are going to call NPP_DestroyStream and the delete it.
        if (!pStream->getIsDocStream())
        {
            if (pStream->getNPStream()) 
            {
                void *notifyData = pStream->getNotifyData();
                if (notifyData) 
                {
                    willCallPlugInFunction();
                    NPP_URLNotify(m_instance, pStream->getNPStream()->url, NPRES_DONE, notifyData);
                    didCallPlugInFunction();

                    pStream->setNotifyData(NULL); // it's been freed by this point
                }

                willCallPlugInFunction();
                NPP_DestroyStream(m_instance, pStream->getNPStream(), NPRES_DONE);
                didCallPlugInFunction();
            }

            // remove it from the list of aux_streams
            removeAuxStream(pStream);

            delete pStream;

            return false;
        }

        return true;
    }

    NPError PluginPDFImplUnix::getURLNotifyHelper(WebCore::KURL kURL, WebCore::Frame * pFrame, void* notifyData)
    {
        NPStreamWrapper* pStreamWrapper = createNewStream(kURL, notifyData);
        if(!pStreamWrapper)
            return NPERR_GENERIC_ERROR;

        // XXX ToDo .. add NPStreamWrapper to list in order to delete when destroyed
        // XXX Todo. If there is a frame, then we will need to set a timer to do the 
        // loadURLForClient. 

        WebCore::HTTPHeaderMap headerMap;
        loadURLForClient(kURL, "GET",NULL, 0, (void *)pStreamWrapper, headerMap, pFrame);
        return NPERR_NO_ERROR;

    }

    void PluginPDFImplUnix::delayedGetTimerFired(WebCore::Timer<PluginPDFImplUnix>*)
    {
        m_delayedGetTimer.stop();
        WebCore::Frame *pFrame =  mWebFrameOwner->frame();
        getURLNotifyHelper(m_getURL, pFrame, m_notifyData);
    }

    NPError PluginPDFImplUnix::allowURL(const char *URL, const char *target)
    {
        // Security: we only allow a certain set of URLs, depends on this URL and the target
        WebCore::Frame *targetFrame = getWebFrameOwner()->frame();
        if ( target != NULL && strlen(target) > 0 )
            targetFrame->tree()->find( target );
        if ( !allowURLRequest( WebCore::KURL(URL), targetFrame ) )
            return NPERR_GENERIC_ERROR;

        return NPERR_NO_ERROR;
    }


    NPError PluginPDFImplUnix::getURLNotify(const char* URL, const char* target, void* notifyData)
    {
        NPError err = allowURL( URL, target );
        if ( err != NPERR_NO_ERROR )
            return err;

        // Acrobat/Reader only uses a window of null (which means send the results back to the plug-in
        // or a window of "_current", which means send to the current frame. 
        WebCore::String  frameTarget("_current");
        bool bSendToFrame = (target ? frameTarget == target : false);

        WebCore::Frame *pFrame = NULL;
        // If we are going to send this to the frame, then start resource loader from the frame
        // We need to, however delay this because in the process of doing the GET,
        // Apollo will tear down our widget. We are in a callback 
        // at this exact point from the Firefox plug-in and if we were to GET right now, we would
        // blow up as the stack unwinds. 
        if (bSendToFrame) {
            if (!m_delayedGetTimer.isActive()) {
                m_getURL = WebCore::KURL(URL);
                m_notifyData = notifyData;
                m_delayedGetTimer.startOneShot(0);
                return NPERR_NO_ERROR;
            }
            else 
                return NPERR_GENERIC_ERROR;
        }
        else  {
            return getURLNotifyHelper(WebCore::KURL(URL), pFrame, notifyData);
        }
    }


    void PluginPDFImplUnix::invalidateRect(const NPRect& invalidRect)
    {
        ASSERT(false); // XXX is this the right thing to do?
        //InvalidateRect(m_gtkWindow, &invalidRect, false);
    }

    void PluginPDFImplUnix::invalidateRegion(NPRegion invalidRegion)
    {
        ASSERT(false); // XXX is this the right thing to do?
        //InvalidateRgn(m_gtkWindow, invalidRegion, false);
    }

    void PluginPDFImplUnix::draw()
    {
        // XXX TBI
        ASSERT(false);
    }

    void PluginPDFImplUnix::invalidate()
    {
        // XXX TBI
        ASSERT(false);
    }


    /**** Bi-directional javascript interface ***************************/

    NPObject* PluginPDFImplUnix::windowScriptNPObject()
    {
        if (!m_windowScriptNPObject) 
        {
            WebCore::Frame *frame = mWebFrameOwner->frame();

            if (frame->settings()->isJavaScriptEnabled()) 
            {
                // JavaScript is enabled, so there is a JavaScript window object.  Return an NPObject bound to the window
                // object.
                KJS::JSObject *win = WebCore::toJSDOMWindow(frame);
                ASSERT(win);
                m_windowScriptNPObject = _NPN_CreateScriptObject(0, win, frame->bindingRootObject());
            } 
            else 
            {
                // JavaScript is not enabled, so we cannot bind the NPObject to the JavaScript window object.
                // Instead, we create an NPObject of a different class, one which is not bound to a JavaScript object.
                m_windowScriptNPObject = _NPN_CreateNoScriptObject();
            }
        }

        return m_windowScriptNPObject;
    }
    
    NPObject* PluginPDFImplUnix::getElementNPObject()
    {
        if (mElement->hasTagName(*mEmbedTag) || mElement->hasTagName(*mObjectTag))
            return static_cast<WebCore::HTMLPlugInElement*>(mElement)->getNPObject();
        else
            return NULL;
    }

    // methods called by netscape callbacks

    NPError PluginPDFImplUnix::setVariable(NPPVariable variable, void* value)
    {
        switch(variable)
        {
            case NPPVpluginWindowBool:
            {
                NPWindowType newWindowType = (value ? NPWindowTypeWindow : NPWindowTypeDrawable);
                if(newWindowType != m_window.type)
                    invalidate();

                m_window.type = newWindowType;
            }
            case NPPVpluginTransparentBool:
            {
                bool newTransparent = (value != 0);

                // Redisplay if transparency is changing
                if(m_isTransparent != newTransparent)
                    invalidate();

                m_isTransparent = newTransparent;
                break;
            }
            default:
            {
                ASSERT(false);
                return NPERR_GENERIC_ERROR;
            }
        }

        return NPERR_NO_ERROR;
    }

    NPError PluginPDFImplUnix::getVariable(NPNVariable variable, void *value)
    {
        switch (variable) 
        {
            case NPNVWindowNPObject:
            {
                NPObject *windowScriptObject = windowScriptNPObject();

                // Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugins/npruntime.html#browseraccess>
                if (windowScriptObject)
                    _NPN_RetainObject(windowScriptObject);
                
                void **v = (void **)value;
                *v = windowScriptObject;

                break;
            }

            case NPNVPluginElementNPObject:
            {
                if (!mElement)
                    return NPERR_GENERIC_ERROR;
                
                NPObject *plugInScriptObject = getElementNPObject();

                // Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugins/npruntime.html#browseraccess>
                if (plugInScriptObject)
                    _NPN_RetainObject(plugInScriptObject);

                void **v = (void **)value;
                *v = plugInScriptObject;

                return plugInScriptObject ? NPERR_NO_ERROR : NPERR_GENERIC_ERROR;
            }               

            // For XEmbed Support
            case NPNVxDisplay:
            {
                NPBool rtv = false;
                NPP_GetValue(m_instance, NPPVpluginNeedsXEmbed, &rtv);
                if (rtv) {
                    (*(Display **)value) = GDK_DISPLAY();
                    m_pluginNeedsXEmbed = true;
                    return NPERR_NO_ERROR;
                }

#if ADOBE_NPPDF_HACK
                // Adobe nppdf hack (picked up from Mozilla Firefox source code)
                // Adobe nppdf uses Xt (X Toolkit) to get a display from the browser and then
                // use it to embed adobe reader inside.
                // adobe nppdf calls XtGetApplicationNameAndClass(display, &instance, &class) and so
                // we have to init Xt toolkit. Before getting XtDisplay, just call gtk_xtbin_new(w,0) once

                // TODO - gtkxtbin is not a part of standard gtk. Mozilla has implemented this on its
                // own and so webkit would also need the same.

                static GtkWidget *xtbin = 0;
                if (!xtbin) {
                    xtbin = gtk_xtbin_new(GDK_ROOT_PARENT(),0);
                    // it crashes on destroy, let it leak    // also a part of mozilla code
                    // gtk_widget_destroy(xtbin);
                }
                (*(Display **)value) =  GTK_XTBIN(xtbin)->xtdisplay;
                return NPERR_NO_ERROR;
#endif
                return NPERR_GENERIC_ERROR;
            }

            case NPNVSupportsXEmbedBool: 
            {
                *(NPBool*)value = true;
                return NPERR_NO_ERROR;
            }

            case NPNVxtAppContext:
            {
                return NPERR_GENERIC_ERROR;
            }

            default:
            {
                ASSERT(false);
                return NPERR_GENERIC_ERROR;
            }
        }

        return NPERR_NO_ERROR;        
    }


    /******************RESOURCE LOADER ******************/

    NPStreamWrapper *PluginPDFImplUnix::streamWrapperFromClient( void *client )
    {
        //    If we're dead, we don't try anything at all.
        if (m_instanceIsDead || !m_hInst)
            return NULL;

        //    It should be the client
        NPStreamWrapper * pStream = reinterpret_cast<NPStreamWrapper *>( client );

        //    If we don't have a client yet, then it's the docStream
        if (pStream == NULL)
            pStream = m_docStream;

        //    If the docstream is null then we couldn't load the plugin (but we
        //    don't cancel the stream in that case) and we ignore everything.
        if ( pStream == NULL )
            return NULL;

        // If the stream is dead, bail.
        if ( pStream->getIsDead() )
            return NULL;

        //    Oh well I guess we actually have to use the stream...
        return pStream;
    }

    void PluginPDFImplUnix::clientReceivedData( void *client, WebCore::ResourceHandle*, const char* pDataBytes, int dataLength )
    {
        NPStreamWrapper * pStream = streamWrapperFromClient( client );
        if (pStream == NULL)
            return;

        if(m_sType == NP_ASFILE || m_sType == NP_ASFILEONLY)
        {
            ASSERT(m_tempFile);

            if(m_tempFile)
                m_tempFile.write(pDataBytes, dataLength); 

            return;
        }

        // The API says you are supposed to call NPP_WriteReady first. 
        // This is not needed for Reader, so we ignore it. 
        // XXX Todo for plug-in support beyond Reader.
        if (pStream->getNPStream()) 
        {
            //    Acrobat cannot handle more than 2MB at a time in its byte-cache.
            //    We have to split it up; split into 1MB chunks.
            int32 curOffset = pStream->getStreamOffset();
            int32 curLength = dataLength;
            const char *curData = pDataBytes;
            while ( curLength > 0 )
            {
                int32 thisLength = (curLength > 1024 * 1024 ) ? 1024 * 1024 : curLength;
                int32 bytesWritten = NPP_Write(m_instance, pStream->getNPStream(), curOffset, thisLength, (void *) curData );

                ASSERT(bytesWritten == thisLength);

                curLength -= thisLength;
                curData += thisLength;
                curOffset += thisLength;
            }

            pStream->setStreamOffset(pStream->getStreamOffset()+ dataLength);
        }
        // Theoretically, we are supposed to check the return value from NPP_Write. 
        // If it is negative, we are supposed to destroy the stream. 
        // Reader never returns a negative value here and we therefore ignore it.
        // We are also supposed to check to see if the number of bytes consumed was less than we sent,
        // and then send the remaining bytes on a subsequent NPP_WriteReady and NPP_Write sequence.  
        // Reader, however, always consumes the bytes we send and we therefore ignore this.
    }

    void PluginPDFImplUnix::clientReceivedAllData(void *client, WebCore::ResourceHandle* resourceHandle )
    {
        NPStreamWrapper * pStream = streamWrapperFromClient( client );
        if (pStream == NULL)
            return;

        if(m_sType == NP_ASFILE || m_sType == NP_ASFILEONLY)
        {
            m_tempFile.close();
            NPP_StreamAsFile(m_instance, pStream->getNPStream(), m_tempFile.getPath());
            //m_tempFile.remove();             // TODO : Deleting this file fails NPP_StreamAsFile 
        }

        // If it is not the doc stream, we are going to call NPP_DestroyStream and the delete it.
        if (releaseStream(pStream)) 
        {
            if (m_mode == NP_FULL)
            {
                // If the plugin is not embedded, we need to tell the frame to stop sending us data
                mWebFrameOwner->clearResourceHandleRedirect();
            }
            // Now we need to check to see if all of the data we were expecting was sent. 
            // If not, then we need to close the stream with a reason of NPRES_NETWORK_ERR.
            // If we don't do this, Reader will hang.
            // XXX todo-> when we support byte ranges requests, handling network errors is going to 
            // very tricky. 
            if ((pStream->getNPStream() && pStream->getNPStream()->end && 
                (pStream->getStreamOffset()) < static_cast<int32>(pStream->getNPStream()->end))){
                stop(NPRES_NETWORK_ERR);
            }
            else if ( pStream->getNPStream()->end <= 0 )
            {
                // EWH won't open anything until it gets a stream length,
                // and nppdf32.dll won't do the ASFILE hack if it's in
                // apollo, so we need to send the stream length.  Copy
                // the existing hack of sending a space character just
                // so we can tell it the end.
                char dummy = ' ';
                int32 numBytes = 1;
                pStream->getNPStream()->end = pStream->getStreamOffset() + numBytes;

                willCallPlugInFunction();
                int32 bytesWritten = NPP_Write(m_instance, pStream->getNPStream(), pStream->getStreamOffset(), numBytes, &dummy );
                ASSERT(bytesWritten == numBytes);
                didCallPlugInFunction();
            }
        }
    }

    void PluginPDFImplUnix::clientReceivedRedirect    (void *client, WebCore::ResourceHandle* resourceHandle, const WebCore::KURL& url )
    {
        // TODO_AIRLinux
        ASSERT(false);
    }

    void PluginPDFImplUnix::clientReceivedResponse    (void *client, WebCore::ResourceHandle* pResourceHandle, const WebCore::ResourceResponse& resourceResponse)
    {
        NPStreamWrapper * pStream = streamWrapperFromClient( client );
        if (pStream == NULL)
            return;

        NPStream *pNPStream = pStream->getNPStream();

        unsigned long end = resourceResponseContentLength(resourceResponse);
        pNPStream->end = end == 0xFFFFFFFF ? 0 : end;


        // If we posted, we will save the URL at that time since the resource loader strips off the URL fragments
        WebCore::KURL  const loadedURL(pStream->getIsDocStream() ? getUrl() : pStream->getURL());
        pStream->setURL(loadedURL);

        WebCore::String mimeType = resourceResponse.httpHeaderField("Content-Type");
        
        // file:// URLs don't have a mime type in the metadata, but we were constructed with one.
        if ( mimeType.length() == 0 )
            mimeType = mMimeType;

        // Apollo puts a blank space before the mime type.  Strip it off. 
        WebCore::String strippedMime = mimeType.stripWhiteSpace();

        // Nppdf32 plugin can't handle charsets in the mime type.  Strip it off.
        strippedMime.truncate(strippedMime.find(';'));

        // This stream is being managed by a resource loader we control
        // (versus one that the frame controls).
        // We need to keep track of this in order to stop it if our object goes away.
        pStream->setIsStarted(true);
        
        //    Special case: DOUBLE POST. If the URL ended in #FDF, but the return value was not the mime
        //    type for Acrobat, then it will call postURL from inside NPP_NewStream, with exactly the same
        //    data but with "_current" as the frame name.  Unfortunately, THERE IS A BUG IN NS_PDFX.cpp
        //    where it gives a bogus buffer.  We heuristically handle that case in postURL by recognizing
        //    that case and instead of using the buffer we got passed, using the saved buffer from
        //    the request that we're currently processing here: pResourceHandle->d->m_request.
        m_inClientReceivedResponse = true;
        m_clientReceivedResponseResourceHandle = pResourceHandle;

        // XXX ToDo: When Apollo support byte ranges, pass TRUE as the seekable param, but only 
        // after checking response headers though to see server supports byte range requests. 
        bool bIsSeekable = false; 
        willCallPlugInFunction();
        NPError npErr =  NPP_NewStream(m_instance, (NPMIMEType)strippedMime.latin1().data(), pNPStream, bIsSeekable, &m_sType);
        didCallPlugInFunction();

        //    Special case: DOUBLE POST.
        m_inClientReceivedResponse = false;
        m_clientReceivedResponseResourceHandle = NULL;

        if (npErr != NPERR_NO_ERROR) 
        {
            ASSERT(false);
            return;
        }

        // If the http status code is not 200 .. then proceed to close the stream with a network error
        // TBD: what about ftp, feed and other inputs?  What kind of errors can they have?
        if (!loadedURL.isLocalFile() && resourceResponse.httpStatusCode() != 200)
        {
            if (pStream->getIsDocStream() == false) 
            {
                pStream->setIsDead(true); // The stream is dead. Ignore any data sent to it

                void *notifyData = pStream->getNotifyData();
                if (notifyData) 
                {
                    willCallPlugInFunction();
                    NPP_URLNotify(m_instance, pStream->getNPStream()->url, NPRES_NETWORK_ERR, notifyData);
                    didCallPlugInFunction();

                    pStream->setNotifyData(NULL); // it's been freed by this point
                }

                willCallPlugInFunction();
                NPP_DestroyStream(m_instance, pStream->getNPStream(), NPRES_NETWORK_ERR);
                didCallPlugInFunction();
                pStream->setIsStarted(false);

                // remove it from the list of aux_streams
                // removeAuxStream(pStream);
                // delete pStream;

                return; // don't do anything else, bad state
            }
            else 
            {
                stop(NPRES_NETWORK_ERR);

                return; // don't do anything else, bad state
            }

            // Tell the resourceloader to stop
            // XXX Fix Me:  This causes a crash in 3/2 Apollo build 
            // pResourceLoader->cancel();
        }
        
        if(m_sType == NP_ASFILE || m_sType == NP_ASFILEONLY)
        {
            m_tempFile.initAsTemp();
            ASSERT(m_tempFile);
        }
    }

    void* PluginPDFImplUnix::createPluginScriptableObject()
    {
        //    If the plug-in doesn't implement NPP_GetValue, or we got some
        //    sort of error and killed the instance, we can't return
        //    anything.
        if (!NPP_GetValue || m_instanceIsDead)
            return NULL;
            
        NPObject *value = NULL;
        willCallPlugInFunction();
        NPError error = NPP_GetValue(m_instance, NPPVpluginScriptableNPObject, &value);
        didCallPlugInFunction();
        if (error != NPERR_NO_ERROR)
            return NULL;

        return value;
    }

    void PluginPDFImplUnix::releasePluginScriptableObject(void* object)
    {
        _NPN_ReleaseObject((NPObject*)object);
    }
}
