#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>

#import <WebKitApollo/ObjCRenamePrefix.h>
#import <WebKit/WebScriptObject.h>
#import <WebKit/WebPluginViewFactory.h>

#include "WebFrameScrollView.h"
#include "WebPluginPDFImplMac.h"
#include "WebPDFPluginProtocol.h"
#include "ResourceLoader.h"
#include "Document.h"
#include "FrameTree.h"
#include "FrameView.h"
#include "WebResourceRequestImpl.h"

#include "CString.h"

#define LOG_PDFIMPL_CALLS		(1 && defined(DEBUG) && DEBUG)

#define DEBUG_RETAIN_RELEASE	(0 && defined(DEBUG) && DEBUG)

#define	DUMP_VIEWS				(0 && defined(DEBUG) && DEBUG)

#if defined(DEBUG) && DEBUG
	static char sDBGRectStr[200];
	char *SprintfNSRect( NSRect *rect )
	{
		float wleft = rect->origin.x;
		float wtop = rect->origin.y;
		float wwidth = rect->size.width;
		float wheight = rect->size.height;
		sprintf( sDBGRectStr, "[ %.1f, %.1f / %.1f, %.1f ]",
							wleft, wtop, wleft + wwidth, wtop + wheight );
		return sDBGRectStr;
	}
#endif

#if DUMP_VIEWS
	inline void DumpViewTree( NSView *view, int level )
	{

        static char sNesting[] = "                              "
							 "                              "
							 "                              "
							 "                              "
							 "                              "
							 "                              ";
        
		const char *viewClass = object_getClassName( [ view class ] );
		NSRect viewFrame = [ view convertRect: [ view bounds ] toView: nil ];
		float left = viewFrame.origin.x;
		float top = viewFrame.origin.y;
		float width = viewFrame.size.width;
		float height = viewFrame.size.height;
		
		if ( level == 0 )
		{
			float wleft = viewFrame.origin.x;
			float wtop = viewFrame.origin.y;
			float wwidth = viewFrame.size.width;
			float wheight = viewFrame.size.height;
			NSLog( @"Dumping tree view for window 0x%08X [ %.1f, %.1f / %.1f, %.1f ]",
								[ view window ], wtop, wleft, wtop + wheight, wleft + wwidth);
		}
		
		NSString *str = [ NSString stringWithCString: sNesting length: (level * 2) ];
		NSLog( @"%@ 0x%08X %s [ %.1f, %.1f / %.1f, %.1f ", 
                str, view, viewClass, top, left, top + height, left + width, width, height );

		//	A different view: check subviews
		NSArray *subviews = [ view subviews ];
		NSEnumerator *viewNumerator = [ subviews objectEnumerator ];
		NSView *subview;
		while ( (subview = [ viewNumerator nextObject ]) != nil )
			DumpViewTree( subview, level + 1 );
	}
#endif

    void DumpViewTree( NSWindow* window )
    {
#if DUMP_VIEWS
        NSView *contentView = [ window contentView ];
        DumpViewTree( contentView, 0 );
        NSLog( @"\n" );
#endif
    }


#if defined(DEBUG) && DEBUG
	//	Class to keep track of the plugin superview calls for debugging purposes
	@interface PluginSuperview : NSView {
	}
	
	- (void) setFrame: (NSRect) frameRect;
	
	@end
	
	@implementation PluginSuperview
	
	- (void) setFrame: (NSRect) frameRect
	{
		NSLog( @"PluginSuperview(0x%08X) setFrame(0x%08X) %s", self, super, SprintfNSRect(&frameRect) );
		[ super setFrame: frameRect ];
	}
	
	@end
#else
	#define PluginSuperview 	NSView
#endif

//@protocol WebPlugInViewFactory <NSObject>
//	+ (NSView *)plugInViewWithArguments:(NSDictionary *)arguments;
//@end

@protocol WebPlugin <NSObject>

	- (void) webPlugInInitialize;
	- (void) webPlugInStart;
	- (void) webPlugInStop;
	- (void) webPlugInDestroy;
	- (void) webPlugInSetIsSelected: (BOOL) isSelected;
	- (id) objectForWebScript;

@end

//	Forward declarations.
@class OPluginPDFImplMacBridge;

//	This is the webFrame bridge for the plugin: it implements the necessary WebFrame
//	methods in order to support POSTs to this frame and others.
@interface OPluginPDFWebFrameBridge : NSObject<ApolloWebKitWebFramePluginHost> {
	OPluginPDFImplMacBridge *	mMacBridge;
	WebCore::Frame *		mTargetFrame;
}

//	Initialization/destruction
- (OPluginPDFWebFrameBridge *) initWithMacBridge: (OPluginPDFImplMacBridge *) macBridge andTargetFrame: (WebCore::Frame *) targetFrame;
- (void) dealloc;

//	WebFrame Routines needed by AdobePDFView
- (void) errorLoadingAcrobat: (SInt32) error;
- (void) loadRequest: (NSURLRequest *) request;
- (NSObject<ApolloWebKitWebFramePluginHost> *) findFrameNamed: (NSString *) name;
- (id) dataSource;
- (id) webView;

//	This is actuallya WebPlugInContainer routine, we put it here for convenience
//	and AdobePDFViewer gets the WebFrame from it: self.
- (NSObject<ApolloWebKitWebFramePluginHost> *) webFrame;

@end

//This is the bridge object betweeen the Apollo Web Kit PDF mac subclass
//and AdobePDFViewer, the Acrobat WebKit plugin.  This object acts as
//the WebKit Plugin Host.
@interface OPluginPDFImplMacBridge : NSObject {
	NSView<WebPlugInViewFactory,WebPlugin>*		mPluginView;
	WebKitApollo::PluginPDFImplMac*				mPluginPDFImpl;
	NSMutableDictionary*						mPluginArguments;
	NSView*										mPluginSuperview;
	#if IMP_EXTERNAL_RESOURCE_LOADER
		NSMutableArray *							mCurrentPluginClients;
	#endif
	BOOL										mStarted;
	BOOL										mVisible;
    BOOL                                        mEnabled;
    BOOL                                        mPluginVisible;
}

+ (NSBundle *) adobePDFViewerBundleForPDFImpl: (WebKitApollo::PluginPDFImplMac *) pluginPDFImpl;
- (OPluginPDFImplMacBridge *) initWithPluginPDFImpl: (WebKitApollo::PluginPDFImplMac *) pluginPDFImpl;
- (NSView<WebPlugInViewFactory,WebPlugin>*) pluginView;
- (NSURL *) URLByExpandingCustomSchemesFor: (NSURL *) origURL;
- (NSURL *) URLByContractingCustomSchemesFor: (NSURL *) origURL;
- (void) start;
- (void) stop;
- (void) setFrameToRect: (const WebCore::IntRect&) rect clippingBy: (const WebCore::IntRect&) clipRect;
- (void) setNeedsDisplay: (const WebCore::IntRect&) windowDirtyRect;
- (void) setVisibility: (BOOL) visibility;
- (void) setEnabled: (BOOL) enabled;
- (NSRect) pluginRectFromApolloWindowRect: (const WebCore::IntRect&) windowRect;
- (WebKitApollo::PluginPDFImplMac*) pluginPDFImpl;
- (void) dealloc;
@end

//	Convenience function for string conversions.
static NSString *NSStringFromWebCoreString( const WebCore::String& wcString )
{
	return [ NSString stringWithCString: wcString.utf8().data() encoding: NSUTF8StringEncoding ];
}

//Copied from WebKit KURLMac.mm and KURLCFNet.cpp
NSURL* NSURLFromKURL( const WebCore::KURL& kurl )
{
    WebCore::CString str = kurl.string().latin1();
    const UInt8 *bytes = (const UInt8 *)str.data();
    // NOTE: We use UTF-8 here since this encoding is used when computing strings when returning URL components
    // (e.g calls to NSURL -path). However, this function is not tolerant of illegal UTF-8 sequences, which
    // could either be a malformed string or bytes in a different encoding, like Shift-JIS, so we fall back
    // onto using ISO Latin-1 in those cases.
    CFURLRef result = CFURLCreateAbsoluteURLWithBytes(0, bytes, str.length(), kCFStringEncodingUTF8, 0, true);
    if (!result)
        result = CFURLCreateAbsoluteURLWithBytes(0, bytes, str.length(), kCFStringEncodingISOLatin1, 0, true);
    return [ (NSURL *) result autorelease ];
}

@implementation OPluginPDFImplMacBridge

#if DEBUG_RETAIN_RELEASE
	- (void) retain
	{
		[ super retain ];
	}

	- (void) release
	{
		[ super release ];
	}
#endif

//	This returns TRUE if the plugin has been loaded and responds to the correct method.
//	Until it's loaded successfully the first time, it will attempt to laod it every
//	time a PDF widget is created, so the user doesn't have to Quit Apollo if they
//	have to change something in Acrobat to make it work.
+ (NSBundle *) adobePDFViewerBundleForPDFImpl: (WebKitApollo::PluginPDFImplMac *) pluginPDFImpl
{
	static NSBundle *sAdobePDFViewerBundle = nil;
	if ( sAdobePDFViewerBundle == nil )
	{
		int pdfStatus = pluginPDFImpl->getWebFrameOwner()->loadPDFPlugin( reinterpret_cast<void **>(&sAdobePDFViewerBundle) );
		if ( pdfStatus != 0 )
		{
			pluginPDFImpl->getWebFrameOwner()->handlePDFError( pdfStatus );
		}
	}

	return sAdobePDFViewerBundle;
}

typedef enum {
	kPDFImplMacSchemeNotAppPath,
	kPDFImplMacSchemeAppResourcePath,
	kPDFImplMacSchemeAppStoragePath
} PDFImplMacSchemeType;

- (const uint16_t *) pathForSchemeType: (PDFImplMacSchemeType) schemeType returningLength: (unsigned long *) pathLength
{
	//	If this is an app or app-storageURL, we need to map it to the 
	//	real file URL.  We get UTF16 characters back.
	//	We only expand if it's app or app-storage and there is a path.
	*pathLength = 0;
	switch ( schemeType )
	{
		case kPDFImplMacSchemeAppResourcePath:
			return mPluginPDFImpl->getWebFrameOwner()->getAppResourcePath( pathLength );
		case kPDFImplMacSchemeAppStoragePath:
			return mPluginPDFImpl->getWebFrameOwner()->getAppStoragePath( pathLength );
		default:
			return NULL;
	}
}

static PDFImplMacSchemeType URLSchemeTypeAndPath( NSURL *appURL, NSString **appPath )
{
	PDFImplMacSchemeType schemeType = kPDFImplMacSchemeNotAppPath;
	*appPath = nil;

	NSString *URLScheme = [ [ appURL scheme ] lowercaseString ];
	if ( [ URLScheme isEqualToString: @"app" ] )
		schemeType = kPDFImplMacSchemeAppResourcePath;
	else if ( [ URLScheme isEqualToString: @"app-storage" ] )
		schemeType = kPDFImplMacSchemeAppStoragePath;
	else
		return schemeType;

	if ( [ [ appURL absoluteString ] length ] > ( [ URLScheme length ] + 2 ) )
	{
		NSString *urlPath = [ [ appURL absoluteString ] substringFromIndex: [ URLScheme length ] + 1 ];
		if ( [ urlPath hasPrefix: @"/" ] && ([ urlPath length ] > 1) )
		{
			urlPath = [ urlPath substringFromIndex: 1 ];
			if ( ![ urlPath hasPrefix: @"/" ] )
			{
				*appPath = urlPath;
				return schemeType;
			}
			else
			{
				#if !defined(NDEBUG) || !NDEBUG
					NSLog( @"URL of %@, too many slashes, not mapping", [ appURL absoluteString ] );
				#endif
			}
		}
		else
		{
			#if !defined(NDEBUG) || !NDEBUG
				NSLog( @"URL of %@, no slashes, not mapping", [ appURL absoluteString ] );
			#endif
		}
	}
	
	//	Didn't work out.
	return kPDFImplMacSchemeNotAppPath;
}


//	Starting with an app or app-storage URL, map it to a file URL.
//	We assume the original URL is percent-encoded with UTF8 encoding.
- (NSURL *) URLByExpandingCustomSchemesFor: (NSURL *) origURL
{
	NSString *urlPath = nil;
	PDFImplMacSchemeType schemeType = URLSchemeTypeAndPath( origURL, &urlPath );
	if ( schemeType == kPDFImplMacSchemeAppResourcePath || schemeType == kPDFImplMacSchemeAppStoragePath )
	{
		//	appPath returned is a singleton so we don't own it.
		unsigned long appPathLength = 0;
		uint16_t *appPathUTF16 = const_cast<uint16_t *>( [ self pathForSchemeType: schemeType returningLength: &appPathLength ] );
		if ( appPathUTF16 != NULL )
		{
			NSString *mappedFilePath = [ NSString stringWithCharacters: reinterpret_cast<const unichar *>(appPathUTF16) length: appPathLength ];

			if ( mappedFilePath != nil )
			{
				NSString *fullPath = [ NSString stringWithFormat: @"%@/%@", mappedFilePath, urlPath ];
				return [ NSURL fileURLWithPath: fullPath ];
			}
			else
			{
				#if !defined(NDEBUG) || !NDEBUG
					NSLog( @"URL of %@, got nil mapped file path", [ origURL absoluteString ] );
				#endif
			}
		}
		else
		{
			#if !defined(NDEBUG) || !NDEBUG
				NSLog( @"URL of %@, got nil app path", [ origURL absoluteString ] );
			#endif
		}
	}

	return origURL;
}

- (NSURL *) URLByContractingCustomSchemesFor: (NSURL *) origURL
{
	NSString *urlPath = nil;
	PDFImplMacSchemeType schemeType = URLSchemeTypeAndPath( origURL, &urlPath );
	switch ( schemeType )
	{
		case kPDFImplMacSchemeAppResourcePath:
			return [ NSURL URLWithString: [ NSString stringWithFormat: @"app:/%@", urlPath ] ];
		break;

		case kPDFImplMacSchemeAppStoragePath:
			return [ NSURL URLWithString: [ NSString stringWithFormat: @"app-storage:/%@", urlPath ] ];
		break;

		default:
			return origURL;
	}
	
}

- (OPluginPDFImplMacBridge *) initWithPluginPDFImpl: (WebKitApollo::PluginPDFImplMac *) pluginPDFImpl
{
	self = [ super init ];
	if ( self != nil )
	{
		mPluginView = nil;
		mPluginPDFImpl = pluginPDFImpl;
		mPluginArguments = nil;
		mPluginSuperview = nil;
		mStarted = NO;
		#if IMP_EXTERNAL_RESOURCE_LOADER
			mCurrentPluginClients = nil;
		#endif
		mVisible = NO;
        mPluginVisible = NO;
        mEnabled = NO;
        
		NSBundle *pluginBundle = [ OPluginPDFImplMacBridge adobePDFViewerBundleForPDFImpl: pluginPDFImpl ];
		if ( pluginBundle != nil )
		{
			Class viewFactory = [ pluginBundle principalClass ];

			//"arguments" argument, dictionary of attributes.  AdobePDFViewer is expecting from WebKit
			//   Key                         value                 Use (* == must exist)
			// "WebPlugInContainerKey"      WebFrame           getting datasource (avoiding double-loading)
			//                                                 loading non-PDF weblinks
			//
			// "WebPlugInBaseURLKey"        NSString           base URL if URL below is relative
			//                                            
			// "WebPlugInAttributesKey"     NSDictionary       object/embed parameters
			//                     "type"            NSString         mimeType
			//                    *"src" or "data"   NSString         URL source

			//AdobePDFViewer is expecting from ApolloWebKit (or non-normal WebKit)
			// "AcrobatAttributesKey"     NSDictionary       acrobat-specific parameters
			//                     "Environment"     NSString         kind of webkit ("ApolloWebKit")
			//                     "ResourceLoader"  id               class to use instead of NSURLConnection
			//                     "WebFrame"        id<WebFrameCalls>    class to use instead of webFrame
			//	... this is passed into Acrobat as a CFDictionary, so Apollo can send things through AdobePDFViewer
			
			mPluginArguments = [ [ NSMutableDictionary dictionaryWithCapacity: 3 ] retain ];
			NSMutableDictionary *attributes = [ NSMutableDictionary dictionaryWithCapacity: 2 ];

			NSURL *url = NSURLFromKURL( mPluginPDFImpl->getUrl() );
			NSURL *mappedURL = [ self URLByExpandingCustomSchemesFor: url ];
			if ( mappedURL == nil )	//paranoia
				mappedURL = [ NSURL URLWithString: @"about:blank" ];
			[ attributes setObject: [ mappedURL absoluteString ] forKey:  @"src" ];

			[ attributes setObject: [ NSString stringWithCString: mPluginPDFImpl->getMimeType().utf8().data() encoding: NSUTF8StringEncoding ] forKey:  @"type" ];
			[ mPluginArguments setObject: attributes forKey: @"WebPlugInAttributesKey" ];
			
			//Special argument for AdobePDFViewer so it knows it's running in ApolloWebKit
			//this is not in System WebKit.
			NSMutableDictionary *acroAttributes = [ NSMutableDictionary dictionaryWithCapacity: 2 ];
			[ acroAttributes setObject: @"ApolloWebKit" forKey: @"Environment" ];
			[ acroAttributes setObject: self forKey: @"ApolloWebKitPDFPluginHost" ];
			
			//	Get the URL of the container (the frame) and if it's different, send it in.
			WTF::RefPtr<WebCore::Frame> frame = pluginPDFImpl->getWebFrameOwner()->frame();
			NSURL *frameURL = NSURLFromKURL( frame->loader()->url() );
			
			//	If the frame has a different URL, then it's our container
			NSURL *mappedFrameURL = [ self URLByExpandingCustomSchemesFor: frameURL ];
			if ( ! [ [ [ mappedFrameURL absoluteString ] lowercaseString ] isEqualToString: [ [ mappedURL absoluteString ] lowercaseString ] ] )
				[ acroAttributes setObject: mappedFrameURL forKey: @"HostContainer" ];

			//	AdobePDFViewer has to know what kind of request was made for this response, since it
			//	will have to duplicate the request.  It does this through a WebDataSource object and
			//	gets a request object to see what it is.  Create one from the one we're given; if
			//	we're just doing a GET, we don't have to make this, only if it's a POST.
			//	We assume that no POST will be app or app-storage so we don't use mappedURLs.
			const WebCore::ResourceRequest& resourceRequest = frame->loader()->originalRequest();
			if ( resourceRequest.httpMethod() == "POST" )
			{
				NSMutableURLRequest *nsRequest = [ NSMutableURLRequest requestWithURL: NSURLFromKURL( resourceRequest.url() ) ];
				[ nsRequest setHTTPMethod: @"POST" ];
				Vector<char> postData;
				resourceRequest.httpBody()->flatten(postData);
				if ( postData.size() > 0 )
				{
					NSData *nsData = [ NSData dataWithBytes: postData.data() length: postData.size() ];
					[ nsRequest setHTTPBody: nsData ];
				}
				
				WebCore::HTTPHeaderMap headerMap = resourceRequest.httpHeaderFields();
				unsigned long numHeaders = headerMap.size();
				NSMutableDictionary *hdrDict = [ NSMutableDictionary dictionaryWithCapacity: numHeaders ];

				for(WebCore::HTTPHeaderMap::iterator i = headerMap.begin(); i != headerMap.end(); ++i)
				{
					const WebCore::String name = i->first;
					const WebCore::String value = i->second;
					if ( name.length() > 0 && value.length() > 0 )
						[ hdrDict setValue: NSStringFromWebCoreString( value ) forKey: NSStringFromWebCoreString( name ) ];
				}
				
				[ nsRequest setAllHTTPHeaderFields: hdrDict ];
				[ acroAttributes setObject: nsRequest forKey: @"HostRequest" ];
			}

			//	We have a slightly different class for javascript undefined objects (it's actually
			//	exactly the same code but the obj-c class is renamed to not conflict with the
			//	system Webkit used by Acrobat... but when it returns the 'undefined' value it
			//	has to be an exact match with ours, so we pass it in).
			
			[ acroAttributes setObject: [ WebUndefined undefined ] forKey: @"WebUndefined" ];

			//	This concludes the arguments inside AcrobatAttributes
			[ mPluginArguments setObject: acroAttributes forKey: @"AcrobatAttributesKey" ];

			//	For POSTing, we need a webframe host that supports some WebFrame methods.
			OPluginPDFWebFrameBridge *webFrameBridge = [ [ [ OPluginPDFWebFrameBridge alloc ] initWithMacBridge: self
																					andTargetFrame: frame.get() ]
																	autorelease ];
			[ mPluginArguments setObject: webFrameBridge forKey: @"WebPlugInContainerKey" ];
			
			mPluginView = [ [ viewFactory plugInViewWithArguments: mPluginArguments ] retain ];
			[ mPluginView webPlugInInitialize ];
		}
	}
	
	return self;
}

- (NSView<WebPlugInViewFactory,WebPlugin> *) pluginView
{
	return mPluginView;
}

- (void) start
{
	//	We might not have been able to load.
	if ( mPluginView != nil )
	{
		//	TBD: get the NSView the right way, which we don't know yet.  Currently
		//	there is only one AIRContentWindow so we'll find the first one.
		WTF::RefPtr<WebCore::IScrollViewApolloImpl> scrollViewImpl = mPluginPDFImpl->getWebFrameOwner()->frame()->view()->getApolloImpl();

		NSWindow* window = scrollViewImpl->getPlatformWindow();

		ASSERT( window != nil );
		if ( window != nil )
		{
			NSView *contentView = [ window contentView ];

			DumpViewTree( window );
            
			//	We set the plug-in view's size manually; the superview is used to clip
			//	the plug-in view within the HTML content, so it shouldn't auto-resize
			//	its subview.
			mPluginSuperview = [ [ PluginSuperview alloc ] init ];
			[ mPluginSuperview setAutoresizesSubviews: NO ];
            [ mPluginSuperview setAutoresizingMask: NSViewMaxXMargin|NSViewMinYMargin ];
            
			[ contentView addSubview: mPluginSuperview ];
		}
	}
}

- (void) stop
{
	//	We are *completely* done.  Make sure we're gone from everywhere
	[ self setEnabled: NO ];

	[ mPluginView webPlugInDestroy ];
	[ mPluginView release ];
	mPluginView = nil;

	[ mPluginSuperview release ];
	mPluginSuperview = nil;

	#if IMP_EXTERNAL_RESOURCE_LOADER
		[ mCurrentPluginClients release ];
		mCurrentPluginClients = nil;
	#endif

	[ mPluginArguments release ];
	mPluginArguments = nil;
}

	//	We have to translate the frame to Cocoa coordinates, which is a pain:
	//	 * adl uses Carbon coordinates, which is (0,0) at the top-left of the main screen and y goes down.
	//	 * Cocoa coordinates are (0,0) at the bottom-left of the main screen and y goes up.
	//	For frames in NSViews, the coords are relative to the superview's bounds, so the location
	//	and height of the main screen is irrelevant. (for any NSView, 'frame' is in the coords
	//	of the superview, and 'bounds' is its own coord system that subviews' frames are in. By
	//	setting an NSView's frame, you are placing it in its superview; by setting an NSView's
	//	bounds, you are moving its subviews around).
	//	The incoming rect is relative to the window; this may be harder to figure out?
- (NSRect) pluginRectFromApolloWindowRect: (const WebCore::IntRect&) windowRect
{
	NSRect newFrame;
	newFrame.origin.x = windowRect.x();
	newFrame.size.width = windowRect.width();
	newFrame.size.height = windowRect.height();
	newFrame.origin.y = [ [ mPluginSuperview superview ] bounds ].size.height - (windowRect.y() + windowRect.height());
	return newFrame;
}

- (void) setFrameToRect: (const WebCore::IntRect&) rect clippingBy: (const WebCore::IntRect&) clipRect
{
    WebCore::IntRect newClipRect = rect;
    newClipRect.intersect(clipRect);

	//	We use the superView to do the clipping
	NSRect newClip = [ self pluginRectFromApolloWindowRect: newClipRect ];
    
#if LOG_PDFIMPL_CALLS
    NSLog( @"pluginMacBridge(0x%08X) setFrame(0x%08X) %s", self, mPluginSuperview, SprintfNSRect(&newClip) );
#endif
	[ mPluginSuperview setFrame: newClip ];
    
    //	The routine returns the value according to the mPluginSuperView frame, not bounds,
	//	so when we set the pluginView's frame we want to account for the offset of the
	//	superview's bounds.
	NSRect newFrame = [ self pluginRectFromApolloWindowRect: rect ];
	newFrame = NSOffsetRect( newFrame, -newClip.origin.x, -newClip.origin.y );
    
	#if LOG_PDFIMPL_CALLS
        NSLog( @"pluginMacBridge(0x%08X) setFrame(0x%08X) %s", self, mPluginView, SprintfNSRect(&newFrame) );
    #endif
        
	[ mPluginView setFrame: newFrame ];
}

- (void) setNeedsDisplay: (const WebCore::IntRect&) windowDirtyRect
{
	NSRect pluginDirtyRect = [ self pluginRectFromApolloWindowRect: windowDirtyRect ];
	[ mPluginView setNeedsDisplayInRect: pluginDirtyRect ];
}

- (void) setVisibility: (BOOL) visibility
{
    mVisible = visibility;
}

- (void) setEnabled: (BOOL)enabled
{
    mEnabled = enabled;
    
    const bool showPlugin = mEnabled && mVisible;
    
	if ( showPlugin != mPluginVisible )
	{
		//	We may never have been able to load it.
		if ( mPluginSuperview != nil )
		{
			if ( showPlugin )
			{                
				ASSERT( [ mPluginView superview ] == nil );
				if ( [ mPluginView superview ] == nil )			// paranoia
				{
                    [mPluginSuperview setHidden: NO];
                    
					[ mPluginSuperview addSubview: mPluginView ];
					[ mPluginView webPlugInStart ];
                    mPluginVisible = showPlugin;
				}
			}
			else
			{
				ASSERT( [ mPluginView superview ] == mPluginSuperview );
				if( [ mPluginView superview ] != nil )			// paranoia
				{
					[ mPluginView webPlugInStop ];
					[ mPluginView removeFromSuperview ];
                    mPluginVisible = showPlugin;
                    
                    [mPluginSuperview setHidden: YES];
				}
			}
		}
	}
}

- (WebKitApollo::PluginPDFImplMac*) pluginPDFImpl
{
	return mPluginPDFImpl;
}

- (void) dealloc
{
	[ self stop ];
	[ super dealloc ];
}

@end

@implementation OPluginPDFWebFrameBridge

- (OPluginPDFWebFrameBridge *) initWithMacBridge: (OPluginPDFImplMacBridge *) macBridge andTargetFrame: (WebCore::Frame *) targetFrame
{
	self = [ super init ];
	if ( self != nil )
	{
		mMacBridge = [ macBridge retain ];
		mTargetFrame = targetFrame;
	}
	return self;
}

- (void) errorLoadingAcrobat: (SInt32) error
{
	#if LOG_PDFIMPL_CALLS
		NSLog (@"OPluginPDFImplMacBridge(%08x) : errorLoadingAcrobat %d", self, error );
	#endif
}

- (void) loadRequest: (NSURLRequest *) request
{
	#if LOG_PDFIMPL_CALLS
		NSLog (@"OPluginPDFImplMacBridge(%08x) : loadRequest %@", self, [ request URL ] );
	#endif

	//	We passed Acrobat a mapped URL (app or app-storage) so we need
	//	to do the same mapping in reverse.
	WebCore::KURL urlRequested( [ mMacBridge URLByContractingCustomSchemesFor: [ request URL ] ] );
	WebCore::KURL urlSource = [ mMacBridge pluginPDFImpl ]->getUrl();
	if ( [ mMacBridge pluginPDFImpl ]->allowURLRequest( urlSource, urlRequested, mTargetFrame ))
	{
		{
			if ( mTargetFrame != NULL )
				[ mMacBridge pluginPDFImpl ]->loadNSURLRequestInFrame( request, mTargetFrame );
			else
			{
				//	TBD: Find out if we can open windows and do so; otherwise load it here.
				[ mMacBridge pluginPDFImpl ]->loadNSURLRequestInFrame( request, NULL );

				#if 0	/* Security hole: don't launch Safari */
					FSRef safariRef;
					NSURL *safariURL;
					if ( noErr == LSGetApplicationForURL( (CFURLRef) [ NSURL URLWithString: @"http://www.adobe.com" ], kLSRolesAll, &safariRef, (CFURLRef *) &safariURL ) )
					{
						LSLaunchURLSpec launchURLSpec;
						BlockZero(&launchURLSpec, sizeof(launchURLSpec));
						launchURLSpec.appURL = (CFURLRef) safariURL;	// LSFindApplicationForInfo retains safariUrl, so we release it below
						CFURLRef destURL = (CFURLRef) [ request URL ];
						launchURLSpec.itemURLs = CFArrayCreate(kCFAllocatorDefault, (const void **) &destURL, 1, &kCFTypeArrayCallBacks);
						(void) LSOpenFromURLSpec( &launchURLSpec, NULL );
						CFRelease( safariURL );
					}
				#endif
			}
		}
	}
}

- (NSObject<ApolloWebKitWebFramePluginHost> *) findFrameNamed: (NSString *) name
{
	#if LOG_PDFIMPL_CALLS
		NSLog (@"OPluginPDFImplMacBridge(%08x) : findFrameNamed %@", self, name );
	#endif
	
	const char *utf8Name = [ name UTF8String ];
	ASSERT(utf8Name);
	if ( utf8Name == NULL )
		return nil;		// badly-formed frame name

	//	'self' is still attached to the macbridge for this plugin, but the
	//	target frame needs to be different.  We will get a loadRequest later
	//	and we want to make sure it goes to the right frame.
	WebCore::Frame *foundFrame = mTargetFrame->tree()->find( utf8Name );
	if ( foundFrame == mTargetFrame )
		return self;
	else if (	foundFrame == NULL
			&&	![ name isEqualToString: @"_blank" ]
			&&	![ name isEqualToString: @"_new" ] )
	{
		//	Nope, not a reserved name and we can't find it.  Drop it on the floor.
		return nil;
	}
	else
	{
		//	AdobePDFViewer does not expect to own this and will retain it if it wants.
		//	If the frame is nil, we'll know to open the default window.  Note that will
		//	only work for GETs.
		OPluginPDFWebFrameBridge *webFrameBridge = [ [ [ OPluginPDFWebFrameBridge alloc ]
																initWithMacBridge: mMacBridge
																andTargetFrame: foundFrame ]
														autorelease ];
		return webFrameBridge;
	}
}

- (id) dataSource
{
	return nil;
}

- (id) webView
{
	return nil;
}


//	This is called by AdobePDFViewer to get the webFrame it's in because we are
//	passed ins the WebPluginContainerKey... so we are doing double duty as both
//	the "WebPluginContainer" and the "WebFrame".  
- (NSObject<ApolloWebKitWebFramePluginHost> *) webFrame
{
	return self;
}

- (void) dealloc
{
	[ mMacBridge release ];
	mTargetFrame = nil;
	[ super dealloc ];
}



@end

#if IMP_EXTERNAL_RESOURCE_LOADER

	//	AdobePDFViewer calls us with these when it's ready to load, and when
	//	Acrobat needs to communicate.
	@interface OPluginPDFImplMacBridge (ApolloWebKitPDFPluginHost)

		- (void) startLoadingURL: (NSURL *) URL
					forClient: (id<ApolloWebKitPDFPluginURLLoadingClient>) client;
		- (void) startLoadingURL: (NSURL *) URL
					forClient: (id<ApolloWebKitPDFPluginURLLoadingClient>) client
					withMethod: (NSString *) method
					andData: (NSData *) data;
		- (void) stopLoadingURL: (NSURL *) URL
					forClient: (id<ApolloWebKitPDFPluginURLLoadingClient>) client;
		
	@end

	@interface OPluginPDFImplMacBridge (ResourceLoading)
	/*
		- (void) client: (id) client receivedRedirect: (NSURL *) URL;
		- (void) client: (id) client receivedResponse: (NSURLResponse *) response;
		- (void) client: (id) client receivedData: (const char *) data;
		- (void) clientReceivedAllData: (id) client;
	*/

		- (void) client: (void *) client receivedRedirect: (const WebCore::KURL&) URL;
		- (void) client: (void *) client receivedResponse: (WebCore::PlatformResponse) response fromLoader: (WebCore::ResourceLoader *) loader;
		- (void) client: (void *) client receivedData: (const char*) data length: (int) length;
		- (void) clientReceivedAllData: (void *) client;

	@end

	@implementation OPluginPDFImplMacBridge (ApolloWebKitPDFPluginHost)

	- (void) startLoadingURL: (NSURL *) URL
				forClient: (id<ApolloWebKitPDFPluginURLLoadingClient>) client
	{
		[ self startLoadingURL: URL forClient: client withMethod: @"GET" andData: nil ];
	}


	- (void) startLoadingURL: (NSURL *) URL
				forClient: (id<ApolloWebKitPDFPluginURLLoadingClient>) client
				withMethod: (NSString *) method
				andData: (NSData *) data
	{
		if ( mCurrentPluginClients == nil )
			mCurrentPluginClients = [ [ NSMutableArray alloc ] initWithCapacity: 5 ];
		[ mCurrentPluginClients addObject: client ];
		WebCore::KURL url( URL );
		mPluginPDFImpl->pluginLoadURLRequest( url, static_cast<void *>(client) );
	}


	- (void) stopLoadingURL: (NSURL *) URL
				forClient: (id<ApolloWebKitPDFPluginURLLoadingClient>) client
	{
		//	We may get things out-of-order; make sure we still have this guy.
		if ( [ mCurrentPluginClients containsObject: (id) client ] )
			[ mCurrentPluginClients removeObject: (id) client ];
	}
		
	@end

	@implementation OPluginPDFImplMacBridge (ResourceLoading)

	//	We received a redirect: tell the client about that.  We don't have any way
	//	of preventing this; presumably the Apollo environment has handled security.
	- (void) client: (void *) client receivedRedirect: (const WebCore::KURL&) URL
	{
		if ( [ mCurrentPluginClients containsObject: (id) client ] )
		{
			if ( [ (id) client respondsToSelector: @selector(receivedRedirect:) ] )
			{
				[ (id<ApolloWebKitPDFPluginURLLoadingClient>) client receivedRedirect: NSURLFromKURL( URL ) ];
			}
		}
	}

	//	We received a response; tell the client about it.  There is no object that
	//	encapsulates headers and status and the request (NSMutableURLResponse would
	//	be great but it doesn't exist),
	- (void) client: (void *) client receivedResponse: (WebCore::PlatformResponse) response fromLoader: (WebCore::ResourceLoader *) loader
	{
		if ( [ mCurrentPluginClients containsObject: (id) client ] )
		{
			if ( [ (id) client respondsToSelector: @selector(receivedResponseOfExpectedLength:withHeaders:andStatusCode:andMIMEType:) ] )
			{
				SInt64 contentLength = mPluginPDFImpl->resourceLoaderContentLength( loader );
				NSMutableDictionary *headers = [ NSMutableDictionary dictionaryWithCapacity: 10 ];
				[ headers setObject: [ NSNumber numberWithInt: contentLength ] forKey: @"Content-Length" ];
				WebCore::String contentType = loader->queryMetaData("HTTP_CONTENT-TYPE");
				NSString *nsContentType = [ NSStringFromWebCoreString( contentType ) stringByTrimmingCharactersInSet: [ NSCharacterSet whitespaceCharacterSet ] ];
				if ( nsContentType != NULL )
					[ headers setObject: nsContentType forKey: @"Content-Type" ];
				[ (id<ApolloWebKitPDFPluginURLLoadingClient>) client receivedResponseOfExpectedLength: contentLength
							withHeaders: headers andStatusCode: response->m_httpStatusCode
							andMIMEType: nsContentType ];
			}
		}
	}

	- (void) client: (void *) client receivedData: (const char*) data length: (int) length
	{
		if ( [ mCurrentPluginClients containsObject: (id) client ] )
		{
			if ( [ (id) client respondsToSelector: @selector(receivedData:) ] )
			{
				[ (id<ApolloWebKitPDFPluginURLLoadingClient>) client receivedData: [ NSData dataWithBytes: data length: length ] ];
			}
		}
	}

	- (void) clientReceivedAllData: (void *) client
	{
		if ( [ mCurrentPluginClients containsObject: (id) client ] )
		{
			if ( [ (id) client respondsToSelector: @selector(receivedAllData) ] )
			{
				[ (id<ApolloWebKitPDFPluginURLLoadingClient>) client receivedAllData ];
			}

			[ mCurrentPluginClients removeObject: (id) client ];
		}
	}

	@end

#endif


//Mac implementation of PluginPDFImpl
namespace WebKitApollo
{
	PluginPDFImplMac::PluginPDFImplMac( WebCore::Element* const element,
										const WebCore::KURL& url,
										const Vector<WebCore::String>& paramNames, 
										const Vector<WebCore::String>& paramValues,
										const WebCore::String& mimeType,
										WebFrameImpl* const pWebFrameOwner )
		:	PluginWK( element, url, paramNames, paramValues, mimeType, pWebFrameOwner )
		,	mObjCBridge( nil )
		,	mEverGotNonzeroSize( false )
	{
		load();
	}

	PluginPDFImplMac::~PluginPDFImplMac()
	{
		//	Even though 'dealloc' calls 'stop', there is a circular retain
		//	reference on it so we have to stop it to get everything else
		//	to release the mObjCBridge first.
		[ (OPluginPDFImplMacBridge *) mObjCBridge stop ];
		[ (OPluginPDFImplMacBridge *) mObjCBridge release ];
	}

	//Load the plug-in: return true if it succeeded
	bool PluginPDFImplMac::load()
	{
		//Load the plug-in with the objective-c thing.
		if ( mObjCBridge == nil )
			mObjCBridge = (void *) [ [ OPluginPDFImplMacBridge alloc ] initWithPluginPDFImpl: this ];
		return ( [ (OPluginPDFImplMacBridge *) mObjCBridge pluginView ] != nil );
	}
	
	void PluginPDFImplMac::frameGeometryHasChanged( const WebCore::IntRect &oldRect, const WebCore::IntRect& windowRect, const WebCore::IntRect& windowClipRect )
	{
		WebCore::IntRect rect = frameGeometry();

		#if LOG_PDFIMPL_CALLS
			NSLog( @"PluginPDFImplMac::changeFrameGeometry [ %5d %5d ] [ %5d %5d ]", rect.x(), rect.y(), rect.width(), rect.height() );
		#endif
		
		//	If this is the first nonzero size we get, attach the view to the superview,
		//	resize the view and start it.
		if ( !mEverGotNonzeroSize && !rect.isEmpty() )
		{
			mEverGotNonzeroSize = true;
			[ (OPluginPDFImplMacBridge *) mObjCBridge start ];
		}
		
		//	We don't tell the plugin to start until we get a nonzero size; once we get
		//	one, though, we tell it resize even if it's zero (so resizing to nonzero,
		//	then to zero, will work).
		if ( mEverGotNonzeroSize || !windowRect.isEmpty() )
			[ (OPluginPDFImplMacBridge *) mObjCBridge setFrameToRect: windowRect clippingBy: windowClipRect ];
	}

	void * PluginPDFImplMac::createPluginScriptableObject()
	{
		return static_cast<void *>( [ [ (OPluginPDFImplMacBridge *) mObjCBridge pluginView ] objectForWebScript ] );
	}

	void PluginPDFImplMac::releasePluginScriptableObject(void * object)
	{
	}

	void PluginPDFImplMac::paint( WebCore::GraphicsContext* pGraphicsContext, const WebCore::IntRect& dirtyRect )
	{
		#if LOG_PDFIMPL_CALLS
			NSLog( @"PluginPDFImplMac::paint [ %d %d ] [ %d %d ]", dirtyRect.x(), dirtyRect.y(), dirtyRect.width(), dirtyRect.height() );
		#endif
		
		//Paint rect doesn't clip to our rect: do so now.
		WebCore::IntRect windowDirtyRect = dirtyRect;
		windowDirtyRect.intersect( frameGeometry() );
		
		//We do paint differently: we have to tell it to update, but not synchronously.
		windowDirtyRect.setLocation( contentsToWindow( windowDirtyRect.location() ) );
		
		[ (OPluginPDFImplMacBridge *) mObjCBridge setNeedsDisplay: windowDirtyRect ];
	}
    
    void PluginPDFImplMac::show()
    {
        // show doesn't actually enable the plug-in, only updatePluginWindow can do that
        [ (OPluginPDFImplMacBridge *) mObjCBridge setVisibility: YES ];
    }
    
    void PluginPDFImplMac::hide()
    {
        [ (OPluginPDFImplMacBridge *) mObjCBridge setVisibility: NO ];
    }
	
    void PluginPDFImplMac::updatePluginWindow(bool, bool showWindowedPlugins)
	{
	    if(mObjCBridge == nil)
	        return;
        
        [ (OPluginPDFImplMacBridge *) mObjCBridge setEnabled: showWindowedPlugins ];
            
		//	All events, including paint, may indicate that we have to
		//	move/resize.  This is because our drawing model (nested NSViews)
		//	doesn't match the Apollo drawing model (one NSView).  Setting
		//	us to the same geometry will make this call do the window-
		//	relative math and all the clipping stuff, and call back
		//	frameGeometryHasChanged if it really has.
		setFrameGeometry( frameGeometry() );
    }
    
	void PluginPDFImplMac::loadNSURLRequestInFrame( void *request, WebCore::Frame *targetFrame )
	{
		//	void * so C++ classes that include this class don't have to know about NSURLRequest class.
		NSURLRequest *nsurlRequest = reinterpret_cast<NSURLRequest *>(request);

		//	Get the loadURL parameters from the obj-c request object
		WebCore::KURL url( [ nsurlRequest URL ] );
		const char *method = [ [ nsurlRequest HTTPMethod ] UTF8String ];
		const char *data = static_cast<const char *>([ [ nsurlRequest HTTPBody ] bytes ]);
		unsigned int dataLength = [ [ nsurlRequest HTTPBody ] length ];
		
		//	Make the headers into key:value pairs separated by "\r\n"
		WebCore::HTTPHeaderMap urlHeaders;
		NSDictionary *reqHeaders = [ nsurlRequest allHTTPHeaderFields ];
		NSEnumerator *keyEnum = [ reqHeaders keyEnumerator ];
		NSString *key;
		NSCharacterSet *illegalCharSet = [ NSCharacterSet characterSetWithCharactersInString: @"\r\n" ];
		while ( ( key = (NSString *) [ keyEnum nextObject ] ) != nil )
		{
			//	We will add the usual headers, only find the custom ones.
			if ( NSOrderedSame != [ key caseInsensitiveCompare: @"Content-length" ] )
			{
				NSString *value = [ reqHeaders valueForKey: key ];
				
				//	Security: no \r or \n in the key or the value.
				if (	( NSNotFound == [ key rangeOfCharacterFromSet: illegalCharSet ].location ) &&
						( NSNotFound == [ value rangeOfCharacterFromSet: illegalCharSet ].location ) )
					urlHeaders.add([ key UTF8String ], [ value UTF8String ]);
			}
		}

		//	Load it: this may go to another frame.  AdobePDFViewer plug-in will itself handle
		//	any loads that go back to it; our method gets called only when it's expecting
		//	html to come back so the frame itself should load it.
		loadURLForClient( url, method, data, dataLength, nil, urlHeaders, targetFrame );
	}



	#if IMP_EXTERNAL_RESOURCE_LOADER

		void PluginPDFImplMac::pluginLoadURLRequest( WebCore::KURL& url, void *pluginClient )
		{
			loadURLForClient( url, pluginClient );
		}

		void PluginPDFImplMac::clientReceivedRedirect(void *client, WebCore::ResourceLoader*, const WebCore::KURL& url)
		{
			#if LOG_PDFIMPL_LOADER_CALLBACKS
				NSLog( @"PluginPDFImplMac::clientReceivedRedirect" );
			#endif
			[ (OPluginPDFImplMacBridge *) mObjCBridge client: client receivedRedirect: url ];
		}

		void PluginPDFImplMac::clientReceivedResponse(void *client, WebCore::ResourceLoader *resourceLoader, WebCore::PlatformResponse response)
		{
			#if LOG_PDFIMPL_LOADER_CALLBACKS
				NSLog( @"PluginPDFImplMac::clientReceivedResponse" );
			#endif
			[ (OPluginPDFImplMacBridge *) mObjCBridge client: client receivedResponse: response fromLoader: resourceLoader ];
		}

		void PluginPDFImplMac::clientReceivedData(void *client, WebCore::ResourceLoader*, const char* data, int length)
		{
			#if LOG_PDFIMPL_LOADER_CALLBACKS
				NSLog( @"PluginPDFImplMac::clientReceivedData" );
			#endif
			[ (OPluginPDFImplMacBridge *) mObjCBridge client: client receivedData: data length: length ];
		}

		void PluginPDFImplMac::clientReceivedAllData(void *client, WebCore::ResourceLoader*)
		{
			#if LOG_PDFIMPL_LOADER_CALLBACKS
				NSLog( @"PluginPDFImplMac::clientReceivedAllData 1" );
			#endif
			[ (OPluginPDFImplMacBridge *) mObjCBridge clientReceivedAllData: client ];
		}

		void PluginPDFImplMac::clientReceivedAllData(void *client, WebCore::ResourceLoader*, WebCore::PlatformData)
		{
			#if LOG_PDFIMPL_LOADER_CALLBACKS
				NSLog( @"PluginPDFImplMac::clientReceivedAllData 2" );
			#endif
			[ (OPluginPDFImplMacBridge *) mObjCBridge clientReceivedAllData: client ];
		}
	#endif
}

