/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "WebWindowScrollView.h"

#include <Frame.h>

#include <WebKitApollo/WebKit.h>
#include <Widget.h>

#include "WebFrameImpl.h"
#include <GraphicsContext.h>
#include "WebViewImpl.h"
#include "FrameView.h"
#include <PlatformMouseEvent.h>
#include <Cursor.h>

namespace WebKitApollo
{
WindowScrollViewImpl::WindowScrollViewImpl( WebWindow* const pWebWindow
                                          , WebFrameImpl* const pWebFrameOwner
                                          , WebCore::FrameView* const owningFrameView
                                          , IFocusManager* const pFocusManager
                                          , IMouseCaptureManager* const pMouseCaptureManager )
    : FrameScrollViewImpl(pWebFrameOwner, owningFrameView, pFocusManager, pMouseCaptureManager)
    , m_pWebWindow(pWebWindow)
    , m_pWebFrameOwner(pWebFrameOwner)
    , m_hostDrawsScrollbars(true)
{
}

WTF::PassRefPtr< WindowScrollViewImpl >
    WindowScrollViewImpl::construct( WebWindow* const pWebWindow
                                   , WebFrameImpl* const pWebFrameOwner
                                   , WebCore::FrameView* const owningFrameView
                                   , IFocusManager* const pFocusManager
                                   , IMouseCaptureManager* const pMouseCaptureManager )
{
    return adoptRef(new WindowScrollViewImpl( pWebWindow
                                            , pWebFrameOwner
                                            , owningFrameView
                                            , pFocusManager
                                            , pMouseCaptureManager));
}

void WindowScrollViewImpl::setHostDrawsScrollbars(bool hostDrawsScrollbars)
{
    m_hostDrawsScrollbars = hostDrawsScrollbars;
    m_lastKnownClientRect = WebCore::IntRect();
    invalidateRect(clientRect());
}

void WindowScrollViewImpl::paint(WebCore::GraphicsContext* pGraphicsContext, const WebCore::IntRect& dirtyRect)
{
    if (m_hostDrawsScrollbars) {
        pGraphicsContext->save();
        WebCore::IntRect cRect(clientRect());
        pGraphicsContext->clip(cRect);
        pGraphicsContext->clip(dirtyRect);
        WebCore::FloatSize const translateOffset(-scrollOffset());
        pGraphicsContext->translate(translateOffset.width(), translateOffset.height());

        WebCore::IntRect contentDirtyRect(viewportToContents(dirtyRect.location()), dirtyRect.size());    
        WebCore::FrameView* view = m_pWebFrameOwner->frame()->view();

        // if the scrolloffset is incorrect then set it to a valid value
        //
        if (view->contentsY() + view->visibleHeight() > view->contentsHeight())
        {
            view->setContentsPos(view->contentsX(), std::max(0, view->contentsHeight() - view->visibleHeight()));
        }
        if (view->contentsX() + view->visibleWidth() > view->contentsWidth())
        {
            view->setContentsPos(std::max(0, view->contentsWidth() - view->visibleWidth()), view->contentsY());
        }

        // if the dirty rect is outside the bounds of the content rect then we need to 
        // constain the dirty rect. 
        //
        if (view->contentsWidth() < contentDirtyRect.x()){
            contentDirtyRect.setX(view->contentsWidth());
        }
        if (view->contentsHeight() < contentDirtyRect.y()){
            contentDirtyRect.setY(view->contentsHeight());
        } 
        if (view->contentsWidth() < (contentDirtyRect.width() + contentDirtyRect.x())){
            int newValue = (view->contentsWidth() - contentDirtyRect.x());
            if ( newValue < 0 )
                newValue = 0;
            contentDirtyRect.setWidth( newValue );
        }
        if (view->contentsHeight() < (contentDirtyRect.height() + contentDirtyRect.y())){
            int newValue = (view->contentsHeight() - contentDirtyRect.y());
            if ( newValue < 0 )
                newValue = 0;
            contentDirtyRect.setHeight( newValue );
        } 
        if (0 > contentDirtyRect.x()){
            contentDirtyRect.setX(0);
        }
        if (0 > contentDirtyRect.y()){
            contentDirtyRect.setY(0);
        } 
        if (0 > (contentDirtyRect.width() + contentDirtyRect.x())){
            contentDirtyRect.setWidth( 0 );
        }
        if (0 > (contentDirtyRect.height() + contentDirtyRect.y())){
            contentDirtyRect.setHeight( 0 );
        } 

        m_pWebFrameOwner->frame()->paint(pGraphicsContext, contentDirtyRect);
        pGraphicsContext->restore();
    }
    else {
        WebCore::IntRect const cRect( clientRect() );
        if ( m_lastKnownClientRect != cRect )
        {
            m_lastKnownClientRect = cRect;
            updateScrollbars();
        }
       
        paintImpl(pGraphicsContext, dirtyRect);
    }
}
    
WebCore::IntRect WindowScrollViewImpl::frameGeometry() const
{
    // For top level windows the frame geometry is just he client rect.
    // The real window's location is determined by the WebChromeClient.
    return clientRect();
}
    
void WindowScrollViewImpl::setCursor( const WebCore::Cursor& cursor )
{
    WebViewImpl* webView = m_pWebFrameOwner->webViewImpl();
    ASSERT( webView );
    WebHost* webHost = webView->webHost();
    ASSERT( webHost );
    webHost->m_pVTable->setMouseCursor( webHost, cursor.impl() );
    FrameScrollViewImpl::setCursor(cursor);
}

void WindowScrollViewImpl::removeFromParent()
{
}

// the meaning of this now is to *actually* be in window coords
WebCore::IntRect WindowScrollViewImpl::windowClipRect() const
{
    WebCore::IntRect rect = clientRect(); // in viewport points
    WebCore::IntPoint p = rect.location();

    p = const_cast<WindowScrollViewImpl*>(this)->viewportToWindow( p );

    rect.setLocation( p );
    
    return rect;
}

void WindowScrollViewImpl::setFrameGeometry(const WebCore::IntRect &rect)
{
    // as far as I can tell, webcore should never try to set the frame geometry of a root frame..
    ASSERT(false);

    // however it is late in the game for M5.  So I'll just implement this anyway.
    WebFloatRect const newWindowRect = {   static_cast<float>(rect.location().x()),
                                           static_cast<float>(rect.location().y()),
                                           static_cast<float>(rect.location().x() + rect.width()),
                                           static_cast<float>(rect.location().y() + rect.height()) };
    m_pWebWindow->m_pVTable->setWindowRect( m_pWebWindow, &newWindowRect );

}

int WindowScrollViewImpl::visibleWidth() const
{
    if (m_hostDrawsScrollbars)
        return m_pWebWindow->m_pVTable->visibleWidth(m_pWebWindow);
    else
        return FrameScrollViewImpl::visibleWidth();
}

int WindowScrollViewImpl::visibleHeight() const
{
    if (m_hostDrawsScrollbars)
        return m_pWebWindow->m_pVTable->visibleHeight(m_pWebWindow);
    else
        return FrameScrollViewImpl::visibleHeight();
}

int WindowScrollViewImpl::contentsWidth() const
{
    if (m_hostDrawsScrollbars)
        return m_pWebWindow->m_pVTable->contentsWidth(m_pWebWindow);
    else
        return FrameScrollViewImpl::contentsWidth();
}

int WindowScrollViewImpl::contentsHeight() const
{
    if (m_hostDrawsScrollbars)
        return m_pWebWindow->m_pVTable->contentsHeight(m_pWebWindow);
    else
        return FrameScrollViewImpl::contentsHeight();
}

int WindowScrollViewImpl::contentsX() const
{
    if (m_hostDrawsScrollbars)
        return m_pWebWindow->m_pVTable->contentsX(m_pWebWindow);
    else
        return FrameScrollViewImpl::contentsX();
}

int WindowScrollViewImpl::contentsY() const
{
    if (m_hostDrawsScrollbars)
        return m_pWebWindow->m_pVTable->contentsY(m_pWebWindow);
    else
        return FrameScrollViewImpl::contentsY();
}

WebCore::IntSize WindowScrollViewImpl::scrollOffset() const
{
    if (m_hostDrawsScrollbars) {
        int scrollXOffset;
        int scrollYOffset;
        m_pWebWindow->m_pVTable->scrollOffset( m_pWebWindow
                                             , &scrollXOffset
                                             , &scrollYOffset);

        WebCore::IntSize const result(scrollXOffset, scrollYOffset);
        return result;
    }
    else
        return FrameScrollViewImpl::scrollOffset();
}

bool WindowScrollViewImpl::scrollBy(int dx, int dy)
{
	bool didScrollResult = false;
	if (m_hostDrawsScrollbars) {
		didScrollResult = m_pWebWindow->m_pVTable->scrollBy(m_pWebWindow, dx, dy);
        if (didScrollResult && frame())
			frame()->sendScrollEvent();
	}
    else
        didScrollResult = FrameScrollViewImpl::scrollBy(dx, dy);
	return didScrollResult;
}

void WindowScrollViewImpl::scrollRectIntoViewRecursively(const WebCore::IntRect& rect)
{
    scrollRectIntoView(rect);
}
    
void WindowScrollViewImpl::setContentsPos(int newX, int newY)
{
    if (m_hostDrawsScrollbars)
        m_pWebWindow->m_pVTable->setContentsPos(m_pWebWindow, newX, newY);
    else
        FrameScrollViewImpl::setContentsPos(newX, newY);
}

namespace
{
    static const WebCore::ScrollbarMode s_ApolloScrollbarModeToWebCoreScrollbarMode[] =
    {
        WebCore::ScrollbarAuto,
        WebCore::ScrollbarAlwaysOff,
        WebCore::ScrollbarAlwaysOn
    };

    static const WebWindowScrollbarMode s_WebCoreScrollbarModeToApolloScrollbarMode[] =
    {
        WebWindowScrollbarModeAuto,
        WebWindowScrollbarModeAlwaysOff,
        WebWindowScrollbarModeAlwaysOn
    };
}

void WindowScrollViewImpl::setVScrollbarMode(WebCore::ScrollbarMode newMode)
{
    if (m_hostDrawsScrollbars)
        m_pWebWindow->m_pVTable->setVScrollbarMode(m_pWebWindow, s_WebCoreScrollbarModeToApolloScrollbarMode[ newMode ]);
    else
        FrameScrollViewImpl::setVScrollbarMode(newMode);
}

void WindowScrollViewImpl::setHScrollbarMode(WebCore::ScrollbarMode newMode)
{
    if (m_hostDrawsScrollbars)
        m_pWebWindow->m_pVTable->setHScrollbarMode(m_pWebWindow, s_WebCoreScrollbarModeToApolloScrollbarMode[ newMode ]);
    else
        FrameScrollViewImpl::setHScrollbarMode(newMode);
}

void WindowScrollViewImpl::suppressScrollbars(bool suppressed, bool repaintOnUnsuppress )
{
    if (m_hostDrawsScrollbars)
        m_pWebWindow->m_pVTable->suppressScrollbars(m_pWebWindow, suppressed, repaintOnUnsuppress == true);
    else
        FrameScrollViewImpl::suppressScrollbars(suppressed, repaintOnUnsuppress);
}

WebCore::ScrollbarMode WindowScrollViewImpl::getVScrollbarMode() const
{
    if (m_hostDrawsScrollbars)
        return s_ApolloScrollbarModeToWebCoreScrollbarMode[m_pWebWindow->m_pVTable->vScrollbarMode(m_pWebWindow)];
    else
        return FrameScrollViewImpl::getVScrollbarMode();
}

WebCore::ScrollbarMode WindowScrollViewImpl::getHScrollbarMode() const
{
    if (m_hostDrawsScrollbars)
        return s_ApolloScrollbarModeToWebCoreScrollbarMode[m_pWebWindow->m_pVTable->hScrollbarMode(m_pWebWindow)];
    else
        return FrameScrollViewImpl::getHScrollbarMode();
}

void WindowScrollViewImpl::resizeContents(int w, int h)
{
    if (m_hostDrawsScrollbars)
		m_pWebWindow->m_pVTable->resizeContents(m_pWebWindow, w, h);
    else
	    FrameScrollViewImpl::resizeContents(w, h);
}

void WindowScrollViewImpl::invalidateRect(const WebCore::IntRect& dirtyRect)
{
    ASSERT(m_pWebWindow);
    ASSERT(m_pWebWindow->m_pVTable);
    ASSERT(m_pWebWindow->m_pVTable->invalidateRect);
    WebCore::IntRect adjustedDirtyRect( dirtyRect );

	// FrameScrollViewImpl::updateContents compensates for the scroll offset before 
	// calling invalidate rect so we don't need to do it here.

	adjustedDirtyRect.intersect(clientRect());
    if (!adjustedDirtyRect.isEmpty()) {
        WebIntRect const invalidApolloRect = { adjustedDirtyRect.x()
                                             , adjustedDirtyRect.y()
                                             , adjustedDirtyRect.right()
                                             , adjustedDirtyRect.bottom()};
        ASSERT(invalidApolloRect.m_left >= 0);
        ASSERT(invalidApolloRect.m_right >= invalidApolloRect.m_left);
        ASSERT(invalidApolloRect.m_right <= clientRect().width());
        ASSERT(invalidApolloRect.m_top >= 0);
        ASSERT(invalidApolloRect.m_bottom >= invalidApolloRect.m_top);
        ASSERT(invalidApolloRect.m_bottom <= clientRect().height());
    
        unsigned char const nowAsWebBool = static_cast<unsigned char>(false);
        ASSERT((nowAsWebBool == 0 ) || (nowAsWebBool == 1 ));
        m_pWebWindow->m_pVTable->invalidateRect(m_pWebWindow, &invalidApolloRect, nowAsWebBool);
    }
}

WebCore::IntPoint WindowScrollViewImpl::viewportToContents(const WebCore::IntPoint& viewportPoint)
{
    if (m_hostDrawsScrollbars) {
        WebIntPoint const inputApolloIntPoint = { viewportPoint.x(), viewportPoint.y() };
        WebIntPoint resultApolloIntPoint;
        m_pWebWindow->m_pVTable->viewportToContents(m_pWebWindow, &inputApolloIntPoint, &resultApolloIntPoint);
        WebCore::IntPoint resultPoint(resultApolloIntPoint.m_x, resultApolloIntPoint.m_y);
        return resultPoint;
    }
    else
        return FrameScrollViewImpl::viewportToContents(viewportPoint);
}

WebCore::IntPoint WindowScrollViewImpl::contentsToViewport(const WebCore::IntPoint& contentsPoint)
{
    if (m_hostDrawsScrollbars) {
        WebIntPoint const inputApolloIntPoint = { contentsPoint.x(), contentsPoint.y() };
        WebIntPoint resultApolloIntPoint;
        m_pWebWindow->m_pVTable->contentsToViewport(m_pWebWindow, &inputApolloIntPoint, &resultApolloIntPoint );
        WebCore::IntPoint const resultPoint( resultApolloIntPoint.m_x, resultApolloIntPoint.m_y );
        return resultPoint;
    }
    else
        return FrameScrollViewImpl::contentsToViewport(contentsPoint);
}

WebCore::IntPoint WindowScrollViewImpl::viewportToWindow(const WebCore::IntPoint& viewportPoint, bool htmlWindowCoords)
{
    if(htmlWindowCoords)
        return viewportPoint;

    WebIntPoint const inputApolloIntPoint = { viewportPoint.x(), viewportPoint.y() };
    WebIntPoint resultApolloIntPoint;
    m_pWebWindow->m_pVTable->viewportToWindow(m_pWebWindow, &inputApolloIntPoint, &resultApolloIntPoint );
    WebCore::IntPoint resultPoint( resultApolloIntPoint.m_x, resultApolloIntPoint.m_y );
    return resultPoint;
}

WebCore::IntPoint WindowScrollViewImpl::windowToViewport(const WebCore::IntPoint& windowPoint)
{
    WebIntPoint const inputApolloIntPoint = { windowPoint.x(), windowPoint.y() };
    WebIntPoint resultApolloIntPoint;
    m_pWebWindow->m_pVTable->windowToViewport(m_pWebWindow, &inputApolloIntPoint, &resultApolloIntPoint);
    WebCore::IntPoint const resultPoint(resultApolloIntPoint.m_x, resultApolloIntPoint.m_y);
    return resultPoint;
}

void WindowScrollViewImpl::setStaticBackground(bool flag)
{
    m_pWebWindow->m_pVTable->setStaticBackground(m_pWebWindow, flag);
}

WebPlatformWindow WindowScrollViewImpl::getPlatformWindow()
{
    return m_pWebWindow->m_pVTable->getPlatformWindow(m_pWebWindow);
}

bool WindowScrollViewImpl::inWindow() const
{
    return true;
}

bool WindowScrollViewImpl::handleMousePressEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (m_hostDrawsScrollbars)
        return m_pWebFrameOwner->handleMousePressEvent(mouseEvent);
    else
        return FrameScrollViewImpl::handleMousePressEvent(mouseEvent);
}

bool WindowScrollViewImpl::handleMouseMoveEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (m_hostDrawsScrollbars)
        return m_pWebFrameOwner->handleMouseMoveEvent(mouseEvent);
    else
        return FrameScrollViewImpl::handleMouseMoveEvent(mouseEvent);
}

bool WindowScrollViewImpl::handleWheelEvent(WebCore::PlatformWheelEvent& wheelEvent)
{
    if (m_hostDrawsScrollbars)
        return m_pWebFrameOwner->handleWheelEvent(wheelEvent);
    else
        return FrameScrollViewImpl::handleWheelEvent(wheelEvent);
}

bool WindowScrollViewImpl::handleMouseReleaseEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (m_hostDrawsScrollbars)
        return m_pWebFrameOwner->handleMouseReleaseEvent(mouseEvent);
    else
        return FrameScrollViewImpl::handleMouseReleaseEvent(mouseEvent);
}

WebCore::PlatformScrollbar* WindowScrollViewImpl::scrollbarUnderMouse(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (m_hostDrawsScrollbars)
        return 0;
    else
        return FrameScrollViewImpl::scrollbarUnderMouse(mouseEvent);
}

WebCore::PlatformScrollbar* WindowScrollViewImpl::getVScrollbar()
{
    if (m_hostDrawsScrollbars)
        return 0;
    else
        return FrameScrollViewImpl::getVScrollbar();
}

WebCore::PlatformScrollbar* WindowScrollViewImpl::getHScrollbar()
{
    if (m_hostDrawsScrollbars)
        return 0;
    else
        return FrameScrollViewImpl::getHScrollbar();
}

WebCore::IScrollViewApolloImpl* WindowScrollViewImpl::getParent()
{
    return NULL;
}

WebCore::IntRect WindowScrollViewImpl::clientRect() const
{
    int const visibleWidth = m_pWebWindow->m_pVTable->visibleWidth( m_pWebWindow );
    int const visibleHeight = m_pWebWindow->m_pVTable->visibleHeight( m_pWebWindow );
    return WebCore::IntRect( 0, 0, visibleWidth, visibleHeight );
}

bool WindowScrollViewImpl::isInVisibleViewHierarchy() const
{
    return true;
}

}
