/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "WebWidgetImpl.h"
#include "WebFocusManager.h"
#include "IScrollViewApolloImpl.h"

namespace WebKitApollo
{
WebWidgetImpl::WebWidgetImpl( IFocusManager* const pFocusManager
                            , IMouseCaptureManager* const pMouseCaptureManager )
    : m_pClient(0)
    , m_pFocusManager(pFocusManager)
    , m_pMouseCaptureManager(pMouseCaptureManager)
    , m_parent(0)
{
    ASSERT( m_pFocusManager != NULL );
    ASSERT( m_pMouseCaptureManager != NULL );
}

WebWidgetImpl::~WebWidgetImpl()
{
    if (hasFocus())
        clearFocus();
    if (hasMouseCapture())
        releaseMouseCapture();
    ASSERT(!getParent());
}

void WebWidgetImpl::setClient(WebCore::WidgetClient* pClient)
{
    m_pClient = pClient;
}

WebCore::WidgetClient* WebWidgetImpl::client() const
{
    return m_pClient;
}

bool WebWidgetImpl::hasFocus() const
{
    return m_pFocusManager->hasFocus(this);
}

void WebWidgetImpl::setFocus()
{
    m_pFocusManager->setFocus(this);
}

void WebWidgetImpl::clearFocus()
{
    m_pFocusManager->clearFocus();
}

bool WebWidgetImpl::hasMouseCapture() const
{
    return m_pMouseCaptureManager->hasMouseCapture(this);
}

void WebWidgetImpl::setMouseCapture()
{
    m_pMouseCaptureManager->setMouseCapture(this);
}

void WebWidgetImpl::releaseMouseCapture()
{
    m_pMouseCaptureManager->releaseMouseCapture();
}

bool WebWidgetImpl::isEnabled() const
{
    return false;
}

void WebWidgetImpl::setEnabled( bool bEnabled )
{
}

void WebWidgetImpl::setIsSelected( bool bSelected )
{
}

WebCore::IntRect WebWidgetImpl::frameGeometry() const
{
    return m_FrameGeometry;
}

void WebWidgetImpl::setFrameGeometry(const WebCore::IntRect &rect)
{
    m_FrameGeometry = rect;
}

bool WebWidgetImpl::handleCut()
{
    return false;
}

bool WebWidgetImpl::handleCopy()
{
    return false;
}

bool WebWidgetImpl::handlePaste()
{
    return false;
}

bool WebWidgetImpl::handleSelectAll()
{
    return false;
}

WebCore::Cursor WebWidgetImpl::cursor()
{
    return m_cursor;
}

void WebWidgetImpl::setCursor(const WebCore::Cursor& cursor)
{
    m_cursor = cursor;
}

void WebWidgetImpl::invalidateRect(const WebCore::IntRect& damageViewRect)
{
    ASSERT(getParent()); //If an impl does not have parent it should have overriden this virtual method.
    WebCore::IntPoint const parentFrameDamageRectOrigin(damageViewRect.x() + frameGeometry().x()
                                                       , damageViewRect.y() + frameGeometry().y());
    WebCore::IntRect const damageRectInParent(parentFrameDamageRectOrigin, damageViewRect.size());
    getParent()->updateContents(damageRectInParent);
}

void WebWidgetImpl::hide()
{
}

void WebWidgetImpl::show()
{
}

WebCore::IScrollViewApolloImpl* WebWidgetImpl::getParent()
{
    return m_parent;
}

void WebWidgetImpl::setParent(WebCore::IScrollViewApolloImpl* parent)
{
    ASSERT(((parent == 0) && (m_parent != 0)) || ((parent != 0) && (m_parent == 0)) || (parent == m_parent));
    m_parent = parent;
}

void WebWidgetImpl::onFocus()
{
}

void WebWidgetImpl::onBlur()
{
}

bool WebWidgetImpl::isInVisibleViewHierarchy() const
{
    bool const result = m_parent && (m_parent->isInVisibleViewHierarchy());
    return result;
}

WebWindow* WebWidgetImpl::getWindow()
{
    return m_parent ? m_parent->getWindow() : 0; // there may not be a parent during destruction
}

}
