/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef WebViewImpl_h
#define WebViewImpl_h

#include <WebKitApollo/WebKit.h>
#include <WebKitApollo/helpers/WebViewHelper.h>
#include <memory>

#include <wtf/RefPtr.h>
#include <wtf/HashSet.h>
#include "WebFocusManager.h"
#include <Timer.h>
#include <FloatRect.h>
#include <ScrollTypes.h>

namespace WebCore
{
    class Editor;
    class KeyboardEvent;
    class Page;
	class Frame;
    class IWidgetApolloImpl;
    class IScrollViewApolloImpl;
    class FrameApollo;
    class FrameView;
    class PlatformKeyboardEvent;
}

namespace WebKitApollo
{

    class FrameScrollViewImpl;
    class WebPluginWidgetImpl;
    class WebWidgetImpl;
    class WebEditorClient;

	class WebDebugListenerImpl;

    WEBKIT_APOLLO_PROTO1 WebView* WEBKIT_APOLLO_PROTO1 webViewCreate(WebHost* pWebHost, WebWindow* pTargetWindow, WebError* pErrorInfo);

    class WebFrameImpl;

    class WebViewImpl : public WebViewHelper< WebViewImpl >,
                        private IFocusManager,
                        private IMouseCaptureManager
    {
        friend class WebViewHelper< WebViewImpl >;
    public:
        WebViewImpl(WebHost* pHost, WebWindow* pTargetWindow);
        virtual ~WebViewImpl();
        
        WebCore::Page* page();
        WebCore::Frame* mainFrame();
        
        void putFrameOnLifeSupport(WTF::PassRefPtr<WebCore::Frame>);
        
        void addPluginImpl(WebPluginWidgetImpl* const);
        void removePluginImpl(WebPluginWidgetImpl* const);

		WebHost *webHost() { return m_pHost; }
        
        inline IFocusManager* focusManager() { return this; }
        inline IMouseCaptureManager* mouseCaptureManager() { return this; }

        void selectionChanged();
        void activateIMEForPlugin();
        void deactivateIMEForPlugin();

        WTF::PassRefPtr<FrameScrollViewImpl> createScrollViewImplForRootFrame(WebFrameImpl* const webFrame, WebCore::FrameView* const owningFrameView);
        WTF::PassRefPtr<FrameScrollViewImpl> createScrollViewImplForSubFrame(WebFrameImpl* const webFrame, WebCore::FrameView* const owningFrameView);
        
        bool doDefaultKeyPressActionForEditor(WebCore::Editor* const editor, WebCore::KeyboardEvent* keyEvent);
        

    private:
        void destroy();
        WebFrame* mainWebFrame();
        void layout();
        void setPageGroupName( const uint16_t* groupName );
        WebString* getPageGroupName();
        void paint(const WebIntRect* const pPaintRect);
        WebCore::Frame* focusedFrame();
        bool scroll(WebCore::ScrollDirection, WebCore::ScrollGranularity);
               
        void copySelectionToClipboard();

        void onContextMenuItemSelect( int const actionID, const uint16_t *title );
        bool onContextMenu( int const localX
                        , int const localY
                        , int const screenX
                        , int const screenY
                        , bool const shiftKey
                        , bool const ctrlKey
                        , bool const altKey
                        , bool const metaKey);

        bool onMouseDown( int const localX
                        , int const localY
                        , int const screenX
                        , int const screenY
                        , WebMouseButton const mouseButton
                        , unsigned int const clickCount
                        , bool const shiftKey
                        , bool const ctrlKey
                        , bool const altKey
                        , bool const metaKey);
        bool onMouseUp( int const localX
                      , int const localY
                      , int const screenX
                      , int const screenY
                      , WebMouseButton const mouseButton
                      , unsigned int const clickCount
                      , bool const shiftKey
                      , bool const ctrlKey
                      , bool const altKey
                      , bool const metaKey);
        bool onMouseMove( int const localX
                        , int const localY
                        , int const screenX
                        , int const screenY
                        , bool const shiftKey
                        , bool const ctrlKey
                        , bool const altKey
                        , bool const metaKey
                        , bool const buttonDown);
        bool onMouseWheel( int const localX
                         , int const localY
                         , int const screenX
                         , int const screenY
                         , int const deltaX
                         , int const deltaY
                         , bool const shiftKey
                         , bool const ctrlKey
                         , bool const altKey
                         , bool const metaKey);
        void onFocusIn(const uint16_t *direction, int length);
        void onFocusOut();

        void onWindowActivate();
        void onWindowDeactivate();
        
        bool hasFocusableContent();

        bool onKeyDown( unsigned int const winCharCode
                      , unsigned int const winKeyCode
                      , bool const isShiftKey
                      , bool const isCtrlKey
                      , bool const isAltKey
                      , bool const isMetaKey
                      , bool const isAutoRepeat);
        bool onKeyUp( unsigned int const winCharCode
					, unsigned int const winKeyCode
					, bool const isShiftKey
					, bool const isCtrlKey
					, bool const isAltKey
                    , bool const isMetaKey);
        bool onKeyPress( unsigned int const winCharCode
					, bool const isShiftKey
					, bool const isCtrlKey
					, bool const isAltKey
                    , bool const isMetaKey
                    , bool const isAutoRepeat);

        bool insertText(const uint16_t* utf16Text, unsigned long numUTF16CodeUnits);
        void setComposition(const uint16_t* utf16Text
            , unsigned long numUTF16CodeUnits
            , WebViewCompositionUnderlineRange* underlineRanges
            , unsigned long numUnderlineRanges
            , int startSelection
            , int endSelection);
        void confirmComposition(const uint16_t* utf16Text, unsigned long numUTF16CodeUnits);
        void confirmComposition(bool preserveSelection);
        bool getSelectedRange(int& startSelection, int& endSelection, bool& isEditable);
        void setSelectedRange(int startSelection, int endSelection);
        bool getCompositionRange(int& startComposition, int& endComposition);
        WebIntRect getFirstRectForCharacterRange(int startRange, int endRange);
        bool getTextForCharacterRange(int startRange, int endRange, uint16_t* text);
        bool onDragEnter( int localX
                        , int localY
                        , int screenX
                        , int screenY
                        , bool shiftKey
                        , bool ctrlKey
                        , bool altKey
                        , bool metaKey
                        , WebScriptProxyVariant* dataTransfer);
        bool onDragOver( int localX
                        , int localY
                        , int screenX
                        , int screenY
                        , bool shiftKey
                        , bool ctrlKey
                        , bool altKey
                        , bool metaKey
                        , WebScriptProxyVariant* dataTransfer);
        bool onDragDrop( int localX
                        , int localY
                        , int screenX
                        , int screenY
                        , bool shiftKey
                        , bool ctrlKey
                        , bool altKey
                        , bool metaKey
                        , WebScriptProxyVariant* dataTransfer);
        void onDragExit( int localX
                        , int localY
                        , int screenX
                        , int screenY
                        , bool shiftKey
                        , bool ctrlKey
                        , bool altKey
                        , bool metaKey
                        , WebScriptProxyVariant* dataTransfer);
        void onDragStart( int localX
                        , int localY
                        , int screenX
                        , int screenY
                        , bool shiftKey
                        , bool ctrlKey
                        , bool altKey
                        , bool metaKey);
        void onDragUpdate( int localX
                         , int localY
                         , int screenX
                         , int screenY
                         , bool shiftKey
                         , bool ctrlKey
                         , bool altKey
                         , bool metaKey);
        void onDragComplete( int localX
                           , int localY
                           , int screenX
                           , int screenY
                           , bool shiftKey
                           , bool ctrlKey
                           , bool altKey
                           , bool metaKey
                           , const uint16_t *operation);
        bool onCut();
        bool onCopy();
        bool onPaste();
        bool onSelectAll();
        void sendResizeEvent();
        void sendScrollEvent();
        void convertToRootContent();
        void setShouldPaintDefaultBackground(bool newValue);
        void updatePlugins(WebCore::Timer<WebViewImpl>*);
        void setTextEncodingOverride(const uint16_t* utf16TextEncoding, unsigned long numUTF16CodeUnits);
        void getTextEncodingOverride(uint16_t** utf16TextEncoding);
        void setTextEncodingFallback(const uint16_t* utf16TextEncoding, unsigned long numUTF16CodeUnits);
        void getTextEncodingFallback(uint16_t** utf16TextEncodings);

        void getContextMenuItems(WebMenuItem*** items, int *nItems);
        void getContextMenuItemInfo(WebMenuItem* item, short **title, unsigned int *titleCharLen, int *type, unsigned char *checked, unsigned char *enabled, int *action, WebMenuItem ***subMenuItems, int *numSubMenuItems);

        //Focus manager methods
        virtual void setFocus(WebCore::IWidgetApolloImpl* const pFocusWidget);
        virtual bool hasFocus(const WebCore::IWidgetApolloImpl* const pWidget) const;
        virtual void clearFocus();

        //capture manager methods.
        virtual void setMouseCapture(WebCore::IWidgetApolloImpl* const pFocusWidget);
        virtual bool hasMouseCapture(const WebCore::IWidgetApolloImpl* const pWidget) const;
        virtual void releaseMouseCapture();


        void ensureFocusAndCaptureWidgets();

        void initSettings();

        void onLifeSupportTimer(WebCore::Timer<WebViewImpl>*);
        void onRemoveFocusTimer(WebCore::Timer<WebViewImpl>*);

        WebEditorClient* webEditorClient() const;

        WebCore::FrameView* focusedOrMainFrameView() const;
        
        bool doDefaultKeyPressAction(const WebCore::PlatformKeyboardEvent&);

        WebViewImpl(const WebViewImpl& other);
        WebViewImpl& operator=(const WebViewImpl& other);
    
        bool m_inDestructor;
        WebHost* const m_pHost;
        WebWindow* m_pTargetWindow;
        WebCore::IWidgetApolloImpl* m_pFocusWidget;
        WebCore::IWidgetApolloImpl* m_pMouseCaptureWidget;
        std::auto_ptr<WebCore::Page> m_pPage;
        WTF::HashSet<WTF::RefPtr<WebCore::Frame> > m_framesOnLifeSupport;
        WebCore::Timer<WebViewImpl> m_lifeSupportTimer;
        WebCore::Timer<WebViewImpl> m_removeFocusTimer;
        
        bool m_canTransformToRootContent;
        
        //Set of all plugin Impl's in this view.
        WTF::HashSet<WebPluginWidgetImpl*> m_pluginImpls;

        bool m_hasHostFocus;
        bool m_isIMEForPluginActive;

		WTF::RefPtr<WebDebugListenerImpl> m_debugListener;

        WebCore::Timer<WebViewImpl> m_updatePluginsTimer;
    };
}


#endif
