/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "WebViewImpl.h"

#include "WebFrameImpl.h"
#include "WebFrameScrollView.h"

#include "PlatformMouseEvent.h"
#include "PlatformKeyboardEvent.h"
#include "PlatformWheelEvent.h"
#include "PlatformScrollBar.h"

#include <Page.h>
#include <FocusController.h>
#include <FrameLoader.h>

#include "ContextMenuItemApollo.h"
#include "DragController.h"
#include "DragData.h"
#include "Editor.h"
#include "EventHandler.h"
#include "Frame.h"
#include "FrameTree.h"
#include "FrameView.h"
#include "IScrollViewApolloImpl.h"
#include "TextIterator.h"
#include "WebChromeClient.h"
#include "WebContextMenuClient.h"
#include "WebDragClient.h"
#include "WebEditorClient.h"
#include "WebPluginImpl.h"
#include "WebWindowScrollView.h"
#include "WebFrameScrollView.h"
#include "kjs_binding.h"
#include "kjs_proxy.h"
#include <ClipboardApollo.h>
#include <ClipboardApolloHelper.h>
#include <Settings.h>
#include <apollo/proxy_obj.h>
#include "ContextMenu.h"
#include "WebInspectorClient.h"
#include <WebKitApollo/WebKeyboardCodes.h>
#include <wtf/InstanceCounter.h>
#include <DocumentLoader.h>
#include <kjs/InitializeThreading.h>
#include "WebDebugListenerImpl.h"
#include "KeyboardEvent.h"

#if PLATFORM(DARWIN) || PLATFORM(UNIX_OS)
#include <sys/time.h>
#elif PLATFORM(WIN_OS)
#include <windows.h>
#endif

namespace WebKitApollo
{
extern WebKitAPIHostFunctions* g_HostFunctions;

WebViewImpl::WebViewImpl(WebHost* pHost, WebWindow* pTargetWindow)
    : m_inDestructor(false)
    , m_pHost(pHost)
    , m_pTargetWindow(pTargetWindow)
    , m_pFocusWidget(0)
    , m_pMouseCaptureWidget(0)
    , m_pPage(new WebCore::Page(new WebChromeClient(pTargetWindow, pHost), new WebContextMenuClient(), new WebEditorClient(this), new WebDragClient(), new WebInspectorClient()))
    , m_lifeSupportTimer(this, &WebViewImpl::onLifeSupportTimer)
    , m_removeFocusTimer(this, &WebViewImpl::onRemoveFocusTimer)
    , m_canTransformToRootContent(true)
    , m_hasHostFocus(false)
    , m_isIMEForPluginActive(false)
    , m_updatePluginsTimer(this, &WebViewImpl::updatePlugins)
{
    initSettings();
    WebFrameImpl::construct(this, m_pHost, m_pPage.get());
    m_pPage->mainFrame()->tree()->setName(WebCore::String());
    m_pPage->mainFrame()->init();
    
    KJS::initializeThreading();
   
	// HTMLDebugListenerImpl::getInstance() returns 0 if the debugger is disabled
	m_debugListener = WebDebugListenerImpl::getInstance();
	if (m_debugListener)
		m_pPage->setDebugger( m_debugListener->getDebugger() );

    WebCore::FrameLoader::setRestrictAccessToLocal(true);
}

WebViewImpl::~WebViewImpl()
{
	m_pPage->setDebugger(0);
	m_debugListener = 0;
    m_lifeSupportTimer.stop();
    m_removeFocusTimer.stop();
    WebCore::Frame* const pFrame = mainFrame();
    m_inDestructor = true;
    ASSERT(pFrame);
    WebCore::FrameLoader* const pLoader = pFrame->loader();
    ASSERT(pLoader);
    pLoader->detachFromParent();
    m_pPage = std::auto_ptr<WebCore::Page>();
}

WebCore::Page* WebViewImpl::page()
{
    ASSERT(m_pPage.get());
    return m_pPage.get();
}

WebCore::Frame* WebViewImpl::mainFrame()
{
    ASSERT(m_pPage.get());
    ASSERT(!m_inDestructor);
    WebCore::Frame* const pMainFrame = m_pPage->mainFrame();
    ASSERT(pMainFrame);
    return pMainFrame;
}

void WebViewImpl::putFrameOnLifeSupport(WTF::PassRefPtr<WebCore::Frame> frameArg)
{
    ASSERT(frameArg);
    WTF::RefPtr<WebCore::Frame> const frame(frameArg);
    m_framesOnLifeSupport.add(frame);
    if ((!m_inDestructor) && (!m_lifeSupportTimer.isActive()))
        m_lifeSupportTimer.startOneShot(0);
}

void WebViewImpl::onLifeSupportTimer(WebCore::Timer<WebViewImpl>* timer)
{
    ASSERT(m_framesOnLifeSupport.size());
    ASSERT(timer == &m_lifeSupportTimer);
    m_framesOnLifeSupport.clear();
}


void WebViewImpl::onRemoveFocusTimer(WebCore::Timer<WebViewImpl>* timer)
{
    ASSERT(timer == &m_removeFocusTimer);
    
    WebCore::FocusController* controller = m_pPage->focusController();

    if (WebCore::Frame* frame = controller->focusedOrMainFrame())
        frame->selectionController()->setFocused(false);

    if(m_pFocusWidget)
        m_pFocusWidget->onBlur();
    
    controller->setActive(false);
}

WebFrame* WebViewImpl::mainWebFrame()
{
    m_canTransformToRootContent = false;
    WebCore::Frame* const pMainFrame = mainFrame();
    WebFrame* result = NULL;
    if (pMainFrame)
    {
        WebFrameImpl* const pMainWebFrameImpl = WebFrameImpl::kit(pMainFrame);
        ASSERT(pMainWebFrameImpl);
        result = pMainWebFrameImpl->getWebFrame();
        ASSERT(result);
    }
    return result;
}

WebCore::Frame* WebViewImpl::focusedFrame()
{
    WebCore::FocusController* controller = m_pPage->focusController();
    return controller->focusedFrame();
}

void WebViewImpl::setFocus(WebCore::IWidgetApolloImpl* const pFocusWidget)
{
    m_canTransformToRootContent = false;
    ASSERT(pFocusWidget);
    if (pFocusWidget != m_pFocusWidget) {
        if (m_pFocusWidget) {
            m_pFocusWidget->onBlur();
        }
        m_pFocusWidget = pFocusWidget;
        ASSERT(m_pFocusWidget);
        m_pFocusWidget->onFocus();
    }
}

bool WebViewImpl::hasFocus(const WebCore::IWidgetApolloImpl* const pWidget) const
{
    ASSERT(pWidget);
    return m_pFocusWidget == pWidget;
}

void WebViewImpl::clearFocus()
{
    m_canTransformToRootContent = false;
    if (!m_inDestructor) {
        if (m_pFocusWidget) {
            m_pFocusWidget->onBlur();
        }
        
        m_pFocusWidget = 0;
    }
}

void WebViewImpl::setMouseCapture(WebCore::IWidgetApolloImpl* const pMouseCaptureWidget)
{
    m_canTransformToRootContent = false;
    ASSERT(pMouseCaptureWidget);
    m_pMouseCaptureWidget = pMouseCaptureWidget;
}

bool WebViewImpl::hasMouseCapture(const WebCore::IWidgetApolloImpl* const pWidget) const
{
    ASSERT(pWidget);
    return m_pMouseCaptureWidget == pWidget;
}

void cancelMousePressedRecursive( WebCore::Frame* frame )
{
    ASSERT(frame);
    ASSERT(frame->eventHandler());
    frame->eventHandler()->setMousePressed( false );
    WebCore::FrameTree* tree = frame->tree();
    ASSERT(tree);
    unsigned childCount = tree->childCount();
    unsigned index = 0;
    for( ;index<childCount; index++) {
        WebCore::Frame* child = tree->child(index);
        ASSERT(child);
        cancelMousePressedRecursive(child);
    }
    
}

void WebViewImpl::releaseMouseCapture()
{
    m_canTransformToRootContent = false;
    if (!m_inDestructor) {
        WebCore::Frame* const pMainFrame = mainFrame();
        if (pMainFrame) {
            cancelMousePressedRecursive(pMainFrame);
            WebCore::FrameView* const pMainFrameView = pMainFrame->view();
            if (pMainFrameView)
                m_pMouseCaptureWidget = pMainFrameView->getApolloImpl().get();
            else
                m_pMouseCaptureWidget = 0;
        }
        else
            m_pMouseCaptureWidget = 0;
    }
}

namespace {
    // handles widgetImpl being null.
    static inline bool widgetIsSameOrDescendantOf(WebCore::FrameView* const fv, WebCore::IWidgetApolloImpl* const widgetImpl)
    {
        ASSERT(fv);
        // Cast the FrameView up to Widget so that the getApolloImpl
        // will return us the IWidgetApolloImpl instead of the IScrollViewApolloImpl.
        WebCore::Widget* const fvAsWidget = fv;
        WTF::PassRefPtr<WebCore::IWidgetApolloImpl> const fvWidgetImpl(fvAsWidget->getApolloImpl());
        
        WebCore::IWidgetApolloImpl* currWidgetImpl = widgetImpl;
        while ((currWidgetImpl) && (fvWidgetImpl != currWidgetImpl))
            currWidgetImpl = currWidgetImpl->getParent();
            
        return fvWidgetImpl == currWidgetImpl;
    
    }
}

void WebViewImpl::ensureFocusAndCaptureWidgets()
{
    WebCore::Frame* const pMainFrame = mainFrame();
    if (pMainFrame) {
        m_canTransformToRootContent = false;
        
        // the frame view of the focused frame must contain the widget that
        // we are going to treat as the focused widget.  If the focused frame
        // does not contain the widget, then we'll set the focused widget
        // to be the view of the focused frame.
        WebCore::FrameView* const viewThatMustContainFocus = focusedOrMainFrameView();
        ASSERT(viewThatMustContainFocus);
        if (!widgetIsSameOrDescendantOf(viewThatMustContainFocus, m_pFocusWidget)) {
            WebCore::Widget* const widgetThatMustContainFocus = viewThatMustContainFocus;
            WTF::PassRefPtr<WebCore::IWidgetApolloImpl> const fvWidgetImpl(widgetThatMustContainFocus->getApolloImpl());
            ASSERT(m_pFocusWidget != fvWidgetImpl);
            if (m_pFocusWidget)
                m_pFocusWidget->onBlur();
            m_pFocusWidget = fvWidgetImpl.get();
            m_pFocusWidget->onFocus();
        }
        if (!m_pMouseCaptureWidget) {
            WebCore::FrameView* const pMainFrameView = pMainFrame->view();
            m_pMouseCaptureWidget = pMainFrameView->getApolloImpl().get();
        }
        ASSERT(m_pFocusWidget);
        ASSERT(m_pMouseCaptureWidget);
    }
}

void WebViewImpl::destroy()
{
    m_canTransformToRootContent = false;
    delete this;
}

void WebViewImpl::layout()
{
    m_canTransformToRootContent = false;
    WebCore::Frame* const theMainFrame = mainFrame();
    if (theMainFrame) {
        WebCore::FrameView* const mainFrameView = theMainFrame->view();
        if (mainFrameView) {
            WebFrameImpl* const mainWebFrame = WebFrameImpl::kit(theMainFrame);
            mainWebFrame->layoutRecursive();
            // ASSERT(!mainFrameView->needsLayout());
        }
    }
}

static double currentTime()
{
#if PLATFORM(DARWIN) || PLATFORM(UNIX_OS)
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec + 0.000001 * tv.tv_usec;
#elif PLATFORM(WIN_OS)
    return 0.001 * GetTickCount();
#else
#error "No implementation of currentTime()."
#endif
}

bool WebViewImpl::onMouseDown( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , WebMouseButton const mouseButton
                             , unsigned int const clickCount
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey)
{
    ensureFocusAndCaptureWidgets();
    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);
    ASSERT(static_cast<unsigned long>(WebCore::LeftButton) == static_cast<unsigned long>(WebMouseButtonLeft));
    ASSERT(static_cast<unsigned long>(WebCore::MiddleButton) == static_cast<unsigned long>(WebMouseButtonMiddle));
    ASSERT(static_cast<unsigned long>(WebCore::RightButton) == static_cast<unsigned long>(WebMouseButtonRight));
    WebCore::MouseButton webCoreMouseButton = static_cast<WebCore::MouseButton>(mouseButton);
    WebCore::PlatformMouseEvent mouseEvent( localPoint
                                          , screenPoint
                                          , webCoreMouseButton
                                          , WebCore::MouseEventPressed
                                          , static_cast< int >( clickCount )
                                          , shiftKey
                                          , ctrlKey
                                          , altKey
                                          , metaKey
                                          , currentTime());

    ASSERT(m_pMouseCaptureWidget);
    return m_pMouseCaptureWidget->handleMousePressEvent(mouseEvent);
}

void WebViewImpl::onContextMenuItemSelect( int const actionID, const uint16_t *title )
{
    WebCore::ContextMenuItem item(WebCore::ActionType, static_cast<WebCore::ContextMenuAction>(actionID), WebCore::String((const UChar *) title));

    m_pPage->contextMenuController()->contextMenuItemSelected(&item);
}

bool WebViewImpl::onContextMenu( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey)
{
    ensureFocusAndCaptureWidgets();
    WebCore::IntPoint localPoint(localX, localY);
    WebCore::IntPoint screenPoint(screenX, screenY);

    // we need to convert localPoint correctly as the WebKit windows windowToContents call does this, however we do not do this in Apollo
    WebCore::FocusController* controller = m_pPage->focusController();
    WebCore::Frame* frame = controller->focusedOrMainFrame();
    WTF::PassRefPtr<WebCore::IScrollViewApolloImpl> impl = frame->view()->getApolloImpl( );
 
    for (WebCore::IScrollViewApolloImpl *parent = impl->getParent(), *child = impl.get();
         parent;
         child = parent, parent = parent->getParent())
    {
        WebCore::IntPoint const canvasLocation(parent->viewportToContents(localPoint));
        WebCore::IntPoint const subFrameLocation(child->frameGeometry().location());
        localPoint = WebCore::IntPoint(0, 0) + (canvasLocation - subFrameLocation);
    }
    
    WebCore::PlatformMouseEvent mouseEvent( localPoint
                                          , screenPoint
                                          , WebCore::RightButton
                                          , WebCore::MouseEventReleased
                                          , 0
                                          , shiftKey
                                          , ctrlKey
                                          , altKey
                                          , metaKey
                                          , static_cast<double>(time(0)));


    m_pPage->contextMenuController()->clearContextMenu();

    return frame->eventHandler()->sendContextMenuEvent(mouseEvent);
}

bool WebViewImpl::onMouseUp( int const localX
                           , int const localY
                           , int const screenX
                           , int const screenY
                           , WebMouseButton const mouseButton
                           , unsigned int const clickCount
                           , bool const shiftKey
                           , bool const ctrlKey
                           , bool const altKey
                           , bool const metaKey)
{
    ensureFocusAndCaptureWidgets();
    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);
    ASSERT(static_cast<unsigned long>(WebCore::LeftButton) == static_cast<unsigned long>(WebMouseButtonLeft));
    ASSERT(static_cast<unsigned long>(WebCore::MiddleButton) == static_cast<unsigned long>(WebMouseButtonMiddle));
    ASSERT(static_cast<unsigned long>(WebCore::RightButton) == static_cast<unsigned long>(WebMouseButtonRight));
    WebCore::MouseButton webCoreMouseButton = static_cast<WebCore::MouseButton>(mouseButton);
    WebCore::PlatformMouseEvent mouseEvent( localPoint
                                          , screenPoint
                                          , webCoreMouseButton
                                          , WebCore::MouseEventReleased
                                          , static_cast< int >( clickCount )
                                          , shiftKey
                                          , ctrlKey
                                          , altKey
                                          , metaKey
                                          , currentTime());

    ASSERT(m_pMouseCaptureWidget);
    return m_pMouseCaptureWidget->handleMouseReleaseEvent(mouseEvent);
}

bool WebViewImpl::onMouseMove( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey
                             , bool const buttonDown)
{
    ensureFocusAndCaptureWidgets();
    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);
    WebCore::MouseButton webCoreMouseButton = static_cast<WebCore::MouseButton>(buttonDown ? WebCore::LeftButton : WebCore::NoButton);
    WebCore::PlatformMouseEvent mouseEvent( localPoint
                                          , screenPoint
                                          , webCoreMouseButton
                                          , WebCore::MouseEventMoved
                                          , 0
                                          , shiftKey
                                          , ctrlKey
                                          , altKey
                                          , metaKey
                                          , currentTime());
    ASSERT(m_pMouseCaptureWidget);
    return m_pMouseCaptureWidget->handleMouseMoveEvent( mouseEvent );
}

bool WebViewImpl::onMouseWheel( int const localX
                              , int const localY
                              , int const screenX
                              , int const screenY
                              , int const deltaX
                              , int const deltaY
                              , bool const shiftKey
                              , bool const ctrlKey
                              , bool const altKey
                              , bool const metaKey)
{
    ensureFocusAndCaptureWidgets();
    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);
    bool isAccepted = false;
    bool isContinuous = false;
    WebCore::PlatformWheelEvent wheelEvent( localPoint
                                          , screenPoint
                                          , static_cast< float >( deltaX )
                                          , static_cast< float >( deltaY )
                                          , isAccepted
                                          , shiftKey
                                          , ctrlKey
                                          , altKey
                                          , metaKey
                                          , isContinuous );
    ASSERT(m_pMouseCaptureWidget);
    return m_pMouseCaptureWidget->handleWheelEvent( wheelEvent );
}

void WebViewImpl::onFocusIn(const uint16_t *direction, int length)
{
    m_canTransformToRootContent = false;
    m_removeFocusTimer.stop();

    ASSERT(m_pPage.get());
    ASSERT(m_pPage->focusController());

    m_hasHostFocus = true;
    if (m_isIMEForPluginActive)
        webHost()->m_pVTable->activateIMEForPlugin(webHost());

    WebCore::FocusController* controller = m_pPage->focusController();
    WebCore::Frame* frame = controller->focusedFrame();

    static bool didInitStrings = false;
    U_STRING_DECL(bottomUStr, "bottom", 6);
    U_STRING_DECL(topUStr, "top", 3);
    if(!didInitStrings)
    {
        U_STRING_INIT(bottomUStr, "bottom", 6);
        U_STRING_INIT(topUStr, "top", 3);
        didInitStrings = true;
    }
    
    if(u_strcmp(bottomUStr, (const UChar*)direction) == 0)
    {
        controller->setFocusedNode(0, frame);
        controller->setFocusedFrame(0);
        controller->advanceFocus(WebCore::FocusDirectionBackward, 0);
    }
   
    else if(u_strcmp(topUStr, (const UChar*)direction) == 0)
    {
        controller->setFocusedNode(0, frame);
        controller->setFocusedFrame(0);
        controller->advanceFocus(WebCore::FocusDirectionForward, 0);
    }
    else if (frame)
        frame->selectionController()->setFocused(true);

    // onWindowActivate should behave similarly
    if(m_pFocusWidget)
        m_pFocusWidget->onFocus();
    
    controller->setActive(true);

    webHost()->m_pVTable->selectionChanged(webHost());
}

bool WebViewImpl::hasFocusableContent()
{
    WebCore::FocusController* controller = m_pPage->focusController();
    
    WebCore::Node* forwardNode = controller->getNextFocusNode(WebCore::FocusDirectionForward, 0);
    WebCore::Node* backwardNode = controller->getNextFocusNode(WebCore::FocusDirectionBackward, 0);

    return forwardNode || backwardNode;
}

void WebViewImpl::onFocusOut()
{
    m_canTransformToRootContent = false;
    ASSERT(m_pPage.get());
    ASSERT(m_pPage->focusController());

    m_hasHostFocus = false;
    if (m_isIMEForPluginActive) {
        webHost()->m_pVTable->compositionAbandoned(webHost());
        webHost()->m_pVTable->deactivateIMEForPlugin(webHost());
    }

    if (m_pPage->focusController()->focusedOrMainFrame())
        m_removeFocusTimer.startOneShot(0);
}

void WebViewImpl::onWindowActivate()
{
    ASSERT(m_pPage.get());

    WebCore::FocusController *controller = m_pPage->focusController();
    ASSERT(controller);

    if (WebCore::Frame* frame = controller->focusedOrMainFrame())
        frame->selectionController()->setFocused(true);

    // this should behave similarly to onFocusIn
    if(m_pFocusWidget)
        m_pFocusWidget->onFocus();
    
    controller->setActive(true);
}

void WebViewImpl::onWindowDeactivate()
{
    onRemoveFocusTimer(&m_removeFocusTimer);
}

// this is to handle the following missing method:
//      ScrollView::scroll(ScrollDirection direction, ScrollGranularity granularity) 
bool WebViewImpl::scroll(WebCore::ScrollDirection direction, WebCore::ScrollGranularity granularity)
{
    // check to see if m_pFocusWidget is actually the focus'd frame
    WebCore::FrameView* const view = focusedOrMainFrameView();
    WebCore::Widget* widget = view;
    WTF::PassRefPtr<WebCore::IScrollViewApolloImpl> const focusedFrameViewImpl = view->getApolloImpl();
    WTF::PassRefPtr<WebCore::IWidgetApolloImpl> const focusedFrameWidgetImpl = widget->getApolloImpl();

    WebCore::IScrollViewApolloImpl* impl = (focusedFrameWidgetImpl == m_pFocusWidget) ? 
        focusedFrameViewImpl.get() :
        m_pFocusWidget->getParent();
        
    ASSERT(impl);
    
    WebCore::IScrollViewApolloImpl* rootScrollView = 0;
        
    // normally here we would call: mainFrame()->view()->scroll(direction, granularity);
    WebCore::Frame* const frame = view->frame();
    ASSERT(frame);
    // first try overflow which can cause DIV scrolling    
    if(frame->eventHandler()->scrollOverflow(direction, granularity))
        return true;
    
    while(impl)
    {
        rootScrollView = impl;
        WebCore::PlatformScrollbar* bar = (direction == WebCore::ScrollUp || direction == WebCore::ScrollDown) ? 
            impl->getVScrollbar() : 
            impl->getHScrollbar();

        // if we have platform scrollbars
        if(bar && bar->scroll(direction, granularity))
            return true;

       impl = impl->getParent();
    }

    // otherwise we have to calculate this on our own and send it to the HTMLControl
    // from Scrollbar::scroll, on other platforms we would be passing this call directly to the PlatformScrollbars
    ASSERT(rootScrollView);
    
    // Sometimes the rootScrollView will not be the view of the main frame on the page
    // this can happen when the focused frame is a subframe that is not currently being displayed.
    // In that, case we'll just eat the keyboard event telling us to scroll.
    // One day we may decide to actually scoll this non-visible frame, but I don't think it is worth the trouble right now.
    WTF::PassRefPtr<WebCore::IScrollViewApolloImpl> const mainFrameScrollViewImpl(m_pPage->mainFrame()->view()->getApolloImpl());
    if (mainFrameScrollViewImpl != rootScrollView)
        return true;
    
    
    ASSERT(rootScrollView == mainFrameScrollViewImpl);
    
    int stepX = 0;
    int stepY = 0;

    switch(direction)
    {
        case WebCore::ScrollUp:      stepY = -1; break;
        case WebCore::ScrollLeft:    stepX = -1; break;
        case WebCore::ScrollRight:   stepX = 1; break;
        case WebCore::ScrollDown:    stepY = 1; break;
        default: ASSERT(false);
    }
    
    switch(granularity)
    {
        case WebCore::ScrollByLine:
        {
            stepX *= LINE_STEP; 
            stepY *= LINE_STEP; 
            break;
        }
        case WebCore::ScrollByPage:
        {
            const int visibleWidth = rootScrollView->visibleWidth();
            const int visibleHeight = rootScrollView->visibleHeight();

            stepX *= (((visibleWidth - PAGE_KEEP) < 0) ? visibleWidth : (visibleWidth - PAGE_KEEP));
            stepY *= (((visibleHeight - PAGE_KEEP) < 0) ? visibleHeight : (visibleHeight - PAGE_KEEP));
            break;
        }
        case WebCore::ScrollByDocument:
        {
            const int contentsWidth = rootScrollView->contentsWidth();
            const int contentsHeight = rootScrollView->contentsHeight();

            stepX *= contentsWidth;
            stepY *= contentsHeight;
        }
        case WebCore::ScrollByPixel: break;
        default: ASSERT(false);
    }

    return rootScrollView->scrollBy( stepX, stepY );
}

bool WebViewImpl::onKeyDown( unsigned int const winCharCode
                           , unsigned int const winKeyCode
                           , bool const isShiftKey
                           , bool const isCtrlKey
                           , bool const isAltKey
                           , bool const isMetaKey
                           , bool const isAutoRepeat)
{
    ensureFocusAndCaptureWidgets();

    WebCore::PlatformKeyboardEvent keyEvent( WebCore::PlatformKeyboardEvent::RawKeyDown
                                           , winCharCode
                                           , winKeyCode
                                           , isShiftKey
                                           , isCtrlKey
                                           , isAltKey 
                                           , isMetaKey
                                           , isAutoRepeat);
    ASSERT(m_pFocusWidget);
    bool handled = m_pFocusWidget->handleKeyboardEvent( keyEvent );
    if(handled)
        return handled;

    int windowsKeyCode = keyEvent.windowsVirtualKeyCode();
    if ((windowsKeyCode == Web_VK_BACK && keyEvent.shiftKey()) || (windowsKeyCode == Web_VK_RIGHT && keyEvent.ctrlKey()))
        return m_pPage->goForward();
    else if (windowsKeyCode == Web_VK_BACK || (windowsKeyCode == Web_VK_LEFT && keyEvent.ctrlKey()))
        return m_pPage->goBack();
    
    // Need to scroll the page if the arrow keys, space(shift), pgup/dn, or home/end are hit.
    WebCore::ScrollDirection direction;
    WebCore::ScrollGranularity granularity;
    switch (windowsKeyCode) {
        case Web_VK_LEFT:
            granularity = WebCore::ScrollByLine;
            direction = WebCore::ScrollLeft;
            break;
        case Web_VK_RIGHT:
            granularity = WebCore::ScrollByLine;
            direction = WebCore::ScrollRight;
            break;
        case Web_VK_UP:
            granularity = WebCore::ScrollByLine;
            direction = WebCore::ScrollUp;
            break;
        case Web_VK_DOWN:
            granularity = WebCore::ScrollByLine;
            direction = WebCore::ScrollDown;
            break;
        case Web_VK_HOME:
            granularity = WebCore::ScrollByDocument;
            direction = WebCore::ScrollUp;
            break;
        case Web_VK_END:
            granularity = WebCore::ScrollByDocument;
            direction = WebCore::ScrollDown;
            break;
        case Web_VK_PRIOR:
            granularity = WebCore::ScrollByPage;
            direction = WebCore::ScrollUp;
            break;
        case Web_VK_NEXT:
            granularity = WebCore::ScrollByPage;
            direction = WebCore::ScrollDown;
            break;
        default:
            return false;
    }
    
    return scroll(direction, granularity);
}


bool WebViewImpl::onKeyUp( unsigned int const winCharCode
                         , unsigned int const winKeyCode
                         , bool const isShiftKey
                         , bool const isCtrlKey
                         , bool const isAltKey
                         , bool const isMetaKey)
{
    ensureFocusAndCaptureWidgets();
    WebCore::PlatformKeyboardEvent keyEvent(  WebCore::PlatformKeyboardEvent::KeyUp
                                           , winCharCode
                                           , winKeyCode
                                           , isShiftKey
                                           , isCtrlKey
                                           , isAltKey
                                           , isMetaKey 
                                           , false);
    ASSERT(m_pFocusWidget);
    return m_pFocusWidget->handleKeyboardEvent(keyEvent);
}

bool WebViewImpl::onKeyPress( unsigned int const winCharCode
                         , bool const isShiftKey
                         , bool const isCtrlKey
                         , bool const isAltKey
                         , bool const isMetaKey
                         , bool const isAutoRepeat)
{
    ensureFocusAndCaptureWidgets();
    WebCore::PlatformKeyboardEvent keyEvent(  WebCore::PlatformKeyboardEvent::Char
                                           , winCharCode
                                           , 0
                                           , isShiftKey
                                           , isCtrlKey
                                           , isAltKey
                                           , isMetaKey 
                                           , isAutoRepeat);
    ASSERT(m_pFocusWidget);
    bool handled =  m_pFocusWidget->handleKeyboardEvent(keyEvent);
    if (!handled)
        handled = doDefaultKeyPressAction(keyEvent);
    return handled;
}

inline bool WebViewImpl::doDefaultKeyPressAction(const WebCore::PlatformKeyboardEvent& keyEvent)
{
    bool handled = false;
    if (keyEvent.text() == " ") {
        WebCore::Frame *frame = m_pPage->focusController()->focusedOrMainFrame();
        WebCore::ScrollDirection direction = keyEvent.shiftKey() ? WebCore::ScrollUp : WebCore::ScrollDown;
        if (!frame->eventHandler()->scrollOverflow(direction, WebCore::ScrollByPage))
            scroll(direction, WebCore::ScrollByPage);

        handled = true;
    }
    return handled;
}

bool WebViewImpl::doDefaultKeyPressActionForEditor(WebCore::Editor* const editor, WebCore::KeyboardEvent* keyEvent)
{
    ASSERT(editor);
    ASSERT(keyEvent);
    bool handled = false;
    if (keyEvent->charCode() == ' ')
        handled = editor->insertText(WebCore::String(" "), 0);
    return handled;
}

bool WebViewImpl::insertText(const uint16_t* utf16Text, unsigned long numUTF16CodeUnits)
{
    ensureFocusAndCaptureWidgets();
    WebCore::String text(reinterpret_cast<const UChar*>(utf16Text), numUTF16CodeUnits);
    bool const handled = m_pFocusWidget->handleInsertText(text);
    return handled;
}

void WebViewImpl::setComposition(const uint16_t* utf16Text, unsigned long numUTF16CodeUnits,
    WebViewCompositionUnderlineRange *underlineRanges, unsigned long numUnderlineRanges,
    int startSelection, int endSelection)
{
    WebCore::Frame* frame = focusedFrame();

    if (!frame || !frame->selectionController()->isContentEditable())
        return;

    WebCore::String composition(reinterpret_cast<const UChar*>(utf16Text), numUTF16CodeUnits);
    Vector<WebCore::CompositionUnderline> underlines;
    for (unsigned long i=0; i<numUnderlineRanges; i++) {
        uint8_t r, g, b, a;
        r = underlineRanges[i].color.red;
        g = underlineRanges[i].color.green;
        b = underlineRanges[i].color.blue;
        a = underlineRanges[i].color.alpha;
        WebCore::Color c(r, g, b, a);
        WebCore::CompositionUnderline u(underlineRanges[i].start,
            underlineRanges[i].end, c, underlineRanges[i].thick);
        underlines.append(u);
    }

    frame->editor()->setComposition(composition, underlines, startSelection, endSelection);
}

void WebViewImpl::confirmComposition(const uint16_t* utf16Text, unsigned long numUTF16CodeUnits)
{
    WebCore::String composition(reinterpret_cast<const UChar*>(utf16Text), numUTF16CodeUnits);
    if (WebCore::Frame* frame = focusedFrame())
        frame->editor()->confirmComposition(composition);
}

void WebViewImpl::confirmComposition(bool preserveSelection)
{
    if (WebCore::Frame* frame = focusedFrame()) {
        if (preserveSelection)
            frame->editor()->confirmCompositionWithoutDisturbingSelection();
        else
            frame->editor()->confirmComposition();
    }
}

static bool getRangeExtents(WebCore::Frame* frame, WebCore::Range* range, int& startPosition, int& endPosition)
{
    int exception = 0;
    if (!range || !range->startContainer())
        return false;

    WebCore::Element* selectionRoot = frame->selectionController()->rootEditableElement();
    WebCore::Element* scope = selectionRoot ? selectionRoot : frame->document()->documentElement();

    if (range->startContainer(exception) != scope && !range->startContainer(exception)->isDescendantOf(scope))
        return false;
    if (range->endContainer(exception) != scope && !range->endContainer(exception)->isDescendantOf(scope))
        return false;

    RefPtr<WebCore::Range> testRange = WebCore::Range::create(scope->document(), scope, 0, range->startContainer(exception), range->startOffset(exception));
    ASSERT(testRange->startContainer(exception) == scope);
    startPosition = WebCore::TextIterator::rangeLength(testRange.get());

    testRange->setEnd(range->endContainer(exception), range->endOffset(exception), exception);
    ASSERT(testRange->startContainer(exception) == scope);
    endPosition = WebCore::TextIterator::rangeLength(testRange.get());

    return true;
}

PassRefPtr<WebCore::Range> rangeWithExtents(WebCore::Frame* frame, int startPosition, int endPosition)
{
    WebCore::Element* selectionRoot = frame->selectionController()->rootEditableElement();
    WebCore::Element* scope = selectionRoot ? selectionRoot : frame->document()->documentElement();
    return WebCore::TextIterator::rangeFromLocationAndLength(scope, startPosition, endPosition - startPosition);
}

bool WebViewImpl::getSelectedRange(int& startSelection, int& endSelection, bool& isEditable)
{
    if (WebCore::Frame* frame = focusedFrame()) {
        isEditable = frame->selectionController()->isContentEditable();
        return getRangeExtents(frame, frame->selectionController()->toRange().get(), startSelection, endSelection);
    }
    return false;
}

void WebViewImpl::setSelectedRange(int startSelection, int endSelection)
{
    if (WebCore::Frame* frame = focusedFrame()) {
        RefPtr<WebCore::Range> domRange = rangeWithExtents(frame, startSelection, endSelection);
        if (domRange)
            frame->selectionController()->setSelection(WebCore::Selection(domRange.get(), WebCore::SEL_DEFAULT_AFFINITY));
    }
}

bool WebViewImpl::getCompositionRange(int& startComposition, int& endComposition)
{
    if (WebCore::Frame* frame = focusedFrame())
        return getRangeExtents(frame, frame->editor()->compositionRange().get(), startComposition, endComposition);
    return false;
}

WebIntRect WebViewImpl::getFirstRectForCharacterRange(int startRange, int endRange)
{
    WebIntRect webFirstRect = {0, 0, 0, 0};
    if (WebCore::Frame* frame = focusedFrame()) {
        RefPtr<WebCore::Range> range(rangeWithExtents(frame, startRange, endRange));
        if (!range)
            return webFirstRect;
        WebCore::IntRect firstRect = frame->firstRectForRange(range.get());
        webFirstRect.m_left = firstRect.x();
        webFirstRect.m_top = firstRect.y();
        webFirstRect.m_right = firstRect.right();
        webFirstRect.m_bottom = firstRect.bottom();
    }
    return webFirstRect;
}

bool WebViewImpl::getTextForCharacterRange(int startRange, int endRange, uint16_t* utf16Text)
{
    ASSERT(startRange >= 0);
    ASSERT(startRange <= endRange);
    if (WebCore::Frame* frame = focusedFrame()) {
        RefPtr<WebCore::Range> range(rangeWithExtents(frame, startRange, endRange));
        WebCore::String text(range->text());
        ASSERT(text.length() <= unsigned(endRange - startRange + 1));
        ::memcpy(utf16Text, text.characters(), text.length() * sizeof(UChar));
        return true;
    }
    return false;
}

void WebViewImpl::selectionChanged()
{
    WebCore::Frame* targetFrame = m_pPage->focusController()->focusedOrMainFrame();
    if (!targetFrame)
        return;
  
    if (targetFrame->editor()->ignoreCompositionSelectionChange())
        return;
    
    // set data to primary gtk's primary clipboard on selection event 
    copySelectionToClipboard();

    if (targetFrame->editor()->hasComposition()) {
        unsigned int start, end;
        if (targetFrame->editor()->getCompositionSelection(start, end)) {
            webHost()->m_pVTable->compositionSelectionChanged(webHost(), start, end);
        } else {
            targetFrame->editor()->confirmCompositionWithoutDisturbingSelection();
            webHost()->m_pVTable->compositionAbandoned(webHost());
        }
    } else {
        webHost()->m_pVTable->selectionChanged(webHost());
    }
}

void WebViewImpl::activateIMEForPlugin()
{
    if (m_hasHostFocus && !m_isIMEForPluginActive)
        webHost()->m_pVTable->activateIMEForPlugin(webHost());
    m_isIMEForPluginActive = true;
}

void WebViewImpl::deactivateIMEForPlugin()
{
    if (m_hasHostFocus && m_isIMEForPluginActive)
        webHost()->m_pVTable->deactivateIMEForPlugin(webHost());
    m_isIMEForPluginActive = false;
}

WTF::PassRefPtr<FrameScrollViewImpl> WebViewImpl::createScrollViewImplForRootFrame(WebFrameImpl* const webFrame, WebCore::FrameView* const owningFrameView)
{
    return WindowScrollViewImpl::construct(m_pTargetWindow, webFrame, owningFrameView, this, this);
}

WTF::PassRefPtr<FrameScrollViewImpl> WebViewImpl::createScrollViewImplForSubFrame(WebFrameImpl* const webFrame, WebCore::FrameView* const owningFrameView)
{
    return SubFrameScrollViewImpl::construct(webFrame, owningFrameView, this, this);
}

bool WebViewImpl::onDragEnter( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey
                             , WebScriptProxyVariant* dataTransfer)
{
    WebCore::Frame* const pMainFrame = mainFrame();

    KJS::JSLock lock;

    KJS::JSValue* dataTransferValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(dataTransfer);
    KJS::ExecState* exec = pMainFrame->scriptProxy()->globalObject()->globalExec();
    KJS::JSObject* dataTransferObject = dataTransferValue->toObject(exec);

    WebCore::ClipboardApolloHelper dragClipboard(dataTransferObject, exec);

    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);

    WebCore::DragData dragData(&dragClipboard, localPoint, screenPoint, dragClipboard.dragOperationAllowed());
    WebCore::DragOperation operation = m_pPage.get()->dragController()->dragEntered(&dragData);
    dragClipboard.setDragOperation(operation);
    return operation != WebCore::DragOperationNone;
}

bool WebViewImpl::onDragOver( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey
                             , WebScriptProxyVariant* dataTransfer)
{
    WebCore::Frame* const pMainFrame = mainFrame();

    KJS::JSLock lock;

    KJS::JSValue* dataTransferValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(dataTransfer);
    KJS::ExecState* exec = pMainFrame->scriptProxy()->globalObject()->globalExec();
    KJS::JSObject* dataTransferObject = dataTransferValue->toObject(exec);

    WebCore::ClipboardApolloHelper dragClipboard(dataTransferObject, exec);

    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);

    WebCore::DragData dragData(&dragClipboard, localPoint, screenPoint, dragClipboard.dragOperationAllowed());
    WebCore::DragOperation operation = m_pPage.get()->dragController()->dragUpdated(&dragData);
    dragClipboard.setDragOperation(operation);
    return operation != WebCore::DragOperationNone;
}

bool WebViewImpl::onDragDrop( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey
                             , WebScriptProxyVariant* dataTransfer)
{
    WebCore::Frame* const pMainFrame = mainFrame();

    KJS::JSLock lock;

    KJS::JSValue* dataTransferValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(dataTransfer);
    KJS::ExecState* exec = pMainFrame->scriptProxy()->globalObject()->globalExec();
    KJS::JSObject* dataTransferObject = dataTransferValue->toObject(exec);

    WebCore::ClipboardApolloHelper dragClipboard(dataTransferObject, exec);

    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);

    WebCore::DragData dragData(&dragClipboard, localPoint, screenPoint, dragClipboard.dragOperationAllowed());
    return m_pPage.get()->dragController()->performDrag(&dragData);
}

void WebViewImpl::onDragExit( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey
                             , WebScriptProxyVariant* dataTransfer)
{
    WebCore::Frame* const pMainFrame = mainFrame();

    KJS::JSLock lock;

    KJS::JSValue* dataTransferValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(dataTransfer);
    KJS::ExecState* exec = pMainFrame->scriptProxy()->globalObject()->globalExec();
    KJS::JSObject* dataTransferObject = dataTransferValue->toObject(exec);

    WebCore::ClipboardApolloHelper dragClipboard(dataTransferObject, exec);

    WebCore::IntPoint const localPoint(localX,localY);
    WebCore::IntPoint const screenPoint(screenX,screenY);

    WebCore::DragData dragData(&dragClipboard, localPoint, screenPoint, dragClipboard.dragOperationAllowed());
    m_pPage.get()->dragController()->dragExited(&dragData);
}

void WebViewImpl::onDragStart( int const localX
                             , int const localY
                             , int const screenX
                             , int const screenY
                             , bool const shiftKey
                             , bool const ctrlKey
                             , bool const altKey
                             , bool const metaKey)
{
}

void WebViewImpl::onDragUpdate(int localX
                              , int localY
                              , int screenX
                              , int screenY
                              , bool shiftKey
                              , bool ctrlKey
                              , bool altKey
                              , bool metaKey)
{
    WebCore::PlatformMouseEvent dragEvent(WebCore::IntPoint(localX, localY), WebCore::IntPoint(screenX, screenY), WebCore::LeftButton, WebCore::MouseEventMoved, 0, shiftKey, ctrlKey, altKey, metaKey, currentTime());
    mainFrame()->eventHandler()->dragSourceMovedTo(dragEvent);
}

void WebViewImpl::onDragComplete( int const localX
                                 , int const localY
                                 , int const screenX
                                 , int const screenY
                                 , bool const shiftKey
                                 , bool const ctrlKey
                                 , bool const altKey
                                 , bool const metaKey
                                 , const uint16_t *dropAction)
{
    page()->dragController()->dragEnded();

    WebCore::PlatformMouseEvent dragEvent(WebCore::IntPoint(localX, localY), WebCore::IntPoint(screenX, screenY), WebCore::LeftButton, WebCore::MouseEventMoved, 0, shiftKey, ctrlKey, altKey, metaKey, currentTime());
    mainFrame()->eventHandler()->dragSourceEndedAt(dragEvent, WebCore::Clipboard::dragOpFromIEOp(WebCore::String((const UChar *) dropAction)));

    // Synthesize a mouse up event and send it to the mouse capture widget.
    WebCore::PlatformMouseEvent mouseUpEvent(WebCore::IntPoint(localX, localY), WebCore::IntPoint(screenX, screenY), WebCore::LeftButton, WebCore::MouseEventReleased, 0, shiftKey, ctrlKey, altKey, metaKey, currentTime());
    ASSERT(m_pMouseCaptureWidget);
    m_pMouseCaptureWidget->handleMouseReleaseEvent(mouseUpEvent);
}

bool WebViewImpl::onCut()
{
    ensureFocusAndCaptureWidgets();
    ASSERT(m_pFocusWidget);
    return m_pFocusWidget->handleCut();
}

bool WebViewImpl::onCopy()
{
    ensureFocusAndCaptureWidgets();
    ASSERT(m_pFocusWidget);
    return m_pFocusWidget->handleCopy();
}

bool WebViewImpl::onPaste()
{
    ensureFocusAndCaptureWidgets();
    ASSERT(m_pFocusWidget);
    return m_pFocusWidget->handlePaste();
}

bool WebViewImpl::onSelectAll()
{
    ensureFocusAndCaptureWidgets();
    ASSERT(m_pFocusWidget);
    return m_pFocusWidget->handleSelectAll();
}

void WebViewImpl::sendResizeEvent( )
{
    m_canTransformToRootContent = false;
    WebCore::Frame* const pMainFrame = mainFrame();
    if (pMainFrame) 
        pMainFrame->sendResizeEvent( );
}

void WebViewImpl::sendScrollEvent( )
{
    m_canTransformToRootContent = false;
    WebCore::Frame* const pMainFrame = mainFrame();
    if (pMainFrame) 
        pMainFrame->sendScrollEvent( );
}



void WebViewImpl::convertToRootContent()
{
    ASSERT(m_canTransformToRootContent);
    ASSERT(m_pTargetWindow);
    WebCore::Frame* const pMainFrame = mainFrame();
    ASSERT(pMainFrame);
    
    WebFrameImpl::kit(pMainFrame)->transformToRootContent(m_pTargetWindow);
    ensureFocusAndCaptureWidgets();
    ASSERT(mainFrame());
    ASSERT(!m_canTransformToRootContent);
    ASSERT(m_pFocusWidget);
    ASSERT(m_pMouseCaptureWidget);
    m_hasHostFocus = true;
}

void WebViewImpl::setShouldPaintDefaultBackground(bool newValue)
{
    WebCore::Frame* const pMainFrame = mainFrame();
    ASSERT( pMainFrame ); 
    if ( newValue )
        pMainFrame->view()->setBaseBackgroundColor( WebCore::Color::white );
    else
        pMainFrame->view()->setBaseBackgroundColor( WebCore::Color( WebCore::makeRGBA( 0x00, 0xff, 0x00, 0x00 ) ) );
    ASSERT(pMainFrame);
}

void WebViewImpl::getContextMenuItems(WebMenuItem*** items, int *nItems)
{
    if(items)
        *items = 0;

    if(nItems)
        *nItems = 0;

    WebCore::ContextMenu* menu = m_pPage->contextMenuController()->contextMenu();
    WebMenu* webMenu = menu ? menu->platformDescription() : 0;
    if (!webMenu || !items || !nItems)
        return;

    *items = webMenu->getItems();
    *nItems = webMenu->getNumItems();
}

void WebViewImpl::getContextMenuItemInfo(WebMenuItem* item, short **title, unsigned int *titleCharLen, int *type, unsigned char *checked, unsigned char *enabled, int *action, WebMenuItem ***subMenuItems, int *numSubMenuItems)
{
    if(title)
        *title = item->getTitle();

    if(titleCharLen)
        *titleCharLen = item->getTitleCharLen();

    if(type)
        *type = item->getType();

    if(checked)
        *checked = item->getChecked();

    if(enabled)
        *enabled = item->getEnabled();

    if(action)
        *action = item->getAction();

    if(subMenuItems)
        *subMenuItems = item->getSubMenuItems();

    if(numSubMenuItems)
        *numSubMenuItems = item->getNumSubMenuItems();
}

void WebViewImpl::updatePlugins(WebCore::Timer<WebViewImpl>* timer)
{
    ASSERT(timer == &m_updatePluginsTimer);
    bool canShowPlugins = false;
    bool canShowWindowedPlugins = false;
    m_pHost->m_pVTable->canShowPlugins(m_pHost, &canShowPlugins, &canShowWindowedPlugins);

    WTF::HashSet<WebPluginWidgetImpl*>::const_iterator ppCurrPluginImpl(m_pluginImpls.begin());
    WTF::HashSet<WebPluginWidgetImpl*>::const_iterator const ppLastPluginImpl(m_pluginImpls.end());
    while (ppCurrPluginImpl != ppLastPluginImpl) {
        WebPluginWidgetImpl* const pCurrPluginImpl = *ppCurrPluginImpl;
        ++ppCurrPluginImpl;
        ASSERT(pCurrPluginImpl);
        pCurrPluginImpl->updatePluginWindow(canShowPlugins, canShowWindowedPlugins);
    }
}

void WebViewImpl::setTextEncodingOverride(const uint16_t* utf16TextEncoding, unsigned long numUTF16CodeUnits)
{
    ASSERT((numUTF16CodeUnits == 0) || (utf16TextEncoding));
    // need to get to the frame loader and set the override text encoding.
    WebCore::Frame* frame = mainFrame();
    if (frame && frame->loader() && frame->loader()->documentLoader()) {
        WebCore::String encodingName(reinterpret_cast<const UChar*>(utf16TextEncoding), numUTF16CodeUnits);
        frame->loader()->documentLoader()->setOverrideEncoding(encodingName);
        if (encodingName.length() == 0) {
            ASSERT(m_pPage.get());
            WebCore::Settings* const pSettings = m_pPage->settings();
            ASSERT(pSettings);
            encodingName = pSettings->defaultTextEncodingName();
        }
        frame->loader()->reloadAllowingStaleData(encodingName);
    }
}

void WebViewImpl::getTextEncodingOverride(uint16_t** utf16TextEncoding)
{
    ASSERT(utf16TextEncoding);

    // need to get to the frame loader and set the override text encoding.
    WebCore::Frame* frame = mainFrame();
    if (utf16TextEncoding && frame && frame->loader() && frame->loader()->documentLoader()) {
        WebCore::String encodingName = frame->loader()->documentLoader()->overrideEncoding();
        *utf16TextEncoding = (uint16_t *)g_HostFunctions->allocBytes(sizeof(uint16_t)*(encodingName.length()+1));
        if (encodingName.length() > 0)
            memcpy(*utf16TextEncoding, encodingName.characters(), sizeof(uint16_t)*(encodingName.length()));
        (*utf16TextEncoding)[encodingName.length()] = 0;
    }
}

void WebViewImpl::setTextEncodingFallback(const uint16_t* utf16TextEncoding, unsigned long numUTF16CodeUnits)
{
    ASSERT((numUTF16CodeUnits == 0) || (utf16TextEncoding));
    if (numUTF16CodeUnits) {
        ASSERT(m_pPage.get());
        WebCore::Settings* const pSettings = m_pPage->settings();
        ASSERT(pSettings);
        WebCore::String const encodingName(reinterpret_cast<const UChar*>(utf16TextEncoding), numUTF16CodeUnits);
        pSettings->setDefaultTextEncodingName(encodingName);
    }
}

void WebViewImpl::getTextEncodingFallback(uint16_t** utf16TextEncoding)
{
    ASSERT(utf16TextEncoding);
    ASSERT(m_pPage.get());
    WebCore::Settings* const pSettings = m_pPage->settings();
    ASSERT(pSettings);
    if (utf16TextEncoding) {
        const WebCore::String &encodingName = pSettings->defaultTextEncodingName();
        *utf16TextEncoding = (uint16_t *)g_HostFunctions->allocBytes( sizeof(uint16_t)*(encodingName.length()+1) );
        if (encodingName.length() > 0)
            memcpy(*utf16TextEncoding, encodingName.characters(), sizeof(uint16_t)*(encodingName.length()));
        (*utf16TextEncoding)[encodingName.length()] = 0;
    }
}

void WebViewImpl::initSettings()
{
    ASSERT(m_pPage.get());
    WebCore::Settings* const pSettings = m_pPage->settings();
    ASSERT(pSettings);
    pSettings->setStandardFontFamily("Times");
    pSettings->setFixedFontFamily("Courier");
    pSettings->setSerifFontFamily("Times");
    pSettings->setSansSerifFontFamily("Helvetica");
    pSettings->setCursiveFontFamily("Apple Chancery");
    pSettings->setFantasyFontFamily("Papyrus");
    pSettings->setMinimumFontSize(1);
    pSettings->setMinimumLogicalFontSize(9);
    pSettings->setDefaultFontSize(16);
    pSettings->setDefaultFixedFontSize(13);
    pSettings->setLoadsImagesAutomatically(true);
    pSettings->setJavaScriptEnabled(true);
    pSettings->setJavaScriptCanOpenWindowsAutomatically(false);  // requires user gesture
    pSettings->setJavaEnabled(false);
    pSettings->setPluginsEnabled(true);
    pSettings->setPrivateBrowsingEnabled(false);
    pSettings->setDefaultTextEncodingName("ISO-8859-1");
    pSettings->setUserStyleSheetLocation(WebCore::KURL(""));
    pSettings->setShouldPrintBackgrounds(false);
    pSettings->setTextAreasAreResizable(false);
    pSettings->setEditableLinkBehavior(WebCore::EditableLinkDefaultBehavior);
    pSettings->setNeedsAdobeFrameReloadingQuirk(false);
    pSettings->setShrinksStandaloneImagesToFit(false);
}

void WebViewImpl::addPluginImpl(WebPluginWidgetImpl* const pluginImpl)
{
    ASSERT(!m_pluginImpls.contains(pluginImpl));
    m_pluginImpls.add(pluginImpl);
    ASSERT(m_pluginImpls.contains(pluginImpl));

    if (!m_updatePluginsTimer.isActive()) {
        static const int updatePluginTimerPeriodInMiliSeconds = 34;
        m_updatePluginsTimer.startRepeating( static_cast<double>(updatePluginTimerPeriodInMiliSeconds) / 1000 );
    }
}

void WebViewImpl::removePluginImpl(WebPluginWidgetImpl* const pluginImpl)
{
    ASSERT(m_pluginImpls.contains(pluginImpl));
    m_pluginImpls.remove(pluginImpl);
    ASSERT(!m_pluginImpls.contains(pluginImpl));
    if (m_pluginImpls.isEmpty())
        m_updatePluginsTimer.stop();
}

WebEditorClient* WebViewImpl::webEditorClient() const
{
    WebCore::EditorClient* const editorClient = m_pPage->editorClient();
    ASSERT(editorClient);
    WebEditorClient* const webEditorClient = WebEditorClient::kit(editorClient);
    ASSERT(webEditorClient);
    return webEditorClient;
}

void WebViewImpl::setPageGroupName(const uint16_t* groupName)
{
    m_pPage->setGroupName(groupName ? WebCore::String((const UChar*)groupName) : WebCore::String());
}

WebString* WebViewImpl::getPageGroupName()
{
    WebCore::String groupName = m_pPage->groupName();

    return groupName.webString();
}

inline WebCore::FrameView* WebViewImpl::focusedOrMainFrameView() const
{
    WebCore::FocusController* const controller = m_pPage->focusController();
    WebCore::Frame* const frame = controller->focusedOrMainFrame();
    WebCore::FrameView* const view = frame->view();
    return view;
}

WEBKIT_APOLLO_PROTO1 WebView* WEBKIT_APOLLO_PROTO2 webViewCreate(WebHost* pWebHost, WebWindow* pTargetWindow, WebError* pErrorInfo )
{
    ASSERT(pErrorInfo);

    WebViewImpl* const pWebViewImpl = new WebViewImpl(pWebHost, pTargetWindow);
    
    if (pWebViewImpl) {
        pErrorInfo->m_Type = WebErrorNoError;
    }
    else {
        pErrorInfo->m_Type = WebErrorNoMemory;
    }
    return pWebViewImpl->getWebView();
}

}

