/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WebScrollbarWidgetImpl.h"
#include <IScrollViewApolloImpl.h>

namespace WebKitApollo {
WebScrollbarWidgetImpl::WebScrollbarWidgetImpl( IFocusManager* const pFocusManager
                                             , IMouseCaptureManager* const pMouseCaptureManager
                                             , WebCore::PlatformScrollbar* const pScrollbar)
    : WebNonPluginWidgetImpl( pFocusManager, pMouseCaptureManager )
    , m_pScrollbar( pScrollbar )
{
    ASSERT(m_pScrollbar);
}

WebScrollbarWidgetImpl::~WebScrollbarWidgetImpl()
{
}

WTF::RefPtr<WebScrollbarWidgetImpl> WebScrollbarWidgetImpl::construct( IFocusManager* const pFocusManager
                                                                     , IMouseCaptureManager* const pMouseCaptureManager
                                                                     , WebCore::PlatformScrollbar* const pScrollbar)
{
    return WTF::RefPtr<WebScrollbarWidgetImpl>(new WebScrollbarWidgetImpl(pFocusManager, pMouseCaptureManager, pScrollbar));
}


void WebScrollbarWidgetImpl::paint(WebCore::GraphicsContext* pGraphicsContext, const WebCore::IntRect& dirtyRect)
{
    ASSERT(false);
}

void WebScrollbarWidgetImpl::invalidateRect(const WebCore::IntRect& dirtyRect)
{
    ASSERT(false);
}

void WebScrollbarWidgetImpl::setCursor( const WebCore::Cursor& cursor )
{
    ASSERT(m_pScrollbar);
    WTF::PassRefPtr<WebCore::IWidgetApolloImpl> impl = m_pScrollbar->getApolloImpl();
    ASSERT(impl.get());
    WebCore::IScrollViewApolloImpl* scrollView = impl->getParent();
    ASSERT(scrollView);
	scrollView->setCursor( cursor );
}

void WebScrollbarWidgetImpl::show()
{
    ASSERT( false );
}

void WebScrollbarWidgetImpl::hide()
{
    ASSERT(false);
}

bool WebScrollbarWidgetImpl::handleMousePressEvent( const WebCore::PlatformMouseEvent& mouseEvent )
{
    ASSERT(m_pScrollbar);
    WebCore::PlatformMouseEvent normalizedMouseEvent = m_pScrollbar->normalizeMouseEvent( mouseEvent );
    return m_pScrollbar->handleMousePressEventNormalized( normalizedMouseEvent );
}

bool WebScrollbarWidgetImpl::handleMouseMoveEvent( const WebCore::PlatformMouseEvent& mouseEvent )
{
    ASSERT(m_pScrollbar);
    WTF::PassRefPtr<WebCore::IWidgetApolloImpl> impl = m_pScrollbar->getApolloImpl();
    ASSERT(impl.get());
    WebCore::IScrollViewApolloImpl* scrollView = impl->getParent();
    ASSERT(scrollView);
    WebCore::IScrollViewApolloImpl* topMostScrollView = scrollView;
    WebCore::IntSize offset(0, 0);
    while(topMostScrollView->getParent())
    {  
        topMostScrollView = topMostScrollView->getParent();
		if (scrollView != topMostScrollView)
		{
			WebCore::IntRect const frame(scrollView->frameGeometry());
			WebCore::IntPoint const pt(topMostScrollView->contentsToViewport(WebCore::IntPoint(frame.x(), frame.y())));
			offset -= WebCore::IntSize( pt.x(), pt.y() );
		}
    }
    
    if (!(m_pScrollbar == scrollView->getHScrollbar() || m_pScrollbar == scrollView->getVScrollbar()))
    {
        offset += scrollView->scrollOffset();
    }
    
    WebCore::IntPoint const transformedPoint(mouseEvent.pos() + offset);    
    WebCore::PlatformMouseEvent normalizedMouseEvent( transformedPoint,
                                                    WebCore::IntPoint( mouseEvent.globalX(), mouseEvent.globalY() ),
                                                    mouseEvent.button(),
                                                    mouseEvent.eventType(),
                                                    mouseEvent.clickCount(),
                                                    mouseEvent.shiftKey(),
                                                    mouseEvent.ctrlKey(),
                                                    mouseEvent.altKey(),
                                                    mouseEvent.metaKey(),
                                                    mouseEvent.timestamp() );
    return m_pScrollbar->handleMouseMoveEventNormalized( normalizedMouseEvent );
}

bool WebScrollbarWidgetImpl::handleMouseReleaseEvent( const WebCore::PlatformMouseEvent& mouseEvent )
{
    ASSERT(m_pScrollbar);
    WebCore::PlatformMouseEvent normalizedMouseEvent = m_pScrollbar->normalizeMouseEvent( mouseEvent );
    return m_pScrollbar->handleMouseReleaseEventNormalized( normalizedMouseEvent );
}

bool WebScrollbarWidgetImpl::handleWheelEvent( WebCore::PlatformWheelEvent& wheelEvent )
{
    return false;
}

bool WebScrollbarWidgetImpl::handleKeyboardEvent( const WebCore::PlatformKeyboardEvent& keyEvent )
{
    ASSERT(m_pScrollbar);
    return m_pScrollbar->handleKeyboardEvent(keyEvent);
}

bool WebScrollbarWidgetImpl::handleInsertText(const WebCore::String& text)
{
    ASSERT(m_pScrollbar);
    return m_pScrollbar->handleInsertText(text);
}

WebCore::Widget* WebScrollbarWidgetImpl::getWidget()
{
    return m_pScrollbar;
}

}

