/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "WebResourceHandleClientImpl.h"
#include "WebResourceRequestImpl.h"
#include <ResourceHandleClient.h>
#include <ResourceHandle.h>
#include <ResourceResponse.h>
#include <ResourceRequest.h>
#include <wtf/RefPtr.h>
#include <wtf/HashSet.h>

namespace WebKitApollo {

WebResourceHandleClientImpl::WebResourceHandleClientImpl(WebCore::ResourceHandle* pResourceHandle, WebCore::ResourceHandleClient* client)
    : m_resourceHandle(pResourceHandle)
    , m_client(client)
    , m_gotAllData(false)
{
    ASSERT(m_resourceHandle);
    ASSERT(m_client);
}

WebResourceHandleClientImpl::~WebResourceHandleClientImpl()
{
}

WebResourceRequest* WebResourceHandleClientImpl::receivedRedirect(struct WebResourceRequest* proposedNewWebRequest, struct WebResourceResponse* redirectWebResponse)
{
    WebCore::ResourceRequest newRequest(proposedNewWebRequest);
    WebCore::ResourceResponse redirectResponse(redirectWebResponse);
    ASSERT(m_client);
    m_client->willSendRequest(m_resourceHandle, newRequest, redirectResponse);
    WTF::RefPtr<WebResourceRequestImpl> newWebRequest(WebResourceRequestImpl::construct(newRequest));
    return newWebRequest.release().releaseRef()->getWebResourceRequest();
}

void WebResourceHandleClientImpl::receivedResponse(struct WebResourceResponse* webResponse)
{
    WebCore::ResourceResponse response(webResponse);
    ASSERT(m_client);
    m_client->didReceiveResponse(m_resourceHandle, response);
}

void WebResourceHandleClientImpl::receivedData(const unsigned char* pData, unsigned long numDataBytes)
{
    ASSERT(m_client);
    m_client->didReceiveData(m_resourceHandle, reinterpret_cast<const char *>(pData), numDataBytes, numDataBytes);
}

void WebResourceHandleClientImpl::receivedAllData()
{
    ASSERT(!m_gotAllData);
    m_gotAllData = true;
    m_client->didFinishLoading(m_resourceHandle);
}

void WebResourceHandleClientImpl::didFail(const unsigned char* failingURLBytes, unsigned long numFailingURLBytes)
{
    WebCore::String const failingURLStr(reinterpret_cast<const char*>(failingURLBytes), numFailingURLBytes);
    WebCore::String const emptryStr;
    WebCore::ResourceError const resourceError(emptryStr, 0, failingURLStr, emptryStr);
    m_client->didFail(m_resourceHandle, resourceError);
}

void WebResourceHandleClientImpl::release()
{
    delete this;
}


void WebResourceLoaderSynchronousClientImpl::loadResourceSynchronously(WebHost* const webHost, const WebCore::ResourceRequest& request, WebCore::ResourceError& error, WebCore::ResourceResponse& response, WTF::Vector< char >& bytes)
{
    WebCore::ResourceRequest currRequest(request);
    
    
    WTF::HashSet<WebCore::String> redirectSet;
    static const unsigned int maxRedirects = 12;
    bool done = false;
    
    unsigned int redirectCount = 0;
    while ((!done) && (redirectCount < maxRedirects) && (!redirectSet.contains(currRequest.url().string()))) {
        WTF::RefPtr<WebResourceRequestImpl> const pWebResourceRequestImpl(WebResourceRequestImpl::construct(currRequest));
        WebResourceRequest* const pWebResourceRequest = pWebResourceRequestImpl->getWebResourceRequest();
        
        bool gotRedirect = false;
        WebCore::ResourceRequest redirectRequest;
        WebResourceLoaderSynchronousClientImpl* const pWebResourceHandleClientImpl = new WebResourceLoaderSynchronousClientImpl(&response, &error, &bytes, &gotRedirect, &redirectRequest);
        ::WebResourceHandleClient* const pWebResourceHandleClient = pWebResourceHandleClientImpl->getWebResourceHandleClient();
        webHost->m_pVTable->loadResourceSynchronously(webHost, pWebResourceRequest, pWebResourceHandleClient);
        
        if (gotRedirect) {
            ++redirectCount;
            redirectSet.add(currRequest.url().string());
            currRequest = redirectRequest;
            error = WebCore::ResourceError();
        }
        else {
            done = true;
        }
    }
    if (!done) {
        ASSERT((redirectCount == maxRedirects) || (redirectSet.contains(currRequest.url().string())));
        WebCore::String const emptryStr;
        error = WebCore::ResourceError(emptryStr, 0, request.url().string(), emptryStr);
    }
    else {
        ASSERT((redirectCount < maxRedirects) && (!redirectSet.contains(currRequest.url().string())));
    }
}

WebResourceLoaderSynchronousClientImpl::WebResourceLoaderSynchronousClientImpl(WebCore::ResourceResponse* const response, WebCore::ResourceError* const error, WTF::Vector< char >* bytes, bool* const gotRedirect, WebCore::ResourceRequest* const redirectRequest)
    : m_response(response)
    , m_error(error)
    , m_bytes(bytes)
    , m_gotRedirect(gotRedirect)
    , m_redirectRequest(redirectRequest)
{
}

WebResourceLoaderSynchronousClientImpl::~WebResourceLoaderSynchronousClientImpl()
{
}

struct WebResourceRequest* WebResourceLoaderSynchronousClientImpl::receivedRedirect(struct WebResourceRequest* proposedNewWebRequest, struct WebResourceResponse*)
{
   ASSERT(proposedNewWebRequest);
   ASSERT(proposedNewWebRequest->m_pVTable);
   ASSERT(proposedNewWebRequest->m_pVTable->acquire);
   *m_redirectRequest = WebCore::ResourceRequest(proposedNewWebRequest);
   *m_gotRedirect = true;
   return 0;
}

void WebResourceLoaderSynchronousClientImpl::receivedResponse(struct WebResourceResponse* webResponse)
{
    m_response->set(webResponse);
}

void WebResourceLoaderSynchronousClientImpl::receivedData( const unsigned char* pData, unsigned long numDataBytes )
{
    if ( numDataBytes > 0 )
    {
        size_t const currBufferSize = m_bytes->size();
        size_t const newBufferSize = currBufferSize + numDataBytes;
        ASSERT( currBufferSize < newBufferSize );
        m_bytes->resize( newBufferSize );
        char* const pBuffer = m_bytes->data();
        char* const pDest = pBuffer + currBufferSize;
        memcpy(pDest, pData, numDataBytes);
    }
}

void WebResourceLoaderSynchronousClientImpl::receivedAllData()
{
}

void WebResourceLoaderSynchronousClientImpl::didFail(const unsigned char* failingURLBytes, unsigned long numFailingURLBytes)
{
    WebCore::String const failingURLStr(reinterpret_cast<const char*>(failingURLBytes), numFailingURLBytes);
    WebCore::String const emptryStr;
    *m_error = WebCore::ResourceError(emptryStr, 0, failingURLStr, emptryStr);
}

void WebResourceLoaderSynchronousClientImpl::release()
{
    delete this;
}
}
