#include "WebPluginImpl.h"
#if PLATFORM(DARWIN)
#include "WebPluginWK.h"
#elif PLATFORM(WIN_OS)
#include "WebPluginNP.h"
#elif PLATFORM(UNIX_OS)
#include "WebPluginNP.h"
#endif
#include "WebFrameScrollView.h"
#include "ResourceHandle.h"
#include "Document.h"
#include "Event.h"
#include "EventNames.h"
#include "EventListener.h"
#include "PluginInfoStore.h"
#include "FrameTree.h"
#include "FrameView.h"
#include "WebViewImpl.h"
#include "MIMETypeRegistry.h"

#include <runtime_root.h>

#define DEBUG_LOG_PLUGIN_IMPL 0

#if DEBUG_LOG_PLUGIN_IMPL
#if PLATFORM(WIN_OS)
#pragma warning( disable : 4996 )
#include <Windows.h>
#endif

static inline void dbgprintf(const char *format, ...)
{
    va_list args;
    
    va_start(args, format);
    if (args)
    {
        static char string[2048];

        vsnprintf(string, sizeof(string)-1, format, args);
        string[sizeof(string)-1] = '\0';

#if WIN_PLATFORM    // Display in the VC debugger
        OutputDebugStringA(string);
#else
        printf("%s", string);
#endif /* platform-specific */

        va_end(args);
    }

}
#else
static inline void dbgprintf(const char *format, ...) { }
#endif //DEBUG_LOG_PLUGIN_IMPL

namespace WebKitApollo
{

    PluginImpl::PluginImpl( WebCore::Element *element,
                            const WebCore::KURL& url,
                            const Vector<WebCore::String>& paramNames, 
                            const Vector<WebCore::String>& paramValues,
                            const WebCore::String& mimeType,
                            WebFrameImpl* const pWebFrameOwner )
        : WebPluginWidgetImpl(pWebFrameOwner->webViewImpl())
        , mHasFocus(false)
        , mWebFrameOwner(pWebFrameOwner)
        , mElement(element)
        , mURL(url)
        , mMimeType(mimeType)
        , mParamNames(paramNames)
        , mParamValues(paramValues)
        , mOwningWidget(0)
    {
        ASSERT(mWebFrameOwner);
    }

    PluginImpl::~PluginImpl()
    {
    }

    WTF::PassRefPtr< PluginImpl > PluginImpl::construct(    WebCore::Element* const element,
                                                        const WebCore::KURL& url,
                                                        const Vector<WebCore::String>& paramNames, 
                                                        const Vector<WebCore::String>& paramValues,
                                                        const WebCore::String& mimeType,
                                                        WebFrameImpl* const pWebFrameOwner )
    {
        //We're not given a mimetype if the OBJECT element doesn't have the "type" property; we
        //can try to sniff it by extension.
        //XXX TBD: this s*cks.  We need browser content-sniffing.
        WebCore::String localMimeType = mimeType;
        if ( mimeType.length() == 0 )
        {
            WebCore::String title = url.lastPathComponent();
            if ( title.length() > 1 )
            {
                int pos = title.reverseFind('.');
                if ( pos > 0 )
                {
                    WebCore::String ext = title;
                    ext.remove(0, pos + 1);
                    if ( ext.length() > 0 )
                    {
                        localMimeType = WebCore::MIMETypeRegistry::getMIMETypeForExtension( ext );
                    }
                }
            }
        }

        //Depending on the mimetype, we create different client implementations; note that within
        //each platform's implementation it will also check mimetype.
        if ( WebCore::PluginInfoStore::isPDFMIMEType( localMimeType ) )
        {
            #if PLATFORM(DARWIN)
                return PluginWK::construct( element, url, paramNames, paramValues, localMimeType, pWebFrameOwner );
            #elif PLATFORM(WIN_OS)
                return PluginNP::construct( element, url, paramNames, paramValues, localMimeType, pWebFrameOwner );
            #elif PLATFORM(UNIX_OS)
                return PluginNP::construct( element, url, paramNames, paramValues, localMimeType, pWebFrameOwner );
            #else
                ASSERT( false );
                return NULL;
            #endif
        }
        else
        {    
            //other mimetypes would go here
            return NULL;
            //return WTF::RefPtr< PluginImpl >( new PluginImpl( element, url, paramNames, paramValues, localMimeType, pWebFrameOwner ) );
        }
    }

    void PluginImpl::setCursor( const WebCore::Cursor& cursor )
    {
        dbgprintf( "unimplemented PluginImpl::setCursor\n" );
    }

    void PluginImpl::removeFromParent()
    {
       dbgprintf( "unimplemented PluginImpl::removeFromParent\n" );
    }

    void PluginImpl::paint( WebCore::GraphicsContext* pGraphicsContext, const WebCore::IntRect& dirtyRect )
    {
        dbgprintf( "unimplemented PluginImpl::paint\n" );
    }
    
    static WebCore::IntRect getFrameVisibleWindowRect(WebCore::Frame* frame)
    {
        WTF::RefPtr<WebCore::IScrollViewApolloImpl> scrollViewImpl = frame->view()->getApolloImpl();
        FrameScrollViewImpl* frameScrollViewImpl = static_cast<FrameScrollViewImpl*>( scrollViewImpl.get() );

        WebCore::IntPoint topLeft(0,0);
        topLeft = frameScrollViewImpl->viewportToWindow( topLeft );

        WebCore::IntPoint bottomRight(frameScrollViewImpl->visibleWidth(), frameScrollViewImpl->visibleHeight());
        bottomRight = frameScrollViewImpl->viewportToWindow( bottomRight );

        return WebCore::IntRect(topLeft.x(), topLeft.y(), bottomRight.x() - topLeft.x(), bottomRight.y() - topLeft.y());
    }

    static WebCore::IntRect getFrameClippedWindowRect(WebCore::Frame* innerFrame)
    {
        WebCore::IntRect windowRect = getFrameVisibleWindowRect(innerFrame);

        WebCore::Frame* outerFrame = innerFrame;
        WebCore::Frame* nextFrame = outerFrame->tree()->parent();
        while(nextFrame)
        {
            outerFrame = nextFrame;

            WebCore::IntRect outerWindowRect = getFrameVisibleWindowRect(outerFrame);
            windowRect.intersect(outerWindowRect);

            nextFrame = outerFrame->tree()->parent();
        }

        return windowRect;
    }

    WebCore::IntPoint PluginImpl::contentsToWindow( const WebCore::IntPoint& contentsPoint ) const
    {
        WebFrameImpl *const apolloFrame = getWebFrameOwner();
        const WTF::RefPtr<WebCore::Frame>& webCoreFrame = apolloFrame->frame();
        WTF::RefPtr<WebCore::IScrollViewApolloImpl> scrollViewImpl = webCoreFrame->view()->getApolloImpl();
        FrameScrollViewImpl* frameScrollViewImpl = static_cast<FrameScrollViewImpl*>( scrollViewImpl.get() );

        // this returns the point relative to the top left of the enclosing frame accounting for scroll
        WebCore::IntPoint viewportPoint = frameScrollViewImpl->contentsToViewport( contentsPoint );

        return frameScrollViewImpl->viewportToWindow( viewportPoint );
    }
    
    WebCore::ResourceHandleClient* PluginImpl::pluginRedirectClient()
    {
        return 0;
    }
    
    WebCore::Widget* PluginImpl::getWidget()
    {
        ASSERT(mOwningWidget);
        return mOwningWidget;
    }
    
    void PluginImpl::setWidget(WebCore::Widget* w)
    {
        ASSERT(w);
        ASSERT(!mOwningWidget);
        mOwningWidget = w;
    }

    void PluginImpl::setFrameGeometry(const WebCore::IntRect &rect)
    {
        WebFrameImpl *const innerApolloFrame = getWebFrameOwner();
        const WTF::RefPtr<WebCore::Frame>& innerFrame = innerApolloFrame->frame();

        //    Pass it on to the subclass if it's different than the geometry
        //    which means we've been moved or resized within our container,
        //    or if the final window rectf is different, which means the container
        //    has been scrolled.
        WebCore::IntRect newWindowRect = rect;

        // convert to window coords
        newWindowRect.setLocation( contentsToWindow( rect.location() ) );

        // get the window rect of the frame this widget is in, clipped by all it's parent frames
        WebCore::IntRect newClippedFrameWindowRect = getFrameClippedWindowRect(innerFrame.get());
        WebCore::IntRect newClipRect = newWindowRect;
        newClipRect.intersect( newClippedFrameWindowRect );

        dbgprintf( "PIIMP::setFrameGeo: rect/newWindowRect/newClippedFrameWindowRect/newClipRect [ %4d %4d ],[ %4d %4d ]", rect.x(), rect.y(), rect.width(), rect.height() );
        dbgprintf( "\t[ %4d %4d ],[ %4d %4d ]", newWindowRect.x(), newWindowRect.y(), newWindowRect.width(), newWindowRect.height() );
        dbgprintf( "\t[ %4d %4d ],[ %4d %4d ]", newClippedFrameWindowRect.x(), newClippedFrameWindowRect.y(), newClippedFrameWindowRect.width(), newClippedFrameWindowRect.height() );
        dbgprintf( "\t[ %4d %4d ],[ %4d %4d ]", newClipRect.x(), newClipRect.y(), newClipRect.width(), newClipRect.height() );

        if (    !( frameGeometry() == rect )
             || !( newWindowRect == mWindowRect )
             || !( newClipRect == mClippedWindowRect )
             || !( newClippedFrameWindowRect == mClippedFrameWindowRect )
            )
        {
            WebCore::IntRect oldRect = frameGeometry();
            dbgprintf( "\t--------- NEW VALUES --------\n" );

            //    This is the new geometry, in local coords (relative to parent object)
            WebWidgetImpl::setFrameGeometry( rect );

            //    This is the window rect (size of PDF chrome etc), in window-coords
            mWindowRect = newWindowRect;

            //    This is the new clipped rect (PDF chrome etc gets clipped to here), in window coords
            mClippedWindowRect = newClipRect;

            //    This is the new content rect of the containing HTML, in window coords
            mClippedFrameWindowRect = newClippedFrameWindowRect;

            frameGeometryHasChanged( oldRect, mWindowRect, mClippedFrameWindowRect );
        }
        else
        {
            dbgprintf( "\tNO CHANGE\n" );
        }
    }
    
    void* PluginImpl::createPluginScriptableObject()
    {
        // should never get here because we be overriding this in all the subclasses that actually get instantiated.
        // XXX TODO remove code that instantiates PluginImpl(this class), so this method can be removed from this class ( subclasses will still override ).
        ASSERT(false);
        return 0;
    }
    
    void PluginImpl::releasePluginScriptableObject(void* object)
    {
        // should never get here because we be overriding this in all the subclasses that actually get instantiated.
        // XXX TODO remove code that instantiates PluginImpl(this class), so this method can be removed from this class ( subclasses will still override ).
        ASSERT(false);
    }

    // XXX - I'm not really sure we should be overriding these
    bool PluginImpl::handleMousePressEvent( const WebCore::PlatformMouseEvent& mouseEvent )
    {
        dbgprintf( "PluginImpl::handleMousePressEvent\n" );
        return false;
    }

    bool PluginImpl::handleMouseMoveEvent( const WebCore::PlatformMouseEvent& mouseEvent )
    {
        dbgprintf( "PluginImpl::handleMouseMoveEvent\n" );
        return false;
    }

    bool PluginImpl::handleWheelEvent( WebCore::PlatformWheelEvent& wheelEvent )
    {
        dbgprintf( "PluginImpl::handleWheelEvent\n" );
        return false;
    }

    bool PluginImpl::handleMouseReleaseEvent( const WebCore::PlatformMouseEvent& mouseEvent )
    {
        dbgprintf( "PluginImpl::handleMouseReleaseEvent\n" );
        return false;
    }

    bool PluginImpl::handleKeyboardEvent( const WebCore::PlatformKeyboardEvent& keyEvent )
    {
        dbgprintf( "PluginImpl::handleKeyboardEvent\n" );
        return false;
    }

    bool PluginImpl::handleInsertText(const WebCore::String& text)
    {
        dbgprintf("PluginImpl::handleInsertText\n");
        return false;
    }

    bool PluginImpl::hasFocus() const
    {
        return mHasFocus;
    }

    void PluginImpl::setFocus()
    {
        mHasFocus = true;
    }

    void PluginImpl::clearFocus()
    {
        mHasFocus = false;
    }

    void PluginImpl::beginModal() const
    {
        WebFrameImpl *const apolloFrame = getWebFrameOwner();
        apolloFrame->beginModal();
    }

    void PluginImpl::endModal() const
    {
        WebFrameImpl *const apolloFrame = getWebFrameOwner();
        apolloFrame->endModal();
    }

    bool PluginImpl::isEnabled() const
    {
        return true;
    }

    bool PluginImpl::allowURLRequest( const WebCore::KURL sourceURL, const WebCore::KURL requestedURL, WebCore::Frame *targetFrame )
    {
        //    Use the frame's security.
        WebFrameImpl *const apolloFrame = getWebFrameOwner();
        const WTF::RefPtr<WebCore::Frame>& webCoreFrame = apolloFrame->frame();
        WebCore::FrameLoader *frameLoader = webCoreFrame->loader();
        WebCore::FrameLoaderClientApollo *frameLoaderClient = reinterpret_cast<WebCore::FrameLoaderClientApollo *>( frameLoader->client() );
        return frameLoaderClient->canLoad( requestedURL, sourceURL );
    }

    bool PluginImpl::allowURLRequest( const WebCore::KURL requestedURL, WebCore::Frame *targetFrame )
    {
        return allowURLRequest( getUrl(), requestedURL, targetFrame );
    }
    
    //    Resource-loading routines
    
    void PluginImpl::loadURLForClient( const WebCore::KURL& urlToLoad, void *client )
    {
        WebCore::HTTPHeaderMap headerMap;

        loadURLForClient( urlToLoad, "GET", NULL, 0, client, headerMap );
    }

    void PluginImpl::loadURLForClient(  const WebCore::KURL& urlToLoad, const char *method,
                                        const char *data, unsigned int dataLength,
                                        void *client,
                                        const WebCore::HTTPHeaderMap& headerMap,
                                        WebCore::Frame *frame )
    {
        //    In cerain error cases, including where there is no "data" property in
        //    the pdf OBJECT element, we're asked to load an empty URL.  Short-circuit
        //    that here.
        if ( urlToLoad.isEmpty() )
            return;

        WebCore::ResourceRequest resourceRequest( urlToLoad );

        // XXX - is replacement correct?
        WebCore::String sMethod(method);
        resourceRequest.setHTTPMethod( sMethod );
        //resourceRequest.setDoPost( method != NULL && 0 == strcmp( method, "POST" ) );

        //    Add the POST body if there is any
        ASSERT((!data && !dataLength) || (data && dataLength));
        
        if (data && dataLength)
        {
            // XXX - correct?
            resourceRequest.setHTTPBody(WebCore::FormData::create(data, dataLength));
            //resourceRequest.postData.appendData( data, dataLength );
        }

        // XXX - correct?
        resourceRequest.addHTTPHeaderFields( headerMap );
        resourceRequest.setHTTPUserAgent(  mWebFrameOwner->frame()->loader()->userAgent(urlToLoad) );

        if ( frame != NULL )
        {
            //    We can specifically state that the frame itself needs to be loading this instead
            //    of our plug-in. This happens in two cases: when a different frame is targeted,
            //    in which case frame != mWebFrameOwner->getFrame(), or when PDF has made a request
            //    that it believes will return non-PDF/FDF content.
            frame->loader()->load( resourceRequest );
        }
        else
        {
            //    This code matches FrameApolloClientImpl->openURLRequest() almost exactly; this is for
            //    requests that want to go back to the plugin, not the frame.
            RefPtr<WebCore::ResourceHandle> resourceHandle;

            // XXX - defersLoading == false?   mightDownloadFromHandle == false?
            resourceHandle = WebCore::ResourceHandle::create(resourceRequest, this, mWebFrameOwner->frame(), false, true );

            if ( resourceHandle )
            {
                mResourceHandleClientMap.set(resourceHandle, client);
                mClientResourceHandleMap.set(client, resourceHandle);
            }
            else
            {
                mWebFrameOwner->frame()->loader()->stop();
            }
        }
    }

    void PluginImpl::willSendRequest(WebCore::ResourceHandle* resourceHandle, WebCore::ResourceRequest&, const WebCore::ResourceResponse& redirectResponse)
    {
        // XXX - is url correct here, doing right thing?   this used to be in receivedRedirect
        void *client = resourceHandle ? clientFromHandle(resourceHandle) : 0;
        clientReceivedRedirect( client, resourceHandle, redirectResponse.url() );
    }
    
    void PluginImpl::didReceiveResponse(WebCore::ResourceHandle* resourceHandle, const WebCore::ResourceResponse& response)
    {
        void *client = resourceHandle ? clientFromHandle(resourceHandle) : 0;
        clientReceivedResponse( client, resourceHandle, response );
    }

    void PluginImpl::didReceiveData(WebCore::ResourceHandle* resourceHandle, const char* data, int, int lengthReceived)
    {
        void *client = resourceHandle ? clientFromHandle(resourceHandle) : 0;
        clientReceivedData( client, resourceHandle, data, lengthReceived );
    }

    void PluginImpl::didFinishLoading(WebCore::ResourceHandle* resourceHandle)
    {
        void *client = resourceHandle ? clientFromHandle(resourceHandle) : 0;
        clientReceivedAllData( client, resourceHandle );
        
        clearClientHandle( resourceHandle );
        clearHandleFromClient( client );
    }

    /*  XXX - do we still need this?
    void PluginImpl::receivedAllData(WebCore::ResourceHandle* resourceHandle, WebCore::PlatformData platformData)
    {
        clientReceivedAllData( clientFromHandle(resourceHandle), resourceHandle, platformData );
        // Do not clear the client loader here.  This is because both versions of received all data are called.
        // This version is called first, and if it clears the client loader, then the receivedAllData above
        // can't find a client. 
        // NO! Do not do this --> clearClientLoader( resourceHandle );
    }*/

    unsigned long PluginImpl::resourceResponseContentLength( const WebCore::ResourceResponse& resourceResponse )
    {
        WebCore::String contentLength = resourceResponse.httpHeaderField("Content-Length");

        bool bOk = false;
        unsigned long length = contentLength.toInt(&bOk);

        ASSERT((bOk && length) || contentLength.length() == 0);

        return bOk ? length : 0xFFFFFFFF;
    }

}
