/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "WebFrameScrollView.h"
#include <GraphicsContext.h>
#include <FocusController.h>
#include <Frame.h>
#include <FrameTree.h>
#include <FrameView.h>
#include <Page.h>
#include <HTMLFrameElementBase.h>

#include "WebFrameImpl.h"
#include <PlatformMouseEvent.h>
#include <Cursor.h>
#include <IntPoint.h>
#include "WebViewImpl.h"
#include "IPluginWidgetApolloImpl.h"
#include <Logging.h>

namespace WebKitApollo
{

namespace {
#ifndef NDEBUG
WTFLogChannel LogWebFrameScrollViewLeaks =  { 0x00000000, "", WTFLogChannelOn };

struct WebFrameScrollViewCounter { 
    static int count; 
    ~WebFrameScrollViewCounter() 
    { 
        if (count)
            LOG(WebFrameScrollViewLeaks, "LEAK: %d WebFrame\n", count);
    }
};
int WebFrameScrollViewCounter::count = 0;
static WebFrameScrollViewCounter frameScrollViewCounter;
#endif
}

FrameScrollViewImpl::FrameScrollViewImpl( WebFrameImpl* const pWebFrameOwner
                                        , WebCore::FrameView* const owningFrameView
                                        , IFocusManager* const pFocusManager,
                                          IMouseCaptureManager* const pMouseCaptureManager)
    : WebNonPluginWidgetImpl(pFocusManager, pMouseCaptureManager)
    , m_pWebFrameOwner(pWebFrameOwner)
    , m_webFrameOwnerAlive(true)
	, m_isSendingScrollEvent(false)
    , m_vScrollbarMode(WebCore::ScrollbarAuto)
    , m_hScrollbarMode(WebCore::ScrollbarAuto)
    , m_suppressScrollbars(false)
    , m_staticBackground(false)
    , m_showHScrollbar(false)
    , m_showVScrollbar(false)
    , m_owningFrameView(owningFrameView)
{
    ASSERT(m_pWebFrameOwner);
    ASSERT(m_pWebFrameOwner->frame());
    ASSERT(m_owningFrameView);
#ifndef NDEBUG
    ++WebFrameScrollViewCounter::count;
#endif
}

FrameScrollViewImpl::~FrameScrollViewImpl()
{
#ifndef NDEBUG
    --WebFrameScrollViewCounter::count;
#endif
}

void FrameScrollViewImpl::paintImpl(WebCore::GraphicsContext* pGraphicsContext, const WebCore::IntRect& widgetDirtyRect)
{
    pGraphicsContext->save();
    pGraphicsContext->clip( clientRect() );
    pGraphicsContext->clip( widgetDirtyRect );

    {
        pGraphicsContext->save();
        bool hasHScrollbar = m_hScrollbar && m_showHScrollbar;
        if (hasHScrollbar) {
            if (widgetDirtyRect.intersects( m_hScrollbar->frameGeometry()))
                m_hScrollbar->paint( pGraphicsContext, widgetDirtyRect );
        }

        bool hasVScrollbar = m_vScrollbar && m_showVScrollbar;
        if (hasVScrollbar) {
            if (widgetDirtyRect.intersects(m_vScrollbar->frameGeometry()))
                m_vScrollbar->paint(pGraphicsContext, widgetDirtyRect);
        }

        if (hasHScrollbar && hasVScrollbar) {
            WebCore::IntRect hBarRect = m_hScrollbar->frameGeometry();
            WebCore::IntRect vBarRect = m_vScrollbar->frameGeometry();
            WebCore::IntRect deadSpaceRect(hBarRect.x() + hBarRect.width(), vBarRect.y() + vBarRect.height(), vBarRect.width(), hBarRect.height());

            if (widgetDirtyRect.intersects(deadSpaceRect)) {
                pGraphicsContext->setStrokeColor(WebCore::Color(0xf6, 0xf6, 0xf6));
                pGraphicsContext->setStrokeThickness(1);
                pGraphicsContext->setFillColor(WebCore::Color(0xf6, 0xf6, 0xf6));
                pGraphicsContext->drawRect(deadSpaceRect);
            }
        }
        pGraphicsContext->restore();
    }
    WebCore::IntRect const contentsRect(0, 0, visibleWidth(), visibleHeight( ));
    if (widgetDirtyRect.intersects(contentsRect)) {
        pGraphicsContext->translate(static_cast<float>(-m_scrollOffset.width()), static_cast<float>(-m_scrollOffset.height()));

        WebCore::IntRect const clipRect(m_scrollOffset.width(), m_scrollOffset.height(), visibleWidth(), visibleHeight());
        pGraphicsContext->clip(clipRect);
        
        WebCore::IntRect const contentDirtyRect(viewportToContents(widgetDirtyRect.location()), widgetDirtyRect.size());
        
        WebCore::Frame* const theFrame = frame();
        ASSERT(theFrame);
        theFrame->paint(pGraphicsContext, contentDirtyRect);
        
        pGraphicsContext->translate(static_cast<float>(m_scrollOffset.width()), static_cast<float>(m_scrollOffset.height()));
    }
    
    pGraphicsContext->restore();
}

void FrameScrollViewImpl::setCursor(const WebCore::Cursor& cursor)
{
    WebViewImpl* webView = m_pWebFrameOwner->webViewImpl();
    ASSERT( webView );
    WebHost* webHost = webView->webHost();
    ASSERT( webHost );
    webHost->m_pVTable->setMouseCursor( webHost, cursor.impl() );
}

void FrameScrollViewImpl::show()
{
}

void FrameScrollViewImpl::hide()
{
}

bool FrameScrollViewImpl::isActive() const
{
    WebCore::Page* page = frame()->page();
    return page && page->focusController()->isActive();
}

WebCore::ScrollView* FrameScrollViewImpl::scrollView() const
{
    return frameView();
}

namespace {
    static int inline computeVisibleDimension(int const scrollBarWidth, int frameDimension, bool showScrollbar)
    {
        int const usedScrollbarWidth = showScrollbar ? scrollBarWidth : 0;
        int const result = std::max(0, frameDimension - usedScrollbarWidth);
        return result;
    }
}

void FrameScrollViewImpl::owningWebFrameIsDead()
{
    ASSERT(m_pWebFrameOwner);
    ASSERT(m_webFrameOwnerAlive);
    m_webFrameOwnerAlive = false;
}

int FrameScrollViewImpl::visibleWidth() const
{
    ASSERT(m_vScrollbar || (!m_showVScrollbar));
    ASSERT(m_hScrollbar || (!m_showHScrollbar));
    int const scrollBarWidth = m_vScrollbar ? m_vScrollbar->width() : 0;
    ASSERT(scrollBarWidth >= 0);
    return computeVisibleDimension(scrollBarWidth, clientRect().width(), m_showVScrollbar);
}

int FrameScrollViewImpl::visibleHeight() const
{
    ASSERT(m_vScrollbar || (!m_showVScrollbar));
    ASSERT(m_hScrollbar || (!m_showHScrollbar));
    int const scrollBarHeight = m_hScrollbar ? m_hScrollbar->height() : 0;
    ASSERT(scrollBarHeight >= 0);
    return computeVisibleDimension(scrollBarHeight,  clientRect().height(), m_showHScrollbar);
}

int FrameScrollViewImpl::contentsWidth() const
{
    return m_contentsSize.width();
}

int FrameScrollViewImpl::contentsHeight() const
{
    return m_contentsSize.height();
}

int FrameScrollViewImpl::contentsX() const
{
    return m_scrollOffset.width();
}

int FrameScrollViewImpl::contentsY() const
{
    return m_scrollOffset.height();
}

WebCore::IntSize FrameScrollViewImpl::scrollOffset() const
{
    return m_scrollOffset;
}

bool FrameScrollViewImpl::scrollBy(int dx, int dy)
{
    WebCore::IntSize const scrollOffset = m_scrollOffset;
    WebCore::IntSize newScrollOffset = scrollOffset + WebCore::IntSize(dx, dy);
    newScrollOffset.clampNegativeToZero();
    if ( newScrollOffset != scrollOffset && !m_isSendingScrollEvent) {
		m_isSendingScrollEvent = true;
		m_scrollOffset = newScrollOffset;
		updateScrollbars();
		WebCore::IntRect const cRect(clientRect());
		invalidateRect(cRect);
		frame()->sendScrollEvent();
		m_isSendingScrollEvent = false;
        return true;
    }

    return false;
}

void FrameScrollViewImpl::setContentsPos(int newX, int newY)
{
    int const dx = newX - contentsX();
    int const dy = newY - contentsY();
    scrollBy(dx, dy);
}

void FrameScrollViewImpl::setVScrollbarMode(WebCore::ScrollbarMode newMode)
{
    if ( m_vScrollbarMode != newMode ) {
        m_vScrollbarMode = newMode;
        updateScrollbars();
    }
}

void FrameScrollViewImpl::setHScrollbarMode(WebCore::ScrollbarMode newMode)
{
    if ( m_hScrollbarMode != newMode ) {
        m_hScrollbarMode = newMode;
        updateScrollbars();
    }
}

void FrameScrollViewImpl::suppressScrollbars(bool suppressed, bool repaintOnUnsuppress)
{
    m_suppressScrollbars = suppressed;
    if ( repaintOnUnsuppress ) {
        updateScrollbars();
    }
}

WebCore::ScrollbarMode FrameScrollViewImpl::getVScrollbarMode() const
{
    return m_vScrollbarMode;
}

WebCore::ScrollbarMode FrameScrollViewImpl::getHScrollbarMode() const
{
    return m_hScrollbarMode;
}

void FrameScrollViewImpl::resizeContents(int w, int h)
{
    WebCore::IntSize const newSize(w, h);
    if (newSize != m_contentsSize) {
        m_contentsSize = newSize;
        
        WebCore::IntSize clampedScrollOffset(scrollOffset());
        // if the scrolloffset is incorrect then set it to a valid value
        //
        if (contentsY() + visibleHeight() > contentsHeight())
        {
            clampedScrollOffset.setHeight(std::max(0, contentsHeight() - visibleHeight()));
        }
        if (contentsX() + visibleWidth() > contentsWidth())
        {
            clampedScrollOffset.setWidth(std::max(0, contentsWidth() - visibleWidth()));
        }
        setContentsPos(clampedScrollOffset.width(), clampedScrollOffset.height());
        
        updateScrollbars();
    }
}

namespace {

inline bool frameIsAllowedToScroll(const WebCore::Frame* const frame) throw()
{
    ASSERT(frame);
    const WebCore::HTMLFrameOwnerElement* const ownerElement = frame->ownerElement();
    
    // root frames are allowed to scroll.
    if (!ownerElement)
        return true;
    
    // frame's whose tags we don't recognize are allowed to scroll.
    if (!ownerElement->isFrameElementBase())
        return true;
        
    const WebCore::HTMLFrameElementBase* const frameElement =
        static_cast<const WebCore::HTMLFrameElementBase*>(ownerElement);
    
    WebCore::ScrollbarMode const scrollingMode = frameElement->scrollingMode();
    bool const allowedToScroll = scrollingMode != WebCore::ScrollbarAlwaysOff;
    return allowedToScroll;
}

}

void FrameScrollViewImpl::updateScrollbars()
{
    bool showHScrollbar = false;
    bool showVScrollbar = false;
    WebCore::IntRect const cRect(clientRect());
    
    WebCore::Frame* const theFrame = frame();
    ASSERT(theFrame);

    if ((!m_suppressScrollbars) && frameIsAllowedToScroll(theFrame)) {
        switch (m_hScrollbarMode) {
        case WebCore::ScrollbarAuto:
            showHScrollbar = m_contentsSize.width() > cRect.width();
            break;
        case WebCore::ScrollbarAlwaysOff:
            showHScrollbar = false;
            break;
        case WebCore::ScrollbarAlwaysOn:
            showHScrollbar = true;
            break;
        default:
            //unknown scroll bar mode.
            ASSERT(false);
        }

        switch (m_vScrollbarMode)
        {
        case WebCore::ScrollbarAuto:
            showVScrollbar = m_contentsSize.height() > cRect.height();
            break;
        case WebCore::ScrollbarAlwaysOff:
            showVScrollbar = false;
            break;
        case WebCore::ScrollbarAlwaysOn:
            showVScrollbar = true;
            break;
        default:
            //unknown scroll bar mode.
            ASSERT(false);
        }
    }

    //If the scroll bar state is changed we'll need a full repaint.
    if ((m_showHScrollbar != showHScrollbar) || (m_showVScrollbar != showVScrollbar))
        invalidateRect(cRect);

    m_showHScrollbar = showHScrollbar;
    m_showVScrollbar = showVScrollbar;

    if (showHScrollbar && (!m_hScrollbar)) {
        m_hScrollbar = WTF::RefPtr< WebCore::PlatformScrollbar >( new WebCore::PlatformScrollbar( this, WebCore::HorizontalScrollbar, WebCore::RegularScrollbar, theFrame ) );
        ASSERT(m_hScrollbar->hasOneRef());
        ASSERT(frameView());
        frameView()->addChild(m_hScrollbar.get());
    }

    if (showVScrollbar && (!m_vScrollbar)) {
        m_vScrollbar = WTF::RefPtr< WebCore::PlatformScrollbar >( new WebCore::PlatformScrollbar( this, WebCore::VerticalScrollbar, WebCore::RegularScrollbar, theFrame ) );
        ASSERT(m_vScrollbar->hasOneRef());
        ASSERT(frameView());
        frameView()->addChild(m_vScrollbar.get());
    }
    
    ASSERT( m_hScrollbar || (!showHScrollbar));
    int const hScrollbarHeight = m_hScrollbar ? m_hScrollbar->height() : 0;
    int const computedVisibleHeight = computeVisibleDimension(hScrollbarHeight, cRect.height(), showHScrollbar);
    ASSERT(computedVisibleHeight <= cRect.height());
    
    ASSERT( m_vScrollbar || (!showVScrollbar ));
    int const vScrollbarWidth = m_vScrollbar ? m_vScrollbar->width() : 0;
    int const computedVisibleWidth = computeVisibleDimension(vScrollbarWidth, cRect.width(), showVScrollbar);
    ASSERT(computedVisibleWidth <= cRect.width());

    int resizeBoxHAdjustment = 0;
    int resizeBoxVAdjustment = 0;    
#if PLATFORM(DARWIN)
    // On Mac we need to move the scrollbars our of the way of 
    // the grow box (resize handle) for root level content.
    //
    static const int s_macGrowBozSize = 15;
    if ((showHScrollbar && !showVScrollbar) || (showVScrollbar && !showHScrollbar)) {
        IScrollViewApolloImpl* parentScrollView = getParent();
        if (!parentScrollView){
            if (showHScrollbar){
                ASSERT(!showVScrollbar);
                resizeBoxHAdjustment = std::min(computedVisibleWidth, s_macGrowBozSize);
            }
            
            if (showVScrollbar){
                ASSERT(!showHScrollbar);
                resizeBoxVAdjustment = std::min(computedVisibleHeight, s_macGrowBozSize);
            }
        }
    }
#endif

    int const maxHorzScrollPos = showHScrollbar ? m_contentsSize.width() - computedVisibleWidth: 0;
    int const maxVertScrollPos = showVScrollbar ? m_contentsSize.height() - computedVisibleHeight: 0;

    WebCore::IntSize maximumScrollPos(maxHorzScrollPos, maxVertScrollPos);
    maximumScrollPos.clampNegativeToZero();
    m_scrollOffset = m_scrollOffset.shrunkTo(maximumScrollPos);
    m_scrollOffset.clampNegativeToZero();

    if (showHScrollbar) {   
        m_hScrollbar->setProportion(computedVisibleWidth, contentsWidth());
        m_hScrollbar->setValue(contentsX());
        WebCore::IntRect const scrollBarRect(0, computedVisibleHeight, computedVisibleWidth - resizeBoxHAdjustment, cRect.height() - computedVisibleHeight);
        m_hScrollbar->setRect(scrollBarRect);
        int pageStep = computedVisibleWidth - PAGE_KEEP;
        if (pageStep < 0)
            pageStep = computedVisibleWidth;
            
        m_hScrollbar->setSteps(LINE_STEP, pageStep);
        //The the widget part of the scroll bar where the scroll bar is. Stupid hack for now.
        m_hScrollbar->setFrameGeometry(scrollBarRect);
    }

    if (showVScrollbar) {
        m_vScrollbar->setProportion(computedVisibleHeight, contentsHeight());
        m_vScrollbar->setValue(contentsY());
        WebCore::IntRect const scrollBarRect(computedVisibleWidth, 0, cRect.width() - computedVisibleWidth, computedVisibleHeight - resizeBoxVAdjustment);
        m_vScrollbar->setRect(scrollBarRect);
        int pageStep = computedVisibleHeight - PAGE_KEEP;
        if (pageStep < 0)
            pageStep = computedVisibleHeight;
            
        m_vScrollbar->setSteps(LINE_STEP, pageStep);
        //The the widget part of the scroll bar where the scroll bar is. Stupid hack for now.
        m_vScrollbar->setFrameGeometry(scrollBarRect);
    }
}



void FrameScrollViewImpl::scrollbarIsDirty(WebCore::Scrollbar* scrollbar)
{
    if (m_hScrollbar == scrollbar) {
        invalidateRect(m_hScrollbar->frameGeometry());
    }
    if (m_vScrollbar == scrollbar) {
        invalidateRect(m_vScrollbar->frameGeometry());
    }
}

void FrameScrollViewImpl::valueChanged( WebCore::Scrollbar* pChangedScrollbar )
{
    ASSERT(pChangedScrollbar);
    ASSERT(m_vScrollbar != m_hScrollbar);
    ASSERT((m_vScrollbar == pChangedScrollbar) || (m_hScrollbar == pChangedScrollbar));
    if (pChangedScrollbar == m_vScrollbar)
        setContentsPos(contentsX(), pChangedScrollbar->value());
    else
    {
        ASSERT(pChangedScrollbar == m_hScrollbar);
        setContentsPos(pChangedScrollbar->value(), contentsY());
    }
}

void FrameScrollViewImpl::updateContents(const WebCore::IntRect& contentDamageRect, bool)
{
    WebCore::IntRect const widgetDamageRect(contentsToViewport(contentDamageRect.location() ), contentDamageRect.size());
    invalidateRect(widgetDamageRect);
}

WebCore::IntPoint FrameScrollViewImpl::contentsToViewport(const WebCore::IntPoint& contentsPoint)
{
    return contentsPoint - m_scrollOffset;
}

WebCore::IntPoint FrameScrollViewImpl::viewportToContents(const WebCore::IntPoint& viewportPoint)
{
    return viewportPoint + m_scrollOffset;
}

void FrameScrollViewImpl::setStaticBackground(bool bIsStatibBackground)
{
    m_staticBackground = bIsStatibBackground;
}

WebPlatformWindow FrameScrollViewImpl::getPlatformWindow()
{
    return getWindow()->m_pVTable->getPlatformWindow(getWindow());
}

bool FrameScrollViewImpl::inWindow() const
{
    return true;
}

WTF::PassRefPtr<WebCore::IPluginWidgetApolloImpl> FrameScrollViewImpl::toPluginWidgetImpl()
{
    ASSERT(false);
    return WTF::PassRefPtr<WebCore::IPluginWidgetApolloImpl>();
}

bool FrameScrollViewImpl::isInVisibleViewHierarchy() const
{
    return WebNonPluginWidgetImpl::isInVisibleViewHierarchy();
}

WebCore::Widget* FrameScrollViewImpl::getWidget()
{
    return frame()->view();
}

WebCore::Frame* FrameScrollViewImpl::frame() const
{
    ASSERT(m_pWebFrameOwner);
    return m_pWebFrameOwner->frame();
}

WebCore::FrameView* FrameScrollViewImpl::frameView() const
{
    ASSERT(m_owningFrameView);
    return m_owningFrameView;
}

void FrameScrollViewImpl::setClient(WebCore::WidgetClient* pClient)
{
    WebNonPluginWidgetImpl::setClient(pClient);
}

WebCore::WidgetClient* FrameScrollViewImpl::client() const
{
    return WebNonPluginWidgetImpl::client();
}

void FrameScrollViewImpl::setFocus()
{
    WebNonPluginWidgetImpl::setFocus();
}

void FrameScrollViewImpl::invalidateRect(const WebCore::IntRect& rect)
{
    if(getParent())
        WebNonPluginWidgetImpl::invalidateRect(rect);
}

WebCore::IntRect FrameScrollViewImpl::frameGeometry() const
{
     return WebNonPluginWidgetImpl::frameGeometry();
}

bool FrameScrollViewImpl::hasMouseCapture() const
{
    return WebNonPluginWidgetImpl::hasMouseCapture();
}

void FrameScrollViewImpl::setMouseCapture()
{
    WebNonPluginWidgetImpl::setMouseCapture();
}

void FrameScrollViewImpl::releaseMouseCapture()
{
    WebNonPluginWidgetImpl::releaseMouseCapture();
}

PassRefPtr<KJS::Bindings::Instance> FrameScrollViewImpl::getInstance(WebCore::Frame* frame)
{
    return WebNonPluginWidgetImpl::getInstance(frame);
}

void* FrameScrollViewImpl::createPluginScriptableObject()
{
    return WebNonPluginWidgetImpl::createPluginScriptableObject();
}

void FrameScrollViewImpl::releasePluginScriptableObject(void * obj)
{
    WebNonPluginWidgetImpl::releasePluginScriptableObject(obj);
}

void FrameScrollViewImpl::setFrameGeometry(const WebCore::IntRect &rect)
{
    WebNonPluginWidgetImpl::setFrameGeometry(rect);
}

void FrameScrollViewImpl::setEnabled(bool bEnabled)
{
    WebNonPluginWidgetImpl::setEnabled(bEnabled);
}

bool FrameScrollViewImpl::isEnabled() const
{
    return WebNonPluginWidgetImpl::isEnabled();
}

void FrameScrollViewImpl::setIsSelected(bool bSelected)
{
    WebNonPluginWidgetImpl::setIsSelected(bSelected);
}

WebCore::Cursor FrameScrollViewImpl::cursor()
{
    return WebNonPluginWidgetImpl::cursor();
}

static inline bool isChildFrame(WebCore::Frame* const parent, WebCore::Frame* const child)
{
    bool result = false;
    ASSERT(parent);
    ASSERT(child);
    ASSERT(parent->tree());
    WebCore::Frame* currChild = parent->tree()->firstChild();
    while ((!result) && currChild) {
        result = currChild == child;
        ASSERT(currChild->tree());
        currChild = currChild->tree()->nextSibling();
    }
    return result;
}

WebCore::IScrollViewApolloImpl* FrameScrollViewImpl::getParent()
{
    WebCore::IScrollViewApolloImpl* const result = WebNonPluginWidgetImpl::getParent();
    
    // If the owning webframe is dead, then we should not have a parent widget.
    ASSERT((isOwningWebFrameAlive()) || ((!isOwningWebFrameAlive()) && (!result)));
    
    // For the result of these assert's well short circuit out if
    // the owning WebFrame is dead.
    //
    // We should have a frame.
    ASSERT((!isOwningWebFrameAlive()) || (frame()));
    
    // We should have a frame tree.
    ASSERT((!isOwningWebFrameAlive()) || (frame()->tree()));
    
    // If we have non-null parent frame, the parent frame should have a non-null view.
    ASSERT((!isOwningWebFrameAlive()) || (!frame()->tree()->parent()) || (frame()->tree()->parent()->view()));
    
    // If we have a non-null parent frame, make sure its view is the parent view of this scroll view.
    ASSERT(  (!isOwningWebFrameAlive())
          || (!frame()->tree()->parent())
          // Sometimes our parent view is null even though we in the frame tree.
          || (!result)
          // since frame's are constructed with their parent pointer initialized
          // it is possible that we are not a child of our parent yet.  We should be
          // shortly though.
          || (!isChildFrame(frame()->tree()->parent(), frame()))
          || (result && (frame()->tree()->parent()->view()->getApolloImpl() == result)));
    return result;
}

WebCore::PlatformMouseEvent FrameScrollViewImpl::transformMouseEvent(const WebCore::IntPoint& newOrigin, const WebCore::PlatformMouseEvent& mouseEvent)
{
    WebCore::IntSize const offset(newOrigin.x(), newOrigin.y());
    WebCore::IntPoint const transformedPoint(mouseEvent.pos() - offset);
    return WebCore::PlatformMouseEvent( transformedPoint,
                                        WebCore::IntPoint( mouseEvent.globalX(), mouseEvent.globalY() ),
                                        mouseEvent.button(),
                                        mouseEvent.eventType(),
                                        mouseEvent.clickCount(),
                                        mouseEvent.shiftKey(),
                                        mouseEvent.ctrlKey(),
                                        mouseEvent.altKey(),
                                        mouseEvent.metaKey(),
                                        mouseEvent.timestamp() );
}

FrameScrollViewImpl::HitTestResult FrameScrollViewImpl::hitTest(const WebCore::IntPoint& point)
{
    FrameScrollViewImpl::HitTestResult result = kHitTestResultContent;
    if ( m_showHScrollbar )
    {
        if ( m_hScrollbar->frameGeometry().contains(point))
        {
            result = kHitTestResultHScrollbar;
        }
    }
    if ( m_showVScrollbar )
    {
        if ( m_vScrollbar->frameGeometry().contains(point))
        {
            result = kHitTestResultVScrollbar;
        }
    }
    return result;
}

bool FrameScrollViewImpl::handleMousePressEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    FrameScrollViewImpl::HitTestResult const hitTestResult = hitTest(mouseEvent.pos());
    switch ( hitTestResult ) {
    case kHitTestResultContent:
        return m_pWebFrameOwner->handleMousePressEvent(mouseEvent);
        break;
    case kHitTestResultHScrollbar:
        if (m_hScrollbar) {
            return m_hScrollbar->handleMousePressEventNormalized(mouseEvent);
        }
        break;
    case kHitTestResultVScrollbar:
        if (m_vScrollbar) {
            return m_vScrollbar->handleMousePressEventNormalized(mouseEvent);
        }
        break;
    default:
        // Unknown hit test result
        ASSERT(false);
    }

    return false;
}

bool FrameScrollViewImpl::handleMouseMoveEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    FrameScrollViewImpl::HitTestResult const hitTestResult = hitTest(mouseEvent.pos());


    switch (hitTestResult) {
    case kHitTestResultContent:
        return m_pWebFrameOwner->handleMouseMoveEvent(mouseEvent);
        break;
    case kHitTestResultHScrollbar:
        if (m_hScrollbar) {
            bool bHandled = m_hScrollbar->handleMouseMoveEventNormalized( mouseEvent );
            invalidateRect(m_hScrollbar->frameGeometry());
            return bHandled;
        }
        break;
    case kHitTestResultVScrollbar:
        if (m_vScrollbar) {
            bool bHandled = m_vScrollbar->handleMouseMoveEventNormalized( mouseEvent );
            invalidateRect(m_vScrollbar->frameGeometry());
            return bHandled;
        }
        break;
    default:
        // Unknown hit test result
        ASSERT(false);
    }

    return false;
}

bool FrameScrollViewImpl::handleMouseReleaseEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    FrameScrollViewImpl::HitTestResult const hitTestResult = hitTest(mouseEvent.pos());

    //Since we don't do proper mouse capture, we need to send all mouse release
    //events to any scroll bars we have.
    bool bHandled = false;

    if (m_hScrollbar) {
        bHandled = m_hScrollbar->handleMouseReleaseEventNormalized(mouseEvent);
    }
    if (m_vScrollbar){
        bHandled = m_vScrollbar->handleMouseReleaseEventNormalized(mouseEvent);
    }

    switch (hitTestResult) {
    case kHitTestResultContent:
        bHandled |= m_pWebFrameOwner->handleMouseReleaseEvent(mouseEvent);
        break;
    case kHitTestResultHScrollbar:
    case kHitTestResultVScrollbar:
        break;
    default:
        // Unknown hit test result
        ASSERT(false);
    }

    return bHandled;
}

bool FrameScrollViewImpl::handleWheelEvent(WebCore::PlatformWheelEvent& wheelEvent)
{
    return m_pWebFrameOwner->handleWheelEvent(wheelEvent);
}

bool FrameScrollViewImpl::handleKeyboardEvent(const WebCore::PlatformKeyboardEvent& keyEvent)
{
    return m_pWebFrameOwner->handleKeyboardEvent(keyEvent);
}

bool FrameScrollViewImpl::handleInsertText(const WebCore::String& text)
{
    return m_pWebFrameOwner->handleInsertText(text);
}

bool FrameScrollViewImpl::handleCut()
{
    return m_pWebFrameOwner->handleCut();
}

bool FrameScrollViewImpl::handleCopy()
{
    return m_pWebFrameOwner->handleCopy();
}

bool FrameScrollViewImpl::handlePaste()
{
    return m_pWebFrameOwner->handlePaste();
}

bool FrameScrollViewImpl::handleSelectAll()
{
    return m_pWebFrameOwner->handleSelectAll();
}

WebCore::PlatformScrollbar* FrameScrollViewImpl::scrollbarUnderMouse(const WebCore::PlatformMouseEvent& mouseEvent)
{
    FrameScrollViewImpl::HitTestResult const hitTestResult = hitTest(mouseEvent.pos());
    WebCore::PlatformScrollbar* result = 0;
    switch (hitTestResult) {
    case kHitTestResultContent:
        break;
    case kHitTestResultHScrollbar:
        result = m_hScrollbar.get();
    case kHitTestResultVScrollbar:
        result = m_vScrollbar.get();
        break;
    default:
        // Unknown hit test result
        ASSERT(false);
    }
    return result;
}

WebCore::PlatformScrollbar* FrameScrollViewImpl::getVScrollbar()
{
    return m_vScrollbar.get();
}

WebCore::PlatformScrollbar* FrameScrollViewImpl::getHScrollbar()
{
    return m_hScrollbar.get();
}

::WebWindow *FrameScrollViewImpl::getWindow()
{
    WebWindow *result = NULL;
    
    ASSERT(m_pWebFrameOwner);
    if(m_pWebFrameOwner != NULL)
    {
        WebCore::Frame *frame = m_pWebFrameOwner->frame();
        ASSERT(frame != NULL);
        if(frame != NULL)
        {
            WebCore::Frame *parent = frame->tree()->parent();
            ASSERT(parent != NULL);
            if(parent != NULL)
            {
                WebCore::FrameView* view = parent->view();
                WTF::PassRefPtr<IScrollViewApolloImpl> impl = view->getApolloImpl();
                if(impl)
                {
                    result = impl->getWindow();
                }
            }
        }
    }
    
    return result;
}

void FrameScrollViewImpl::setParent(WebCore::IScrollViewApolloImpl* parent)
{
    ASSERT(frame());
    
    ASSERT((!getParent()) || (parent != getParent()->scrollView()->getApolloImpl()));
    ASSERT((!getParent()) || (parent != getParent()->scrollView()->getApolloImpl()));
    // Only three legal cases:
    // 1. Setting parent to null when we already have a null parent results in NOP.
    // 2. Setting null parent to non-null parent.  Make sure we are already in the new parent's children set.
    // 3. Setting parent to null.  Make sure we are no longer in our parent's children set.
    ASSERT(((!getParent()) && (!parent))
        || ((!parent) && (getParent()) && (!(getParent()->scrollView()->children()->contains(frameView()))))
        || ((parent) && (!getParent()) && ((parent->scrollView()->children()->contains(frameView())))));
    
    WebNonPluginWidgetImpl::setParent(parent);
    // If we have a parent we must be in its children set.
    ASSERT((!getParent()) || (getParent()->scrollView()->children()->contains(frameView())));
}

void FrameScrollViewImpl::onFocus()
{
    WebCore::Frame* const theFrame = frame();
    ASSERT(theFrame);
    WebCore::Page* const page = theFrame->page();
    ASSERT(page);
    WebCore::FocusController* const focusController = page->focusController();
    ASSERT(focusController);
    // A call to setFocusedFrame may currently be on the stack.
    // we'll explicitly guard against further recursion here, even though
    // setFocusedFrame already does so.
    WebCore::Frame* const existingFocusedFrame = focusController->focusedFrame();
    if (existingFocusedFrame != theFrame)
        focusController->setFocusedFrame(theFrame);
}

void FrameScrollViewImpl::onBlur()
{
    // Don't need to do anything here because this should 
}

bool FrameScrollViewImpl::widgetNeedsMouseDownFromEventHandler() const
{
    // should only be called on plugin widgets.
    ASSERT(false);
    return WebNonPluginWidgetImpl::widgetNeedsMouseDownFromEventHandler();
}


SubFrameScrollViewImpl::SubFrameScrollViewImpl( WebFrameImpl* const pWebFrameOwner
                                              , WebCore::FrameView* const owningFrameView
                                              , IFocusManager* const pFocusManager
                                              , IMouseCaptureManager* const pMouseCaptureManager)
    : FrameScrollViewImpl( pWebFrameOwner
                         , owningFrameView
                         , pFocusManager
                         , pMouseCaptureManager )
{

}

SubFrameScrollViewImpl::~SubFrameScrollViewImpl()
{
    ASSERT(m_pWebFrameOwner);
}

WTF::PassRefPtr<SubFrameScrollViewImpl>
    SubFrameScrollViewImpl::construct( WebFrameImpl* const pWebFrameOwner
                                     , WebCore::FrameView* const owningFrameView
                                     , IFocusManager* const pFocusManager
                                     , IMouseCaptureManager* const pMouseCaptureManager)
{
    return adoptRef(new SubFrameScrollViewImpl( pWebFrameOwner
                                              , owningFrameView
                                              , pFocusManager
                                              , pMouseCaptureManager));
}

void SubFrameScrollViewImpl::scrollRectIntoViewRecursively(const WebCore::IntRect& rect)
{
    scrollRectIntoView(rect);
    if (getParent()) {
        WebCore::IntPoint rectLoc(rect.location());
        WebCore::IntPoint rectLocInParent(contentsToViewport(rectLoc));
        WebCore::IntRect rectInParent(rectLocInParent, rect.size());
        ASSERT(getParent());
        getParent()->scrollRectIntoViewRecursively(rectInParent);
    }
}

WebCore::IntPoint SubFrameScrollViewImpl::windowToViewport( const WebCore::IntPoint& windowPoint )
{
    ASSERT(getParent());
    WebCore::IntPoint viewportPoint = getParent()->windowToViewport( windowPoint );
    WebCore::IntRect const geometry(frameGeometry());
    viewportPoint.move( -geometry.x(), -geometry.y() );
    return viewportPoint;
}

WebCore::IntPoint SubFrameScrollViewImpl::viewportToWindow(const WebCore::IntPoint& viewportPoint, bool htmlWindowCoords )
{
    WebCore::IntPoint point = viewportPoint; // accounts for immediate enclosing frame scroll
    WebCore::IntRect const geometry(frameGeometry());
    point.move(geometry.x(), geometry.y());     // downgrading point to geometry point in parent frame
    WebCore::IScrollViewApolloImpl* const parentScrollView = getParent();
    ASSERT(parentScrollView);
    point -= parentScrollView->scrollOffset();  // now it's a viewportPoint point in parent frame

    WebCore::IntPoint windowPoint = parentScrollView->viewportToWindow( point, htmlWindowCoords );

    return windowPoint;
}

WebCore::IntRect SubFrameScrollViewImpl::clientRect() const
{
    WebCore::IntRect const geometry(frameGeometry());
    return WebCore::IntRect( WebCore::IntPoint( 0, 0 ), geometry.size() );
}

void SubFrameScrollViewImpl::setFrameGeometry(const WebCore::IntRect& newGeometry)
{
    WebCore::IntRect const oldGeometry = frameGeometry();
    FrameScrollViewImpl::setFrameGeometry(newGeometry);
    
    if (oldGeometry != newGeometry) {
        if ((oldGeometry.width() != newGeometry.width()) || (oldGeometry.height() != newGeometry.height())) {
            WebCore::FrameView* const fv = frameView();
            ASSERT(fv);
            fv->layout();  // updates the content width and height for this frame.
            updateScrollbars();
        }
    }
}

void SubFrameScrollViewImpl::paint( WebCore::GraphicsContext* pGraphicsContext,
                                    const WebCore::IntRect& parentFrameDirtyRect )
{
    WebCore::IntPoint const widgetDirtyRectOrigin(  parentFrameDirtyRect.location().x() - frameGeometry().location().x(),
                                                    parentFrameDirtyRect.location().y() - frameGeometry().location().y() );
    WebCore::IntRect const widgetDirtyRect( widgetDirtyRectOrigin, parentFrameDirtyRect.size() );

    WebCore::FloatSize const translateOffset( static_cast<float>(frameGeometry().location().x()), static_cast<float>(frameGeometry().location().y()) );
    pGraphicsContext->translate( translateOffset.width(), translateOffset.height() );
    paintImpl( pGraphicsContext, widgetDirtyRect );
    pGraphicsContext->translate( -translateOffset.width(), -translateOffset.height() );
}

// the meaning of this now is to *actually* be in window coords
WebCore::IntRect SubFrameScrollViewImpl::windowClipRect() const
{
    WebCore::IntRect rect = clientRect(); // in viewport points
    WebCore::IntPoint p = rect.location();

    p = const_cast<SubFrameScrollViewImpl*>(this)->viewportToWindow( p );

    rect.setLocation( p );
    
    return rect;
}

void SubFrameScrollViewImpl::setParent(IScrollViewApolloImpl* parentScrollView)
{
    ASSERT((!getParent()) || (parentScrollView != getParent()->scrollView()->getApolloImpl()));
    if ((!parentScrollView) && (getParent())) {
        // Frame is being removed from the widget hierarchy.

        WebCore::Frame* const coreFrame = frame();
        ASSERT(coreFrame);
        WebCore::Page* const page = coreFrame->page();
        if (page) {
            ASSERT(page);
            WebCore::FocusController* const focusController = page->focusController();
            ASSERT(focusController);
            if (focusController->focusedFrame() == coreFrame)
                focusController->setFocusedFrame(0);
        }
    }
    FrameScrollViewImpl::setParent(parentScrollView);
}

}
