/*
 * Copyright (C) 2005, 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer. 
 * 2.  Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WebFrameImpl.h"
#include <IScrollViewApolloImpl.h>
#include <FrameView.h>
#include <EventHandler.h>
#include <HTMLFormElement.h>
#include "WebWindowScrollView.h"
#include "WebResourceRequestImpl.h"
#include "WebResourceHandleClientImpl.h"
#include "WebScrollbarWidgetImpl.h"
#include "WebViewImpl.h"
#include "WebPluginImpl.h"
#include "WebPluginWidget.h"

#include <WebKitApollo/WebHost.h>
#include <WebKitApollo/helpers/WebHostHelper.h>
#include <Page.h>
#include <ResourceError.h>
#include <DocumentLoader.h>
#include <FrameTree.h>
#include <HistoryItem.h>
#include <apollo/proxy_obj.h>
#include <kjs_proxy.h>
#include <kjs_binding.h>
#include "runtime_root.h"
#include <ClipboardApollo.h>
#include <ClipboardApolloHelper.h>
#include <Document.h>
#include <DocumentFragment.h>
#include <Editor.h>
#include <CharacterData.h>
#include <EditorClient.h>
#include <markup.h>
#include <ReplaceSelectionCommand.h>
#include <Pasteboard.h>
#include "ResourceHandleInternal.h"
#include "MIMETypeRegistry.h"
#include "WebPopupWindowClientImpl.h"
#include <HTMLFrameOwnerElement.h>
#include <EventNames.h>


#if PLATFORM(APOLLO)

#if PLATFORM(DARWIN)
#include "mac/WebNPPluginWidgetImplMac.h"
#define PLUGINS_ENABLED 1
#endif

#if PLATFORM(WIN_OS)
#include "win/WidgetApolloPluginWin.h"
#define PLUGINS_ENABLED 1
#endif

#if PLATFORM(UNIX_OS)
#include "unix/WidgetApolloPluginUnix.h"
#define PLUGINS_ENABLED 1
#endif
#endif // PLATFORM(APOLLO)

#ifdef PLUGINS_ENABLED
#include "PluginView.h"	
#include "PluginDatabase.h"
#include "PluginInfoStore.h"
#endif

#include <RenderPart.h>
#include "FrameLoaderTypes.h"
#include "IPluginWidgetApolloImpl.h"
#include "WebEditorClient.h"
#include <Logging.h>

namespace WebKitApollo {

extern WebKitAPIHostFunctions* g_HostFunctions;

namespace Private {
class WebHostImplForFrames : public WebHostHelper<WebHostImplForFrames> {
public:
    WebHostImplForFrames(::WebHost* const pParentWebHost)
        : m_pParentWebHost( pParentWebHost )
    {
    }
    
    bool isUnusable()
    {
        return false;
    }

    void updateLocation(unsigned const char* urlBytes)
    {
    }

    void loadEnd(unsigned char success)
    {
    }

    void setTitle(const uint16_t* const pUTF16Title, unsigned long const numTitleCodeUnits)
    {
    }

    void setStatusText(const uint16_t* const pUTF16Status, unsigned long const numStatusCodeUnits)
    {
    }

    const uint16_t* getUserAgentText(unsigned long *pNumUserAgentCodeUnits )
    {
        return m_pParentWebHost->m_pVTable->getUserAgentText( m_pParentWebHost, pNumUserAgentCodeUnits );
    }

    void uncaughtJSException(struct WebScriptProxyVariant* const pExceptionVariant)
    {
        m_pParentWebHost->m_pVTable->uncaughtJSException( m_pParentWebHost, pExceptionVariant );
    }
    
    void runJSAlert(const char* pUTF8Message)
    {
        m_pParentWebHost->m_pVTable->runJSAlert(m_pParentWebHost, pUTF8Message);
    }
    
    unsigned char runJSConfirm(const char* pUTF8Message)
    {
        return m_pParentWebHost->m_pVTable->runJSConfirm(m_pParentWebHost, pUTF8Message);
    }

    unsigned char runJSPrompt(const char* pUTF8Prompt, const char* pUTF8DefaultValue, char** ppUTF8Result)
    {
        return m_pParentWebHost->m_pVTable->runJSPrompt(m_pParentWebHost, pUTF8Prompt, pUTF8DefaultValue, ppUTF8Result);
    }

    WebResourceHandle* loadResource(struct WebResourceRequest* pURLRequest, struct WebResourceHandleClient* pClient)
    {
        return m_pParentWebHost->m_pVTable->loadResource(m_pParentWebHost, pURLRequest, pClient);
    }

    void loadResourceSynchronously(struct WebResourceRequest* pURLRequest, struct WebResourceHandleClient* pClient)
    {
        m_pParentWebHost->m_pVTable->loadResourceSynchronously(m_pParentWebHost, pURLRequest, pClient);
    }

    void handleOnLoadEvents()
    {
    }

    void handleOnDocumentCreated()
    {
    }

    WebView* createNewWindow(struct WebHostCreateWindowArgs* windowArgs)
    {
        return m_pParentWebHost->m_pVTable->createNewWindow(m_pParentWebHost, windowArgs);
    }

    void closeWindow()
    {
        return m_pParentWebHost->m_pVTable->closeWindow(m_pParentWebHost);
    }

    void beginModal()
    {
        return m_pParentWebHost->m_pVTable->beginModal(m_pParentWebHost);
    }

    void endModal()
    {
        return m_pParentWebHost->m_pVTable->endModal(m_pParentWebHost);
    }

    int pdfCapability()
    {
        return m_pParentWebHost->m_pVTable->pdfCapability( m_pParentWebHost );
    }
    
    int loadPDFPlugin( void **pPDFPluginObject )
    {
        return m_pParentWebHost->m_pVTable->loadPDFPlugin( m_pParentWebHost, pPDFPluginObject );
    }
    uint16_t* openFileChooser(unsigned long* pFilePathLength)
    {
        return m_pParentWebHost->m_pVTable->openFileChooser(m_pParentWebHost, pFilePathLength);
    }

    int unloadPDFPlugin( void *pdfPluginObject )
    {
        return m_pParentWebHost->m_pVTable->unloadPDFPlugin( m_pParentWebHost, pdfPluginObject );
    }

    void handlePDFError( int pdfErrorNum )
    {
        m_pParentWebHost->m_pVTable->handlePDFError( m_pParentWebHost, pdfErrorNum );
    }

    WebScriptProxyVariant* getEventObject() const
    {
        return m_pParentWebHost->m_pVTable->getEventObject( m_pParentWebHost );
    }

    WebScriptProxyVariant* getRootPackage() const
    {
        return m_pParentWebHost->m_pVTable->getRootPackage( m_pParentWebHost );
    }

    WebScriptProxyVariant* getWKMethods() const
    {
        return m_pParentWebHost->m_pVTable->getWKMethods( m_pParentWebHost );
    }

    WebScriptProxyVariant* makeDoor(WebScriptProxyVariant* pVariant) const
    {
        return m_pParentWebHost->m_pVTable->makeDoor( m_pParentWebHost, pVariant );
    }

    WebScriptProxyVariant* getHtmlControl() const
    {
        // no htmlcontrol for subframes!
        return 0;
    }

    void setMouseCursor( WebCursorType cursorType )
    {
        m_pParentWebHost->m_pVTable->setMouseCursor( m_pParentWebHost, cursorType );
    }

    void setMouseCapture( )
    {
        m_pParentWebHost->m_pVTable->setMouseCapture( m_pParentWebHost );
    }

    void releaseMouseCapture( )
    {
        m_pParentWebHost->m_pVTable->releaseMouseCapture( m_pParentWebHost );
    }

    void compositionSelectionChanged(int start, int end)
    {
        m_pParentWebHost->m_pVTable->compositionSelectionChanged(m_pParentWebHost, start, end);
    }

    void compositionAbandoned()
    {
        m_pParentWebHost->m_pVTable->compositionAbandoned(m_pParentWebHost);
    }

    void setInputMethodState(bool enable)
    {
        m_pParentWebHost->m_pVTable->setInputMethodState(m_pParentWebHost, enable);
    }

    void activateIMEForPlugin()
    {
        m_pParentWebHost->m_pVTable->activateIMEForPlugin(m_pParentWebHost);
    }

    void deactivateIMEForPlugin()
    {
        m_pParentWebHost->m_pVTable->deactivateIMEForPlugin(m_pParentWebHost);
    }

    void selectionChanged()
    {
        m_pParentWebHost->m_pVTable->selectionChanged(m_pParentWebHost);
    }

    const uint16_t* getAppResourcePath( unsigned long* pAppResourcePathLength )
    {
        return m_pParentWebHost->m_pVTable->getAppResourcePath( m_pParentWebHost, pAppResourcePathLength  );
    }

    const uint16_t* getAppStoragePath( unsigned long* pAppStoragePathLength )
    {
        return m_pParentWebHost->m_pVTable->getAppStoragePath( m_pParentWebHost, pAppStoragePathLength  );
    }

    WebPopupWindow* createPopupWindow( WebPopupWindowClient* popupWindowClient, float computedFontSize )
    {
        return m_pParentWebHost->m_pVTable->createPopupWindow( m_pParentWebHost, popupWindowClient, computedFontSize );
    }

    WebBitmap* createBitmap( unsigned long width, unsigned long height )
    {
        return m_pParentWebHost->m_pVTable->createBitmap( m_pParentWebHost, width, height );
    }

    WebBitmap* createBitmapFromBitmapDataObject( WebScriptProxyVariant* bitmapDataVariant )
    {
        return m_pParentWebHost->m_pVTable->createBitmapFromBitmapDataObject( m_pParentWebHost, bitmapDataVariant );
    }
    
    WebNavigationPolicyAction dispatchDecidePolicyForNavigationAction(WebString*, WebNavigationType, WebResourceRequest*)
    {
        return WebNavigationPolicyActionUse;
    }
    
    WebNavigationPolicyAction dispatchDecidePolicyForNewWindowAction(unsigned const char*, unsigned long, WebNavigationType, WebResourceRequest*, const uint16_t*, unsigned long)
    {
        return WebNavigationPolicyActionUse;
    }

    // Subframes currently have a null interpreter context
    void* getPlatformInterpreterContext() const { return 0; }

    const uint16_t* getLanguage( int* pCharLength )
    {
        return m_pParentWebHost->m_pVTable->getLanguage( m_pParentWebHost, pCharLength );
    }

    void canShowPlugins(bool* canShowPlugins, bool* canShowWindowedPlugins)
    {
        return m_pParentWebHost->m_pVTable->canShowPlugins(m_pParentWebHost, canShowPlugins, canShowWindowedPlugins);
    }


private:
    ::WebHost* const m_pParentWebHost;
};
}
namespace {
#ifndef NDEBUG
WTFLogChannel LogWebFrameLeaks =  { 0x00000000, "", WTFLogChannelOn };

struct WebFrameCounter { 
    static int count; 
    ~WebFrameCounter() 
    { 
        if (count)
            LOG(WebFrameLeaks, "LEAK: %d WebFrame\n", count);
    }
};
int WebFrameCounter::count = 0;
static WebFrameCounter frameCounter;
#endif
}

WebFrameImpl::WebFrameImpl(WebViewImpl* const pOwningWebView, WebHost* const pWebHost, WebCore::Page* const pPage)
    : m_pOwningWebView(pOwningWebView)
    , m_pWebHost(pWebHost)
    , m_pFrame(new WebCore::Frame(pPage, 0, this)) // just a raw pointer, no strong ref.
    , m_dead(false)
    , m_canTransformToRootContent(true)
    , m_resourceHandleClientRedirect(0)
    , m_didRedirectDataToPlugin(false)
    , m_scrollViewImpl(0)
    , m_firstLayoutForFrameView(true)
{
    // Frame was constructed with ref count of one
    // because page has a reference to it.
    // ASSERT that the page's ref is the only ref so far.
    // ASSERT that the page in fact thinks our frame
    // is the main frame.
    ASSERT(m_pFrame->hasOneRef());
    ASSERT(pPage->mainFrame() == m_pFrame);
    
    ASSERT(m_pOwningWebView);
#ifndef NDEBUG
    ++WebFrameCounter::count;
#endif
}

WebFrameImpl::WebFrameImpl(WebViewImpl* const pOwningWebView, WebHost* const pWebHost, WebCore::HTMLFrameOwnerElement* pOwnerElement, WebFrameImpl* const pParentWebFrame)
    : m_pOwningWebView(pOwningWebView)
    , m_pWebHost(pWebHost)
    , m_pFrame(new WebCore::Frame(pParentWebFrame->frame()->page(), pOwnerElement, this)) // just a raw pointer, no strong ref.
    , m_dead(false)
    , m_canTransformToRootContent(false)
    , m_resourceHandleClientRedirect(0)
    , m_didRedirectDataToPlugin(false)
    , m_scrollViewImpl(0)
    , m_firstLayoutForFrameView(true)
{
    // Frame was constructed with ref count of one
    // because that is the way webcore has done it for a while.
    // No body else has a ref to the frame yet, so
    // we can't dec the refcount just yet.
    // The refcount will remain off by one until we call WebCore::Frame::init from createFrame ( which is what calls this constructor ).
    // After calling WebCore::Frame::init, createFrame will assert that the refcount on the WebCore::Frame is no longer 1, and then
    // dec the refcount to resume refcount sanity.
    ASSERT(m_pFrame->hasOneRef());
    // This frame should not be the main frame.
    ASSERT(pParentWebFrame->frame()->page()->mainFrame() != m_pFrame);
#ifndef NDEBUG
    ++WebFrameCounter::count;
#endif
}

WebFrameImpl::~WebFrameImpl()
{
    // Just so you know, the WebViewImpl that m_pOwningWebView points to may be destroyed
    // by the time we get to this point.  This is because of the whole WebCore::Frame life support
    // business.  I don't set m_pOwningWebView to 0 from the WebViewImpl's destructor because
    // I'd prefer that m_pOwningWebView be const.
    m_dead = true;
    
    // Tell our scrollview impl that we are dead now.
    if (m_scrollViewImpl)
        m_scrollViewImpl->owningWebFrameIsDead();

#ifndef NDEBUG
    --WebFrameCounter::count;
#endif
}

WebFrameImpl*
    WebFrameImpl::construct( WebViewImpl* const pOwningWebView
                           , WebHost* const pWebHost
                           , WebCore::Page* const pPage)
{
    return new WebFrameImpl(pOwningWebView, pWebHost, pPage);
}

void WebFrameImpl::transformToRootContent(WebWindow* const pWebWindow )
{
    ASSERT(m_canTransformToRootContent );
    ASSERT(frame());
    ASSERT(frame()->view());
    ASSERT(pWebWindow);
    ASSERT(pWebWindow->m_pVTable);
    m_canTransformToRootContent = false;
    WTF::RefPtr<WebCore::IScrollViewApolloImpl> const scrollViewImpl(frame()->view()->getApolloImpl());
    WindowScrollViewImpl* windowScrollViewImpl(static_cast<WindowScrollViewImpl*>(scrollViewImpl.get()));
    windowScrollViewImpl->setHostDrawsScrollbars(false);
    
}

void WebFrameImpl::layoutRecursive()
{
    forceLayout();
    WebCore::Frame* const coreFrame = frame();
    ASSERT(coreFrame);
    ASSERT(coreFrame->tree());
    WebCore::Frame* currChildFrame = coreFrame->tree()->firstChild();
    while (currChildFrame) {
        kit(currChildFrame)->layoutRecursive();
        currChildFrame = currChildFrame->tree()->nextSibling();
    }
}

bool WebFrameImpl::handleMousePressEvent( const WebCore::PlatformMouseEvent& mouseEvent )
{
    ASSERT(!m_dead);
    return m_pFrame->eventHandler()->handleMousePressEvent( mouseEvent );
}

bool WebFrameImpl::handleWheelEvent( WebCore::PlatformWheelEvent& wheelEvent )
{
    ASSERT(!m_dead);
    return m_pFrame->eventHandler()->handleWheelEvent( wheelEvent );
}

bool WebFrameImpl::handleMouseMoveEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    ASSERT(!m_dead);
    return m_pFrame->eventHandler()->mouseMoved( mouseEvent );
}

bool WebFrameImpl::handleMouseReleaseEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    ASSERT(!m_dead);
    return m_pFrame->eventHandler()->handleMouseReleaseEvent( mouseEvent );
}

bool WebFrameImpl::handleKeyboardEvent( const WebCore::PlatformKeyboardEvent& keyEvent )
{
    ASSERT(!m_dead);
    return m_pFrame->eventHandler()->keyEvent(keyEvent);
}

bool WebFrameImpl::handleInsertText(const WebCore::String& text)
{
    ASSERT(!m_dead);
    bool handled = true;
    if (!m_pFrame->editor()->hasComposition())
        handled = m_pFrame->editor()->insertText(text, 0);
    else
        m_pFrame->editor()->confirmComposition(text);
    return handled;
}

bool WebFrameImpl::handleCut()
{
    bool handled = false;
    WebCore::Frame* coreFrame = frame();
    ASSERT(coreFrame);
    WebCore::Editor* editor = coreFrame->editor();
    ASSERT(editor);
    if (editor->tryDHTMLCut()) {
        handled = true;
    }
    else if (editor->canCut()) {
        WebCore::SelectionController* selectionController = coreFrame->selectionController();
        ASSERT(selectionController);
        WTF::RefPtr<WebCore::Range> selectedRange(selectionController->toRange());
        if (editor->shouldDeleteRange(selectedRange.get())) {
            WebCore::Pasteboard* generalPasteboard = WebCore::Pasteboard::generalPasteboard(coreFrame);
            ASSERT(generalPasteboard);
            generalPasteboard->writeSelection(selectedRange.get(), false, coreFrame);
            editor->deleteSelectionWithSmartDelete(false);
            handled = true;
        }
    }
    return handled;
}

bool WebFrameImpl::handleCopy()
{
    bool handled = false;
    WebCore::Frame* coreFrame = frame();
    ASSERT(coreFrame);
    WebCore::Editor* editor = coreFrame->editor();
    ASSERT(editor);
    if (editor->tryDHTMLCopy()) {
        handled = true;
    }
    else if (editor->canCopy()) {
        WebCore::SelectionController* selectionController = coreFrame->selectionController();
        ASSERT(selectionController);

        WebCore::Document* document = coreFrame->document();

        WebCore::Pasteboard* generalPasteboard = WebCore::Pasteboard::generalPasteboard(coreFrame);
        ASSERT(generalPasteboard);

        //check if the document is an image document to treat it differently
        WebCore::Node* imageElement = 0;
        if (document && document->isImageDocument()) {
            WebCore::HTMLElement* body = document->body();
            if (body) {
                WebCore::Node* node = body->firstChild();
                if (node && node->renderer()->isImage())
                    imageElement = node;
            }
        }

        if (imageElement) {
            generalPasteboard->writeImage(imageElement, document->url(), document->title());
            handled = true;
        }
        else {
            WTF::RefPtr<WebCore::Range> selectedRange(selectionController->toRange());
            generalPasteboard->writeSelection(selectedRange.get(), false, coreFrame);
            handled = true;
        }
    }
    return handled;
}

namespace {

static bool shouldPasteFragment(const WTF::RefPtr<WebCore::DocumentFragment>& fragment, WebCore::Editor* const editor, const WTF::RefPtr<WebCore::Range>& selectedRange)
{
    bool shouldPaste = false;
    if (fragment && selectedRange) {
        WebCore::Node* firstChild = fragment->firstChild();
        WebCore::Node* lastChild = fragment->lastChild();
        
        WebCore::EditorClient* const editorClient = editor->client();
        if (firstChild && (firstChild == lastChild) && (firstChild->isCharacterDataNode())) {
            ASSERT(firstChild);
            WebCore::CharacterData* const characterDataNode = static_cast<WebCore::CharacterData*>(firstChild);
            WebCore::String const stringToPaste = characterDataNode->data();
            shouldPaste = editorClient->shouldInsertText(stringToPaste, selectedRange.get(), WebCore::EditorInsertActionPasted);
        }
        else {
            shouldPaste = editorClient->shouldInsertNode(fragment.get(), selectedRange.get(), WebCore::EditorInsertActionPasted);
        }
    }
    return shouldPaste;
}

}

bool WebFrameImpl::handlePaste()
{
    bool handled = false;
    WebCore::Frame* coreFrame = frame();
    ASSERT(coreFrame);
    
    WebCore::Editor* editor = coreFrame->editor();
    ASSERT(editor);
    if (editor->tryDHTMLPaste()) {
        handled = true;
    }
    else if (editor->canPaste()) {
        WebCore::SelectionController* selectionController = coreFrame->selectionController();
        ASSERT(selectionController);
        if (selectionController->isContentRichlyEditable()) {
            WebCore::Pasteboard* generalPasteboard = WebCore::Pasteboard::generalPasteboard(coreFrame);
            ASSERT(generalPasteboard);
            
            bool gotText = false;
            WTF::RefPtr<WebCore::Range> selectedRange(selectionController->toRange());
            WTF::RefPtr<WebCore::DocumentFragment> fragmentFromClipboard(generalPasteboard->documentFragment(coreFrame, selectedRange, true, gotText));
            if (fragmentFromClipboard) {
                bool shouldPaste = shouldPasteFragment(fragmentFromClipboard, editor, selectedRange);
                if (shouldPaste) {
                    ASSERT(selectedRange);
                    ASSERT(fragmentFromClipboard);
                    ASSERT(!selectionController->isNone());
                    WebCore::Document* const doc = coreFrame->document();
                    ASSERT(doc);
                    WTF::PassRefPtr<WebCore::EditCommand> const pasteCommand(new WebCore::ReplaceSelectionCommand(doc, fragmentFromClipboard, false, false, true));
                    WebCore::applyCommand(pasteCommand);
                    coreFrame->revealSelection(WebCore::RenderLayer::gAlignToEdgeIfNeeded);
                    handled = true;
                }
            }
        }
        else {
            editor->pasteAsPlainText();
            handled = true;
        }
    }
    return handled;
}

bool WebFrameImpl::handleSelectAll()
{
    WebCore::Frame* const coreFrame = frame();
    ASSERT(coreFrame);
    WebCore::SelectionController* const selectionController = coreFrame->selectionController();
    ASSERT(selectionController);
    selectionController->selectAll();
    return true;
}

namespace {
    inline float getPageScrollFraction()
    {
        static const float fraction = 0.75f;
        return fraction;
    }
}

bool WebFrameImpl::pageUp()
{
    WebCore::Frame* const pFrame = frame();
    ASSERT(pFrame);
    WebCore::EventHandler* const eventHandler = pFrame->eventHandler();
    ASSERT(eventHandler);
    bool handled = eventHandler->scrollOverflow(WebCore::ScrollUp, WebCore::ScrollByPage);
    if (!handled) {
        WebCore::FrameView* const frameView = pFrame->view();
        ASSERT(frameView);
        float const viewportHeight = static_cast<float>(frameView->visibleHeight());
        float const scrollFraction = getPageScrollFraction();
        int const vScrollAmount = - static_cast<int>(viewportHeight * scrollFraction);
        WebCore::IntSize const originalScrollOffset = frameView->scrollOffset();
        frameView->scrollBy(0, vScrollAmount);
        WebCore::IntSize const newScrollOffset = frameView->scrollOffset();
        handled = originalScrollOffset != newScrollOffset;
    }
    return handled;
}

bool WebFrameImpl::pageDown()
{
    WebCore::Frame* const pFrame = frame();
    ASSERT(pFrame);
    WebCore::EventHandler* const eventHandler = pFrame->eventHandler();
    ASSERT(eventHandler);
    bool handled = eventHandler->scrollOverflow(WebCore::ScrollDown, WebCore::ScrollByPage);
    if (!handled) {
        WebCore::FrameView* const frameView = pFrame->view();
        ASSERT(frameView);
        float const viewportHeight = static_cast<float>(frameView->visibleHeight());
        float const scrollFraction = getPageScrollFraction();
        int const vScrollAmount = - static_cast<int>(viewportHeight * scrollFraction);
        WebCore::IntSize const originalScrollOffset = frameView->scrollOffset();
        frameView->scrollBy(0, vScrollAmount);
        WebCore::IntSize const newScrollOffset = frameView->scrollOffset();
        handled = originalScrollOffset != newScrollOffset;
    }
    return handled; 
}

WebCore::Frame* WebFrameImpl::frame() const
{
    ASSERT(m_pFrame);
    return m_pFrame;
}

WebCore::IWidgetApolloImpl* WebFrameImpl::getFrameWidgetImpl()
{
    WebCore::Frame* pFrame = frame();
    ASSERT(pFrame);
    WebCore::FrameView* pView = pFrame->view();
    ASSERT(pView);
    // it is yucky to call get() here. 
    return pView->getApolloImpl().get();
}

void WebFrameImpl::beginModal( ) const
{
    ASSERT(m_pWebHost);
    m_pWebHost->m_pVTable->beginModal( m_pWebHost );
}

void WebFrameImpl::endModal( ) const
{
    ASSERT(m_pWebHost);
    m_pWebHost->m_pVTable->endModal( m_pWebHost );
}

int WebFrameImpl::pdfCapability() const
{
    return m_pWebHost->m_pVTable->pdfCapability( m_pWebHost );
}

int WebFrameImpl::loadPDFPlugin( void **pPDFPluginObject ) const
{
    return m_pWebHost->m_pVTable->loadPDFPlugin( m_pWebHost, pPDFPluginObject );
}

int WebFrameImpl::unloadPDFPlugin( void *pdfPluginObject ) const
{
    return m_pWebHost->m_pVTable->unloadPDFPlugin( m_pWebHost, pdfPluginObject );
}

void WebFrameImpl::handlePDFError( int pdfErrorNum ) const
{
    m_pWebHost->m_pVTable->handlePDFError( m_pWebHost, pdfErrorNum );
}

WebCore::String WebFrameImpl::openFileChooser()
{
    unsigned long filePathLength = 0;   
    uint16_t* filePathAsUTF16 = m_pWebHost->m_pVTable->openFileChooser(m_pWebHost, &filePathLength);
    ASSERT(sizeof(uint16_t) == sizeof(UChar));

    WebCore::String resultFilePath( reinterpret_cast<const UChar*>(filePathAsUTF16), filePathLength );

    if (filePathAsUTF16)
    {
        ASSERT(g_HostFunctions);
        ASSERT(g_HostFunctions->freeBytes);
        g_HostFunctions->freeBytes( filePathAsUTF16 );
    }
    
    return resultFilePath;
}
    
WebScriptProxyVariant* WebFrameImpl::getEventObject() const
{
    return m_pWebHost->m_pVTable->getEventObject(m_pWebHost);
}

const uint16_t* WebFrameImpl::getAppResourcePath( unsigned long* pAppResourcePathLength )
{
    return m_pWebHost->m_pVTable->getAppResourcePath(m_pWebHost, pAppResourcePathLength);
}

const uint16_t* WebFrameImpl::getAppStoragePath( unsigned long* pAppStoragePathLength )
{
    return m_pWebHost->m_pVTable->getAppStoragePath(m_pWebHost, pAppStoragePathLength);
}


//Methods for ApolloWebFrame interface
void WebFrameImpl::loadHTMLString(const char* const utf8HTMLString, size_t const numHTMLStringBytes, const unsigned char* const baseURL)
{
    ASSERT(m_pFrame);
    ASSERT(m_pFrame->loader());
    WebCore::KURL const baseKURL(reinterpret_cast<const char*>(baseURL));
    WebCore::ResourceRequest const request(baseKURL);
    WTF::RefPtr<WebCore::SharedBuffer> htmlBuffer = WebCore::SharedBuffer::create(utf8HTMLString, numHTMLStringBytes);
    WebCore::KURL const unreachableURL;
    WebCore::SubstituteData const htmlData(htmlBuffer.release(), "text/html", "UTF-8", unreachableURL);
    m_pFrame->loader()->load(request, htmlData);
}

void WebFrameImpl::loadRequest(const WebResourceRequest* const webRequest)
{
    WebCore::ResourceRequest request(webRequest);
    WebCore::Frame* const pFrame = frame();
    ASSERT(pFrame);
    WebCore::FrameLoader* const loader = pFrame->loader();
    ASSERT(loader);
    loader->load(request);
}

void WebFrameImpl::reload( )
{
    ASSERT(frame());
    ASSERT(frame()->loader());
    frame()->loader()->reload();
}

void WebFrameImpl::stopLoading( )
{
    frame()->loader()->stop();
}

WebScriptProxyVariant* WebFrameImpl::getGlobalObject()
{
    return WebCore::ApolloScriptBridging::getGlobalObject(frame());
}

void WebFrameImpl::historyGo( int steps )
{
    frame()->loader()->scheduleHistoryNavigation( steps );
}

unsigned WebFrameImpl::getHistoryLength()
{
    WebCore::BackForwardList *list = frame()->page()->backForwardList();
    return list->entries().size();
}

unsigned WebFrameImpl::getHistoryPosition()
{
    WebCore::BackForwardList *list = frame()->page()->backForwardList();
    return list->backListCount();
}

void WebFrameImpl::setHistoryPosition( unsigned position )
{
    WebCore::BackForwardList *list = frame()->page()->backForwardList();
    if (position < getHistoryLength())
        frame()->loader()->goBackOrForward( position - list->backListCount() );
}

void WebFrameImpl::getHistoryAt( unsigned position, char** pUrl, char** pOriginalUrl, bool* pIsPost, uint16_t** pTitle )
{
    using namespace WebCore;
    BackForwardList *list = frame()->page()->backForwardList();
    int distance = position - list->backListCount();
    HistoryItem* item = list->itemAtIndex(distance);
    if (!item) {
        if (distance > 0) {
            int forwardListCount = list->forwardListCount();
            if (forwardListCount > 0)
                item = list->itemAtIndex(forwardListCount);
        } else {
            int backListCount = list->backListCount();
            if (backListCount > 0)
                item = list->itemAtIndex(-backListCount);
        }
    }

    // indicates failure by NOT initializing *pUrl.
    if (item)
    {
        String url = item->url();
        *pUrl = (char *)g_HostFunctions->allocBytes( url.length()+1 );
        if (url.length() > 0)
            memcpy(*pUrl, url.utf8().data(), url.length()+1);
        else
            **pUrl = 0;

        url = item->originalURL();
        *pOriginalUrl = (char *)g_HostFunctions->allocBytes( url.length()+1 );
        if (url.length() > 0)
            memcpy(*pOriginalUrl, url.utf8().data(), url.length()+1);
        else
            **pOriginalUrl = 0;

        const String &title = item->title();
        *pTitle = (uint16_t *)g_HostFunctions->allocBytes( sizeof(uint16_t)*(title.length()+1) );
        if (title.length() > 0)
            memcpy(*pTitle, title.characters(), sizeof(uint16_t)*(title.length()));
        (*pTitle)[title.length()] = 0;

        *pIsPost = (item->formData() != 0);
    }
}

        
void WebFrameImpl::frameLoaderDestroyed()
{
    delete this;
}
        
bool WebFrameImpl::hasWebView() const
{
    return true;
}

bool WebFrameImpl::hasFrameView() const
{
    ASSERT(m_pFrame);
    return m_pFrame->view() != 0;
}

void WebFrameImpl::makeRepresentation(WebCore::DocumentLoader*)
{
}

void WebFrameImpl::transitionToCommittedForNewPage()
{
    WTF::RefPtr<WebCore::FrameView> oldFrameView(m_pFrame->view());
    WTF::RefPtr<FrameScrollViewImpl> const oldScrollViewImpl(m_scrollViewImpl);
    
    // Since the old frame view is going to die in short order, we need to see if we have
    // data redirection to a plugin pending.  If we do, we need to tell the plugin the stream
    // had an error and clear the data redirection.  We tell the plugin that the stream
    // had an error so that the stream will be in the correct state when it is destroyed.
    WebCore::ResourceHandleClient* const resHandleRedirect = resourceHandleClientRedirect();
    if (resHandleRedirect) {
        // The plugin to which we were redirecting to should only exist if we previously had a frame view.
        ASSERT(oldFrameView);
        
        clearResourceHandleRedirect();
        WebCore::ResourceError const resourceError;
        resHandleRedirect->didFail(0, resourceError);
    }


    // If this isn't the main frame, it must have a owner element set, or it
    // won't ever get installed in the view hierarchy.
    bool const isMainFrame = m_pFrame == m_pFrame->page()->mainFrame();
    ASSERT(isMainFrame || (m_pFrame->ownerElement()));
    if (oldFrameView) {
        m_pFrame->setView(0);
    }
    
    WTF::RefPtr<WebCore::FrameView> frameView(adoptRef(new WebCore::FrameView(m_pFrame)));
    ASSERT(frameView->hasOneRef());

    WTF::RefPtr<FrameScrollViewImpl> newScrollViewImpl;
    if (isMainFrame)
    {
        newScrollViewImpl = m_pOwningWebView->createScrollViewImplForRootFrame(this, frameView.get());

        WindowScrollViewImpl* oldWindowScrollView = static_cast<WindowScrollViewImpl*>(oldScrollViewImpl.get());
        if(oldWindowScrollView && !oldWindowScrollView->hostDrawsScrollbars())
        {
            WindowScrollViewImpl* newWindowScrollView = static_cast<WindowScrollViewImpl*>(newScrollViewImpl.get());
            newWindowScrollView->setHostDrawsScrollbars(false);
        }
    }
    else
        newScrollViewImpl = m_pOwningWebView->createScrollViewImplForSubFrame(this, frameView.get());

    ASSERT(newScrollViewImpl->hasOneRef());
    frameView->attachApolloImpl(newScrollViewImpl);  // After this point the new frameView is fully constructed.
    ASSERT(!newScrollViewImpl->hasOneRef());
    
    m_scrollViewImpl = newScrollViewImpl;
    
    ASSERT(m_pFrame);
    // frame now holds strong reference on frameView we just constructed.
    m_pFrame->setView(frameView.get());
    ASSERT(!(frameView->hasOneRef()));

    // Added this code to fix bug [1603295] - HTML: setting systemChrome=none / transparent=true doesn't make the app transparent
    //
    if ((oldFrameView) && (oldFrameView->baseBackgroundColor().alpha() < 0xff))
        frameView->setBaseBackgroundColor( oldFrameView->baseBackgroundColor() );
    
    // This is kooky code from WebCoreFrameBridge.mm, installInFrame.
    // This code is important because if we are making a new WebCore::FrameView
    // for a WebCore::Frame that already had a WebCore::FrameView and that frame
    // was a subframe, the renderer for the frame element in the DOM holds a reference
    // to the old WebCore::FrameView.  This code updates that reference in the renderer
    // to point at the new WebCore::FrameView we just made.  God knows why it works this
    // way.  Hopefully somebody will change this such that we either do not make new FrameView's
    // all the time or at least do a better job of encapsulating this mess in WebCore.
    if (m_pFrame->ownerRenderer())
        m_pFrame->ownerRenderer()->setWidget(frameView.get());
    
    if (WebCore::HTMLFrameOwnerElement* frameOwner = m_pFrame->ownerElement())
        m_pFrame->view()->setScrollbarsMode(frameOwner->scrollingMode());

    // Once we have called setWidget on the frame's associated
    // renderer ( if it had one ), then the last ref on the
    // old FrameView should be the RefPtr we made at the top of this
    // method.  If this assert fails then something about
    // how we manage FrameView's is wrong.  It is a delicate
    // dance and must always be inferred from PLATFORM(MAC).
    ASSERT((!oldFrameView) || (oldFrameView->hasOneRef()));
    
    ASSERT((!oldFrameView) || (!oldFrameView->getApolloImpl()->getParent()));
    // If we clear this scope's reference to the oldFrameView it will be destroyed
    // at which point the only hard reference left to the old scrollview impl will be
    // the hard reference in this functions scope.
    oldFrameView.clear();
    ASSERT((!oldScrollViewImpl) || (oldScrollViewImpl->hasOneRef()));

    frameView->initScrollbars();
    
    // Any time we change out the frameView, we need to set m_firstLayoutForFrameView to true.
    m_firstLayoutForFrameView = true;

    
    
}

void WebFrameImpl::savePlatformDataToCachedPage(WebCore::CachedPage*)
{
}

void WebFrameImpl::transitionToCommittedFromCachedPage(WebCore::CachedPage*)
{
}

void WebFrameImpl::forceLayout()
{
    WebCore::Frame* const theFrame = frame();
    ASSERT(theFrame);
    ASSERT(hasFrameView());
    theFrame->forceLayout(true);
    
    WebCore::FrameView* const frameView = theFrame->view();
    ASSERT(frameView);
    ASSERT(!frameView->needsLayout());
    
    WebCore::IntSize const currFrameSize(frameView->frameGeometry().size());
    
    if ((!m_firstLayoutForFrameView) && (currFrameSize != m_lastLayoutSize)) {
        theFrame->sendResizeEvent();
        ASSERT(theFrame->view() == frameView);
        if (frameView->needsLayout())
            theFrame->forceLayout(true);
    }
    
    m_lastLayoutSize = currFrameSize;
    m_firstLayoutForFrameView = false;
    
    ASSERT(!frameView->needsLayout());
}

void WebFrameImpl::forceLayoutForNonHTML()
{
}

void WebFrameImpl::setCopiesOnScroll()
{
}

void WebFrameImpl::detachedFromParent2()
{
}

void WebFrameImpl::detachedFromParent3()
{
}

void WebFrameImpl::detachedFromParent4()
{
}

void WebFrameImpl::loadedFromPageCache()
{
}

void WebFrameImpl::assignIdentifierToInitialRequest(unsigned long identifier, WebCore::DocumentLoader*, const WebCore::ResourceRequest&)
{
}

void WebFrameImpl::dispatchWillSendRequest(WebCore::DocumentLoader*, unsigned long identifier, WebCore::ResourceRequest&, const WebCore::ResourceResponse& redirectResponse)
{
}

void WebFrameImpl::dispatchDidReceiveAuthenticationChallenge(WebCore::DocumentLoader*, unsigned long identifier, const WebCore::AuthenticationChallenge&)
{
}

void WebFrameImpl::dispatchDidCancelAuthenticationChallenge(WebCore::DocumentLoader*, unsigned long identifier, const WebCore::AuthenticationChallenge&)
{
}

void WebFrameImpl::dispatchDidReceiveResponse(WebCore::DocumentLoader*, unsigned long identifier, const WebCore::ResourceResponse& resourceResponse)
{
}

void WebFrameImpl::dispatchDidReceiveContentLength(WebCore::DocumentLoader*, unsigned long identifier, int lengthReceived)
{
}

void WebFrameImpl::dispatchDidFinishLoading(WebCore::DocumentLoader*, unsigned long identifier)
{
}

void WebFrameImpl::dispatchDidFailLoading(WebCore::DocumentLoader*, unsigned long identifier, const WebCore::ResourceError&)
{
}

bool WebFrameImpl::dispatchDidLoadResourceFromMemoryCache(WebCore::DocumentLoader*, const WebCore::ResourceRequest&, const WebCore::ResourceResponse&, int length)
{
    return false;
}

void WebFrameImpl::dispatchDidHandleOnloadEvents()
{
    if (m_pWebHost) {
        ASSERT(m_pWebHost->m_pVTable);
        ASSERT(m_pWebHost->m_pVTable->handleOnLoadEvents);
        m_pWebHost->m_pVTable->handleOnLoadEvents(m_pWebHost);
    }
}

void WebFrameImpl::dispatchDidReceiveServerRedirectForProvisionalLoad()
{
}

void WebFrameImpl::dispatchDidCancelClientRedirect()
{
}

void WebFrameImpl::dispatchWillPerformClientRedirect(const WebCore::KURL&, double interval, double fireDate)
{
}

void WebFrameImpl::dispatchDidChangeLocationWithinPage()
{
}

void WebFrameImpl::dispatchWillClose()
{
}

void WebFrameImpl::dispatchDidReceiveIcon()
{
}

void WebFrameImpl::dispatchDidStartProvisionalLoad()
{
    WebCore::Frame* const pFrame = frame();
    ASSERT(pFrame);
    ASSERT(pFrame->loader());
    ASSERT(pFrame->loader()->activeDocumentLoader());
    updateLocation(pFrame->loader()->activeDocumentLoader()->requestURL());
}

void WebFrameImpl::dispatchDidReceiveTitle(const WebCore::String& title)
{
    ASSERT(!m_dead);
    if (m_pWebHost) {
        ASSERT(sizeof(uint16_t) == sizeof(UChar));
        const uint16_t* const titleUTF16 = reinterpret_cast<const uint16_t*>(title.characters());
        unsigned long const numTitleCodeUnits = title.length();
        m_pWebHost->m_pVTable->setTitle(m_pWebHost, titleUTF16, numTitleCodeUnits);
    }
}

void WebFrameImpl::dispatchDidCommitLoad()
{
    WebCore::Frame* const theFrame = frame();
    
    // If either of these is non-null it means that windowObjectDead was not called before this method
    // was called.  We used to deadify script bridging objects in here, but that is no longer
    // soon enough in the loading process.  It is still important that the script bridging objects
    // are deadified before this method get called to make sure we never run a JS function
    // from an old html document against the current html document's object model.
    ASSERT(!m_childDoor);
    ASSERT(!m_childDoorProxy);

    theFrame->view()->setContentsPos(0, 0);
    
    ASSERT(theFrame->loader());
    ASSERT(theFrame->loader()->documentLoader());
    updateLocation(theFrame->loader()->documentLoader()->requestURL());
}

void WebFrameImpl::dispatchDidFailProvisionalLoad(const WebCore::ResourceError&)
{
}

void WebFrameImpl::dispatchDidFailLoad(const WebCore::ResourceError&)
{
}

void WebFrameImpl::dispatchDidFinishDocumentLoad()
{
}

void WebFrameImpl::dispatchDidFinishLoad()
{
}

void WebFrameImpl::dispatchDidFirstLayout()
{
}

WebCore::Frame* WebFrameImpl::dispatchCreatePage()
{
    WebHost* const host = webHost();
    ASSERT(host);
    ASSERT(host->m_pVTable);
    ASSERT(host->m_pVTable->createNewWindow);
    WebHostCreateWindowArgs createArgs = {
        sizeof(WebHostCreateWindowArgs),
        WebHostCreateWindowArgs::USE_DEFAULT,
        WebHostCreateWindowArgs::USE_DEFAULT,
        WebHostCreateWindowArgs::USE_DEFAULT,
        WebHostCreateWindowArgs::USE_DEFAULT,
        true,
        true,
        true,
        true,
        true,
        true,
        false
    };
    WebView* const webView = host->m_pVTable->createNewWindow(host, &createArgs);
    WebCore::Frame* newPageFrame = 0;
    if (webView) {
        WebViewImpl* const webViewImpl = WebViewImpl::getImpl(webView);
        WebCore::Page* const page = webViewImpl->page();
        newPageFrame = page->mainFrame();
    }
    return newPageFrame;
}

void WebFrameImpl::dispatchShow()
{
    WebCore::Frame* const pFrame = frame();
    ASSERT(pFrame);
    WebCore::Page* const page = pFrame->page();
    ASSERT(page);
    WebCore::Chrome* const chrome = page->chrome();
    chrome->show();
}

void WebFrameImpl::dispatchDecidePolicyForMIMEType(WebCore::FramePolicyFunction function, const WebCore::String& MIMEType, const WebCore::ResourceRequest&)
{
    ASSERT(m_pFrame);
    ASSERT(m_pFrame->loader());
    (m_pFrame->loader()->*function)(WebCore::PolicyUse);
}

namespace {
    // ASSERT at compile time that the WebNavigationPolicy and WebNavigationType enumeration match
    // their webcore counterparts.  This way we can use reinterpret_cast to do the conversion between the WebCore
    // and WebKit types.
    COMPILE_ASSERT(WebCore::PolicyUse == static_cast<WebCore::PolicyAction>(WebNavigationPolicyActionUse), PolicyUseNotEqualWebNavigationPolicyActionUse);
    COMPILE_ASSERT(WebCore::PolicyIgnore == static_cast<WebCore::PolicyAction>(WebNavigationPolicyActionIgnore), PolicyIgnoreNotEqualWebNavigationPolicyActionIgnore);
    COMPILE_ASSERT(WebCore::PolicyDownload == static_cast<WebCore::PolicyAction>(WebNavigationPolicyActionDownload), PolicyDownloadNotEqualWebNavigationPolicyActionDownload);
    COMPILE_ASSERT(static_cast<WebNavigationType>(WebCore::NavigationTypeLinkClicked) == WebNavigationTypeLinkClicked, NavigationLinkClickedNotEqualWebNavigationTypeLinkClicked);
    COMPILE_ASSERT(static_cast<WebNavigationType>(WebCore::NavigationTypeFormSubmitted) == WebNavigationTypeFormSubmitted, NavigationTypeFormSubmittedNotEqualWebNavigationTypeFormSubmitted);
    COMPILE_ASSERT(static_cast<WebNavigationType>(WebCore::NavigationTypeBackForward) == WebNavigationTypeBackForward, NavigationTypeBackForwardNotEqualWebNavigationTypeBackForward);
    COMPILE_ASSERT(static_cast<WebNavigationType>(WebCore::NavigationTypeReload) == WebNavigationTypeReload, NavigationTypeReloadNotEqualWebNavigationTypeReload);
    COMPILE_ASSERT(static_cast<WebNavigationType>(WebCore::NavigationTypeFormResubmitted) == WebNavigationTypeFormResubmitted, NavigationTypeFormResubmittedNotEqualWebNavigationTypeFormResubmitted);
    COMPILE_ASSERT(static_cast<WebNavigationType>(WebCore::NavigationTypeOther) == WebNavigationTypeOther, NavigationTypeOtherNotEqualWebNavigationTypeOther);
}

void WebFrameImpl::dispatchDecidePolicyForNewWindowAction(WebCore::FramePolicyFunction function, const WebCore::NavigationAction& navAction, const WebCore::ResourceRequest& resourceRequest, const WebCore::String& frameName)
{
    ASSERT(m_pFrame);
    ASSERT(m_pFrame->loader());
    
    WebCore::KURL const navActionURL(navAction.url());
    WebCore::CString const navActionURLString(navActionURL.string().latin1());
    
    const unsigned char* navActionURLBytes = reinterpret_cast<const unsigned char*>(navActionURLString.data());
    unsigned long const navActionURLLen = navActionURLString.length();
    ASSERT((navActionURLBytes) || (navActionURLLen == 0));
    
    WebCore::NavigationType const navType = navAction.type();
    WebNavigationType const webNavType = static_cast<WebNavigationType>(navType);
    
    WTF::RefPtr<WebResourceRequestImpl> const webResourceRequestImpl(WebResourceRequestImpl::construct(resourceRequest));
    WebResourceRequest* const webResourceRequest = webResourceRequestImpl->getWebResourceRequest();
    
    COMPILE_ASSERT(sizeof(uint16_t) == sizeof(UChar), sizeOfUInt16MustEqualSizeOfUChar);
    
    const uint16_t* const frameNameUTF16 = reinterpret_cast<const uint16_t*>(frameName.characters());
    unsigned long const numFrameNameUTF16CodeUnits = frameName.length();
    
    ASSERT(m_pWebHost);
    ASSERT(m_pWebHost->m_pVTable);
    ASSERT(m_pWebHost->m_pVTable->dispatchDecidePolicyForNewWindowAction);
    
    WebNavigationPolicyAction const webNavigationPolicyAction =
        m_pWebHost->m_pVTable->dispatchDecidePolicyForNewWindowAction(m_pWebHost
                                                                     , navActionURLBytes
                                                                     , navActionURLLen
                                                                     , webNavType
                                                                     , webResourceRequest
                                                                     , frameNameUTF16
                                                                     , numFrameNameUTF16CodeUnits);
                                                          
    WebCore::PolicyAction const policyAction = static_cast<WebCore::PolicyAction>(webNavigationPolicyAction);
    (m_pFrame->loader()->*function)(policyAction);
}

void WebFrameImpl::dispatchDecidePolicyForNavigationAction(WebCore::FramePolicyFunction function, const WebCore::NavigationAction& navAction, const WebCore::ResourceRequest& resourceRequest)
{
    ASSERT(m_pFrame);
    ASSERT(m_pFrame->loader());

    WebCore::PolicyAction policyAction = WebCore::PolicyUse;

    WebCore::NavigationType const navType = navAction.type();
    WebCore::KURL const navActionURL(navAction.url());
    switch (navType)
    {
    case WebCore::NavigationTypeLinkClicked:
    case WebCore::NavigationTypeFormSubmitted:
    case WebCore::NavigationTypeOther:          // includes window.open case and HTMLLoader.load() case
        {
            FrameLoaderClientApollo *client = static_cast<FrameLoaderClientApollo *>(m_pFrame->loader()->client());
            // If there's a referrer, check that.
            // Otherwise, this is one of thise situations (HTMLLoader API call, etc.)
            // where the request should be allowed irrespective of the current page.
            WebCore::String const referrer(resourceRequest.httpReferrer());
            if (!referrer.isEmpty())
                if (!client->canLoad(navActionURL, referrer))
                    policyAction = WebCore::PolicyIgnore;

            // Now we prevent opened windows from escalating to local or app content
            if (WebCore::Frame *opener = m_pFrame->loader()->opener()) {
                if (WebCore::Document *doc = opener->document()) {
                    if (!client->canLoad(navActionURL, doc))
                        policyAction = WebCore::PolicyIgnore;
                }
            }
        }
        break;
    case WebCore::NavigationTypeBackForward:
    case WebCore::NavigationTypeReload:
    case WebCore::NavigationTypeFormResubmitted:
    default:
        // leave PolicyAction as PolicyUse
        break;
    }

    // If the above switch statement did not block the load,
    // then call through to the WebHost and let it have a crack
    // at blocking the request.
    if (policyAction == WebCore::PolicyUse) {
        WebCore::KURL const navActionURL(navAction.url());
        
        WebCore::NavigationType const navType = navAction.type();
        WebNavigationType const webNavType = static_cast<WebNavigationType>(navType);
        
        WTF::RefPtr<WebResourceRequestImpl> const webResourceRequestImpl(WebResourceRequestImpl::construct(resourceRequest));
        WebResourceRequest* const webResourceRequest = webResourceRequestImpl->getWebResourceRequest();
        
        ASSERT(m_pWebHost);
        ASSERT(m_pWebHost->m_pVTable);
        ASSERT(m_pWebHost->m_pVTable->dispatchDecidePolicyForNewWindowAction);
        
		WebString* navActionURLString = navActionURL.string().webString();
        WebNavigationPolicyAction const webNavigationPolicyAction =
            m_pWebHost->m_pVTable->dispatchDecidePolicyForNavigationAction(m_pWebHost
                                                                          , navActionURLString
                                                                          , webNavType
                                                                          , webResourceRequest);
		navActionURLString->m_vTable->free(navActionURLString);
        policyAction = static_cast<WebCore::PolicyAction>(webNavigationPolicyAction);
    }

    (m_pFrame->loader()->*function)(policyAction);
}

void WebFrameImpl::cancelPolicyCheck()
{
}

void WebFrameImpl::dispatchUnableToImplementPolicy(const WebCore::ResourceError&)
{
}

void WebFrameImpl::dispatchWillSubmitForm(WebCore::FramePolicyFunction function, WTF::PassRefPtr<WebCore::FormState>)
{
    ASSERT(m_pFrame);
    ASSERT(m_pFrame->loader());
    (m_pFrame->loader()->*function)(WebCore::PolicyUse);
}

void WebFrameImpl::dispatchDidLoadMainResource(WebCore::DocumentLoader*)
{
}

void WebFrameImpl::revertToProvisionalState(WebCore::DocumentLoader*)
{
}

void WebFrameImpl::setMainDocumentError(WebCore::DocumentLoader* loader, const WebCore::ResourceError& error)
{
}

void WebFrameImpl::postProgressStartedNotification()
{
}

void WebFrameImpl::postProgressEstimateChangedNotification()
{
}

void WebFrameImpl::postProgressFinishedNotification()
{
}

void WebFrameImpl::setMainFrameDocumentReady(bool)
{
}

void WebFrameImpl::startDownload(const WebCore::ResourceRequest&)
{
}

void WebFrameImpl::willChangeTitle(WebCore::DocumentLoader*)
{
}

void WebFrameImpl::didChangeTitle(WebCore::DocumentLoader*)
{
}

void WebFrameImpl::committedLoad(WebCore::DocumentLoader* loader, const char* bytes, int length)
{
    WebCore::ResourceHandleClient* rhl = resourceHandleClientRedirect();

    if(rhl)
    {
        rhl->didReceiveData(NULL, bytes, -1, length);
        return;
    }
    else
    {
        WebCore::Frame* const pFrame = frame();
        ASSERT(pFrame);
        WebCore::FrameLoader* const pFrameLoader = pFrame->loader();
        ASSERT(pFrameLoader);
		const WebCore::SubstituteData& substituteData = pFrameLoader->documentLoader()->substituteData();
		WebCore::String encoding;
		bool userChosen;
		if ( substituteData.isValid() )
		{
			encoding = substituteData.textEncoding();
			userChosen = false;
		}
		else
		{
			encoding = pFrameLoader->documentLoader()->overrideEncoding();
			userChosen = ! encoding.isNull();
			if (encoding.isNull())
				encoding = pFrameLoader->documentLoader()->response().textEncodingName();
		}
		pFrameLoader->setEncoding(encoding, userChosen);
		pFrameLoader->addData( bytes, length );
    }

    // we may now be sending data to plug-in, so check again
    rhl = resourceHandleClientRedirect();

    if (rhl)
        rhl->didReceiveData(NULL, bytes, -1, length);
    else if (m_didRedirectDataToPlugin)
    {
        WebCore::Frame* const pFrame = frame();
        ASSERT(pFrame);
        WebCore::FrameLoader* const pFrameLoader = pFrame->loader();
        ASSERT(pFrameLoader);
        WebCore::DocumentLoader* const pDocLoader = pFrameLoader->activeDocumentLoader();
        ASSERT(pDocLoader);
        WebCore::ResourceError const loadError;
        pDocLoader->cancelMainResourceLoad(loadError);
        clearResourceHandleRedirect();
    }
}

void WebFrameImpl::finishedLoading(WebCore::DocumentLoader* loader)
{
    WebCore::ResourceHandleClient* rhl = resourceHandleClientRedirect();

    if(rhl) {
        ASSERT(m_didRedirectDataToPlugin);
        // If there we are redirect of data to a plugin, time to tell the plugin no more data is coming.
        rhl->didFinishLoading(0);
        clearResourceHandleRedirect();// Don't need to vector data to the plug-in anymore
    }
    else
    {
        WebCore::FrameLoader* const pFrameLoader = frame()->loader();
        ASSERT(pFrameLoader);
        ASSERT(pFrameLoader->documentLoader());
        // Get the encoding and set it back because FrameLoader really wants setEncoding to be called
        // before any calls to addData.  Calling setEncoding more than needed is harmless.
 		const WebCore::SubstituteData& substituteData = pFrameLoader->documentLoader()->substituteData();
		WebCore::String encoding;
		bool userChosen;
		if ( substituteData.isValid() )
		{
			encoding = substituteData.textEncoding();
			userChosen = false;
		}
		else
		{
			encoding = pFrameLoader->documentLoader()->overrideEncoding();
			userChosen = ! encoding.isNull();
			if (encoding.isNull())
				encoding = pFrameLoader->documentLoader()->response().textEncodingName();
		}

		pFrameLoader->setEncoding(encoding, userChosen);

        // TODO_AIRLinux : This needs to be fixed for UNIX. It crashes here.
        m_pFrame->loader()->addData( 0, 0 );
    }
}

void WebFrameImpl::updateGlobalHistory(const WebCore::KURL&)
{
}

bool WebFrameImpl::shouldGoToHistoryItem(WebCore::HistoryItem*) const
{
    return true;
}

namespace {
static const char resourceErrorDomain[] = "WebKitApollo";
static const int cancelledErrorCode = 1;
static const int blockedErrorCode = 2;
static const int cannotShowURLErrorCode = 3;
static const int interruptForPolicyChangeErrorCode = 4;
static const int cannotShowMIMETypeErrorCode = 5;
static const int fileDoesNotExistErrorCode = 6;
}

WebCore::ResourceError WebFrameImpl::cancelledError(const WebCore::ResourceRequest& req)
{
    return WebCore::ResourceError(resourceErrorDomain, cancelledErrorCode, req.url().string(), WebCore::String());
}

WebCore::ResourceError WebFrameImpl::blockedError(const WebCore::ResourceRequest& req)
{
    return WebCore::ResourceError(resourceErrorDomain, blockedErrorCode, req.url().string(), WebCore::String());
}

WebCore::ResourceError WebFrameImpl::cannotShowURLError(const WebCore::ResourceRequest& req)
{
    return WebCore::ResourceError(resourceErrorDomain, cannotShowURLErrorCode, req.url().string(), WebCore::String());
}

WebCore::ResourceError WebFrameImpl::interruptForPolicyChangeError(const WebCore::ResourceRequest& req)
{
    return WebCore::ResourceError(resourceErrorDomain, interruptForPolicyChangeErrorCode, req.url().string(), WebCore::String());
}

WebCore::ResourceError WebFrameImpl::cannotShowMIMETypeError(const WebCore::ResourceResponse& response)
{
    return WebCore::ResourceError(resourceErrorDomain, cannotShowMIMETypeErrorCode, response.url().string(), WebCore::String());
}

WebCore::ResourceError WebFrameImpl::fileDoesNotExistError(const WebCore::ResourceResponse& response)
{
    return WebCore::ResourceError(resourceErrorDomain, fileDoesNotExistErrorCode, response.url().string(), WebCore::String());
}

bool WebFrameImpl::shouldFallBack(const WebCore::ResourceError&)
{
    return false;
}

bool WebFrameImpl::canHandleRequest(const WebCore::ResourceRequest&) const
{
    return true;
}

bool WebFrameImpl::canShowMIMEType(const WebCore::String& MIMEType) const
{
    if (WebCore::MIMETypeRegistry::isSupportedImageMIMEType(MIMEType)
     || WebCore::MIMETypeRegistry::isSupportedImageResourceMIMEType(MIMEType)
     || WebCore::MIMETypeRegistry::isSupportedNonImageMIMEType(MIMEType)
#ifdef PLUGINS_ENABLED
     || WebCore::PluginInfoStore::isPDFMIMEType(MIMEType)
     || WebCore::PluginInfoStore::isFlashMIMEType(MIMEType)
#endif
     )
        return true;
    return false;
}

bool WebFrameImpl::representationExistsForURLScheme(const WebCore::String& URLScheme) const
{
    return false;
}

WebCore::String WebFrameImpl::generatedMIMETypeForURLScheme(const WebCore::String& URLScheme) const
{
    return WebCore::String();
}

void WebFrameImpl::frameLoadCompleted()
{
}

void WebFrameImpl::saveViewStateToItem(WebCore::HistoryItem*)
{
}

void WebFrameImpl::restoreViewState()
{
}

void WebFrameImpl::provisionalLoadStarted()
{
}

void WebFrameImpl::didFinishLoad()
{
}

void WebFrameImpl::prepareForDataSourceReplacement()
{
}

WTF::PassRefPtr<WebCore::DocumentLoader> WebFrameImpl::createDocumentLoader(const WebCore::ResourceRequest& request, const WebCore::SubstituteData& data)
{
    RefPtr<WebCore::DocumentLoader> pDocLoader(new WebCore::DocumentLoader(request, data));
    return pDocLoader.release();
}

void WebFrameImpl::setTitle(const WebCore::String& title, const WebCore::KURL&)
{
}

WebCore::String WebFrameImpl::userAgent(const WebCore::KURL&)
{
    ASSERT(m_pWebHost);
    ASSERT(m_pWebHost->m_pVTable);
    unsigned long numUserAgentCodeUnits = 0;
    const uint16_t* userAgentText = m_pWebHost->m_pVTable->getUserAgentText(m_pWebHost, &numUserAgentCodeUnits);
    return WebCore::String(reinterpret_cast<const UChar*>(userAgentText), numUserAgentCodeUnits);
}

bool WebFrameImpl::canCachePage() const
{
    return false;
}

void WebFrameImpl::download(WebCore::ResourceHandle* resourceHandle, const WebCore::ResourceRequest& resourceRequest, const WebCore::ResourceRequest&, const WebCore::ResourceResponse& resourceResponse)
{
}

WTF::PassRefPtr<WebCore::Frame> WebFrameImpl::createFrame(const WebCore::KURL& url, const WebCore::String& name, WebCore::HTMLFrameOwnerElement* ownerElement,
                                                          const WebCore::String& referrer, bool allowsScrolling, int marginWidth, int marginHeight)
{
    ASSERT( ! m_dead );
    if (!m_pWebHostImplForChildFrames.get())
    {
        m_pWebHostImplForChildFrames =
            std::auto_ptr<Private::WebHostImplForFrames>( new Private::WebHostImplForFrames(m_pWebHost));
    }
    ASSERT(m_pWebHostImplForChildFrames.get());
    WebHost* const pWebHostForChildFrames =
        m_pWebHostImplForChildFrames->getWebHost();

    WebFrameImpl* const pChildWebFrame(new WebFrameImpl( m_pOwningWebView
                                                       , pWebHostForChildFrames
                                                       , ownerElement
                                                       , this));
    // ** EVIL **
    // child web frame points at the webcore frame, but does not own the WebCore::Frame.
    // nothing else points at the webcore frame at the momement.  The WebCore::Frame
    // is just sort of hanging out for the moment.
    // Calling the magic init method on the webcore frame will cause
    // the FrameView to be constructed which will in turn take a reference on the
    // WebCore::Frame and increment the refcount of the WebCore::Frame.
    //
    // Make sure the WebCore::Frame's refcount is one.
    WebCore::Frame* const rawNewFrame = pChildWebFrame->frame();
    ASSERT(rawNewFrame->hasOneRef());
    
    // Make sure we don't already have a frame by this name.
    ASSERT( ! isChildFrame( name ) );
    
    rawNewFrame->tree()->setName(name);
    
    // Make sure the WebCore::Frame's refcount is one.
    ASSERT(rawNewFrame->hasOneRef());
    
    ASSERT(!rawNewFrame->view());

    //Watson 1910885: FrameTree crash
	//we have to insert the frame into the tree before init 
	//<rdar://problem/5823684> - Crash manipulating frame tree of a new frame before the new frame
    //    has been installed in a frame tree.
    //    The root of this problem was that calling init() on a new frame could end up calling arbitrary
    //    javascript that might end up removing the frame from the tree.  This opened up a small can of worms
    //    such as the frame not having yet been installed in its frame tree, and other assumed behavior while
    //    destroying the frame.	
    WTF::RefPtr<WebCore::Frame> newFrame(pChildWebFrame->frame());
    ASSERT(newFrame);
    frame()->tree()->appendChild(newFrame);
    
    // Calling init results in a call to transitionToCommittedForNewPage ( defined below ).
    // transitionToCommittedForNewPage creates the FrameView which will take a strong reference
    // to the newFrame.
    rawNewFrame->init();

    //init may trigger code to remove the frame from the tree, therefore we have
	//to check that before going further
    if (!newFrame->tree()->parent()) {
        ASSERT(!newFrame->view());  // the view should be null, because the tear down logic
                                    // that starts nuking the frame, set's the view to 0,
                                    // which should cause the view to be destroyed.
		//necessary, otherwise you will end up leaking info
		//LEAK: 1 RenderObject
		//LEAK: 1 Node
		//LEAK: 1 Frame
		//LEAK: 1 WebFrame
		//LEAK: 1 WebFrame
		rawNewFrame->deref();
        return 0;
    }

    ASSERT(rawNewFrame->view());
    ASSERT(rawNewFrame->view()->frame() == rawNewFrame);
    ASSERT(!(rawNewFrame->hasOneRef()));
    rawNewFrame->deref();
    
    WebCore::Frame* const thisFrame = frame();                                                
    ASSERT(thisFrame);
    
    // We need to set the newFrame's WidgetImpl to point at the IScrollViewApolloImpl
    // of the frameView of this frame.
    WebCore::FrameView* const thisFrameView = thisFrame->view();
    WebCore::FrameView* const newFrameView = newFrame->view();
    ASSERT(thisFrameView); (void) thisFrameView;
    ASSERT(thisFrameView->getApolloImpl());
    ASSERT(newFrameView); (void) newFrameView;
    ASSERT(newFrameView->getApolloImpl());
    ASSERT((!(newFrameView->getApolloImpl()->getParent())) || (newFrameView->getApolloImpl()->getParent() == thisFrameView->getApolloImpl()));
    
    //Stealing code from WebFrame.mm
    WebCore::FrameLoader* const loader = frame()->loader();
    ASSERT(loader);
    WebCore::HistoryItem* const parentItem = loader->currentHistoryItem();
    WebCore::FrameLoadType const loadType = loader->loadType();
    WebCore::FrameLoadType childLoadType = WebCore::FrameLoadTypeRedirectWithLockedHistory;
    WebCore::KURL urlToLoad(url);
    if (parentItem && parentItem->children().size() &&
        (WebCore::isBackForwardLoadType(loadType)
         || (loadType == WebCore::FrameLoadTypeReload)
         || (loadType == WebCore::FrameLoadTypeReloadAllowingStaleData))) {
        
        WebCore::HistoryItem* const childItem = parentItem->childItemWithName(name);
        if (childItem) {
            urlToLoad = WebCore::KURL(childItem->originalURLString());
            childLoadType = loadType;
            if (isBackForwardLoadType(loadType))
                newFrame->loader()->setProvisionalHistoryItem(childItem);
            else
                newFrame->loader()->setCurrentHistoryItem(childItem);
        }
    }
    //---- Done stealing code -----
    newFrame->loader()->load(urlToLoad, referrer, childLoadType, WebCore::String(), 0, 0);

    // loading the frame may have resulted in the new frame being removed from the frame tree,
    // if so we should return NULL.
    if (!newFrame->tree()->parent()) {
        ASSERT(!newFrame->view());  // the view should be null, because the tear down logic
                                    // that starts nuking the frame, set's the view to 0,
                                    // which should cause the view to be destroyed.
        return 0;
    }

    ASSERT(newFrame->view());

    return newFrame.release();
}
namespace {

std::auto_ptr<WebCore::Widget> platformCreateFlashPluginView(const WebCore::IntSize& pluginSize
                                                            , WebCore::Element* element
                                                            , const WebCore::KURL& url
                                                            , const WTF::Vector<WebCore::String>& paramNames
                                                            , const WTF::Vector<WebCore::String>& paramValues
                                                            , const WebCore::String& mimeType
                                                            , WebFrameImpl* frameImpl
                                                            , bool loadManually)
{
#ifdef PLUGINS_ENABLED
    WebCore::PluginView* pluginView = WebCore::PluginView::create(frameImpl->frame()
                                                                 , pluginSize
                                                                 , element
                                                                 , url
                                                                 , paramNames
                                                                 , paramValues
                                                                 , mimeType
                                                                 , loadManually);
    std::auto_ptr<WebCore::Widget> widgetPtr(pluginView);

    ASSERT( pluginView );

    if(pluginView)
    {
        if (pluginView->status() == WebCore::PluginStatusLoadedSuccessfully)
            return widgetPtr;
    }
#endif

    return std::auto_ptr<WebCore::Widget>();
}

}

WebCore::Widget* WebFrameImpl::createPlugin(const WebCore::IntSize& pluginSize, WebCore::Element* element, const WebCore::KURL& url, const WTF::Vector<WebCore::String>& paramNames, const WTF::Vector<WebCore::String>& paramValues, const WebCore::String& mimeTypeIn, bool loadManually)
{
#ifdef PLUGINS_ENABLED
    std::auto_ptr<WebCore::Widget> widget;

    WebCore::String mimeType = mimeTypeIn;
    if (mimeType.isEmpty())
        mimeType = WebCore::MIMETypeRegistry::getMIMETypeForExtension(url.path().substring(url.path().reverseFind('.')+1));

    // total hack for now until I get plugin stuff factored correctly....
    if (WebCore::PluginInfoStore::isFlashMIMEType(mimeType))
        widget = platformCreateFlashPluginView(pluginSize, element, url, paramNames, paramValues, mimeType, this, loadManually);
    else {
        WTF::RefPtr<PluginImpl>
            pPluginImpl(PluginImpl::construct(element, url, paramNames, paramValues, mimeType, this));
        if (pPluginImpl)
        {
            widget = WebPluginWidget::construct(pPluginImpl);
        }
    }

    return widget.release();
#else
    return 0;
#endif
}

void WebFrameImpl::redirectDataToPlugin(WebCore::Widget* widget)
{
    ASSERT(!resourceHandleClientRedirect());
    //    TBD: in the case that a URL to a nonexistent local PDF was encountered, the plug-in is
    //    created from the extension before the non-existence of the PDF is known, and the
    //    redirection is never removed.   We need to figure out a place to remove it before
    //    we add this assert back in.  rsherry 2007.05.21
    // ASSERT(!m_didRedirectDataToPlugin);
    if (widget) {
    
        // The frame has detected that it's a plugin: we need to redirect to this widget, if
        // it wants it.
        WTF::RefPtr<WebCore::IPluginWidgetApolloImpl> const pluginWidgetImpl(widget->getApolloImpl()->toPluginWidgetImpl());
        ASSERT(pluginWidgetImpl);
        if (pluginWidgetImpl) {
            m_didRedirectDataToPlugin = true;
            //  If it does want the redirect, we also need to give it the original response
            //  structure we got (since we're past that now).
            WebCore::ResourceHandleClient* pluginRedirectClient = pluginWidgetImpl->pluginRedirectClient();
            if (pluginRedirectClient) {
                m_resourceHandleClientRedirect = pluginRedirectClient;

                WebCore::DocumentLoader *loader = frame()->loader()->activeDocumentLoader();
                const WebCore::ResourceResponse& response = loader->response();
                pluginRedirectClient->didReceiveResponse(0, response);
            }
        }
    }
}

WebCore::Widget* WebFrameImpl::createJavaAppletWidget(const WebCore::IntSize&, WebCore::Element*, const WebCore::KURL& baseURL, const WTF::Vector<WebCore::String>& paramNames, const WTF::Vector<WebCore::String>& paramValues)
{
    return 0;
}

WebCore::ObjectContentType WebFrameImpl::objectContentType(const WebCore::KURL& url, const WebCore::String& mimeTypeIn)
{
    WebCore::String mimeType = mimeTypeIn;
    if (mimeType.isEmpty()) {
        mimeType = WebCore::MIMETypeRegistry::getMIMETypeForExtension(url.path().substring(url.path().reverseFind('.')+1));
        if(mimeType.isEmpty())
            return WebCore::ObjectContentNone;
    }

    if (mimeType.isEmpty())
        return WebCore::ObjectContentFrame; // Go ahead and hope that we can display the content.

#ifdef PLUGINS_ENABLED
    //    If it's a PDF MIME type, we want ObjectContentPlugin... if we can show PDFs.
    if ( WebCore::PluginInfoStore::isPDFMIMEType(mimeType) )
    {
        //    Paranoia: these should never be nil.
        if (m_pWebHost && m_pWebHost->m_pVTable && m_pWebHost->m_pVTable->pdfCapability)
        {
            if (m_pWebHost->m_pVTable->pdfCapability(m_pWebHost) == 0)
                return WebCore::ObjectContentNetscapePlugin;
        }
    }
    else if (WebCore::PluginInfoStore::isFlashMIMEType(mimeType))
        return WebCore::ObjectContentNetscapePlugin;
#endif
    if (WebCore::MIMETypeRegistry::isSupportedImageMIMEType(mimeType))
        return WebCore::ObjectContentImage;

    if (WebCore::MIMETypeRegistry::isSupportedNonImageMIMEType(mimeType))
        return WebCore::ObjectContentFrame;

    return (WebCore::ObjectContentType)0;
}

WebCore::String WebFrameImpl::overrideMediaType() const
{
    return WebCore::String();
}

void WebFrameImpl::windowObjectCleared()
{
}

void WebFrameImpl::didPerformFirstNavigation() const
{
}

void WebFrameImpl::registerForIconNotification(bool)
{
}

void WebFrameImpl::documentCreated()
{
    ASSERT(!m_dead);
    if (m_pWebHost) {
        ASSERT(m_pWebHost->m_pVTable);
        ASSERT(m_pWebHost->m_pVTable->handleOnDocumentCreated);
        ASSERT((frame()->loader()->state() == WebCore::FrameStateCommittedPage) || (frame()->loader()->state() == WebCore::FrameStateComplete));
        setGeneralPasteboard(0);
        m_pWebHost->m_pVTable->handleOnDocumentCreated(m_pWebHost);
    }
}

void* WebFrameImpl::getPlatformInterpreterContext() const
{
    ASSERT(!m_dead);
    if (m_pWebHost) {
        ASSERT(m_pWebHost->m_pVTable);
        ASSERT(m_pWebHost->m_pVTable->getPlatformInterpreterContext);
        return m_pWebHost->m_pVTable->getPlatformInterpreterContext(m_pWebHost);
    }
    return 0;
}

WebResourceHandle* WebFrameImpl::startResourceHandle(WebCore::ResourceHandle* pResourceHandle, WebCore::ResourceHandleClient* pResourceHandleClient, const WebCore::ResourceRequest& request)
{
    ASSERT(!m_dead);
    ASSERT(m_pFrame);
    
    WTF::RefPtr<WebResourceRequestImpl> const pWebResourceRequestImpl(WebResourceRequestImpl::construct(request));
    WebResourceRequest* const pWebResourceRequest = pWebResourceRequestImpl->getWebResourceRequest();
    
    WebResourceHandleClientImpl* pWebResourceHandleClientImpl = new WebResourceHandleClientImpl(pResourceHandle, pResourceHandleClient);
    WebResourceHandleClient* const pWebResourceHandleClient = pWebResourceHandleClientImpl->getWebResourceHandleClient();
    
    WebResourceHandle* const retValue = m_pWebHost->m_pVTable->loadResource(m_pWebHost, pWebResourceRequest, pWebResourceHandleClient);                                
    return retValue;
}

void WebFrameImpl::loadResourceSynchronously(const WebCore::ResourceRequest& request, WebCore::ResourceError& error, WebCore::ResourceResponse& response, WTF::Vector<char>& data)
{
    ASSERT(!m_dead);
    ASSERT(m_pFrame);
    WebResourceLoaderSynchronousClientImpl::loadResourceSynchronously(m_pWebHost, request, error, response, data);
}

void WebFrameImpl::attachScrollbarWidgetImpl(WebCore::PlatformScrollbar* pPlatformScrollbar)
{
    ASSERT(!m_dead);
    
    // We need to pass in the parent IWidgetApolloImpl
    ASSERT(frame());
    WebCore::FrameView* frameView = frame()->view();
    ASSERT(frameView);
    WTF::PassRefPtr<WebCore::IScrollViewApolloImpl> parentScrollViewApolloImpl( frameView->getApolloImpl() );
    
    ASSERT(m_pOwningWebView);
    ASSERT(m_pOwningWebView->focusManager());
    ASSERT(m_pOwningWebView->mouseCaptureManager());
    WTF::RefPtr<WebScrollbarWidgetImpl> const pWidgetImpl(WebScrollbarWidgetImpl::construct(m_pOwningWebView->focusManager()
                                                                                           , m_pOwningWebView->mouseCaptureManager()
                                                                                           , pPlatformScrollbar));
    pPlatformScrollbar->attachApolloImpl(pWidgetImpl);
}

void WebFrameImpl::putFrameOnLifeSupport(WTF::PassRefPtr<WebCore::Frame> frame)
{
    ASSERT(frame);
    ASSERT(frame == m_pFrame);
    ASSERT(m_pOwningWebView);
    m_pOwningWebView->putFrameOnLifeSupport(frame);
}

bool WebFrameImpl::isChildFrame(const WebCore::AtomicString& childFrameName)
{
    ASSERT(m_pFrame);
    ASSERT(m_pFrame->tree());
    return m_pFrame->tree()->child(childFrameName) != 0;
}

void WebFrameImpl::windowObjectDead()
{
    WebCore::Frame* const theFrame = frame();
    ASSERT(theFrame);
    ASSERT(theFrame->scriptProxy());
    
    WebCore::JSDOMWindow* const jsDOMWindow = theFrame->scriptProxy()->globalObject();
    if (m_childDoor)
        WebCore::ApolloScriptBridging::markObjectProxyDead(m_childDoor, jsDOMWindow);
    m_childDoor = 0;
    m_childDoorProxy = 0;
    
    // The JS interpreter is cleared when ever a new load is committed,
    // we need to go through the script bridging tables and deadify all
    // bridging object referencing that interpreter.  The most scary case that
    // this fixes is the case of a JS object being registered on a AS3 event
    // dispatcher.  Imagine a navigate of the frame containing the JS object
    // followed by the AS3 event being dispatched.  If we don't deadify the JS
    // function object, the JS function will execute in the context of the new
    // html document, which could be in a different domain.
    WebCore::ApolloScriptBridging::notifyGlobalObjectDead(jsDOMWindow);
}

static const char *appResourceProtocol = "app";

bool WebFrameImpl::hasApplicationPrivileges() const
{
    // It's annoying that some code uses the document URL and some uses the frame url.
    // Adding assert to validate as much as possible that we have chosen the correct one for
    // our security check.
    ASSERT((frame()->loader()->url().isEmpty()) || (WebCore::equalIgnoringRef(frame()->loader()->url(), WebCore::KURL(frame()->document()->url()))));

    // Special privileges are given when the URL has the app scheme.
    WebCore::String frameProtocol = frame()->loader()->url().protocol();
    return frameProtocol.lower() == appResourceProtocol;
}

static bool shouldLoadAsEmptyDocument(const WebCore::KURL &url)
{
    return url.protocol().lower() == "about" || url.isEmpty();
}

bool WebFrameImpl::hasApplicationRestrictions() const
{
    // Special restrictions apply if the frame is priviliged or has scripting access to a privileged parent or owner.
    // See Window::isSafeScript
    if (hasApplicationPrivileges())
        return true;

    if (shouldLoadAsEmptyDocument(frame()->loader()->url())) {
        WebCore::Frame* ancestorFrame = frame()->loader()->opener() ? frame()->loader()->opener() : frame()->tree()->parent();
        while (ancestorFrame && shouldLoadAsEmptyDocument(ancestorFrame->loader()->url()))
            ancestorFrame = ancestorFrame->tree()->parent();
        if (ancestorFrame)
            if (ancestorFrame->loader()->client()->hasApplicationPrivileges())
                return true;
    }

    return false;
}

bool WebFrameImpl::canLoadScript(const WebCore::KURL& url) const
{
    if (hasApplicationPrivileges()) {
        // If this is inline script, only process it when parsing
        // otherwise, possible injection attack.
        if (url.isEmpty()) {
            if (frame()->document()->parsing())
                return true;
            if (frame()->document()->processingDOMContentLoadedEvent())
                return true;
            if (frame()->document()->processingLoadEvent())
                return true;
            return false;
        }

        // Allow application resource URL scripts.
        WebCore::String protocol = url.protocol();
        if (protocol.lower() == appResourceProtocol)
            return true;

        // But we want to deny any other URLs, period.
        return false;
    }

    // Otherwise whatever, but the canLoad constraints are enforced elsewhere
    return true;
}

bool WebFrameImpl::canSetStringTimer() const
{
    // Same as whether you can insert inline script
    return canLoadScript(WebCore::KURL());
}

KJS::EvalLevel WebFrameImpl::currentEvalLevel() const
{
    if (hasApplicationRestrictions())
    {
        // In application code, only allow full eval while loading that code. 
        if (frame()->document()->parsing() || frame()->document()->processingDOMContentLoadedEvent() || frame()->document()->processingLoadEvent() || loadingApplicationScript())
            return KJS::EvalAny;
        return KJS::EvalConstantExpr;
    }
    return KJS::EvalAny;
}

bool WebFrameImpl::loadingApplicationScript() const
{
    WebCore::KURL url(m_loadingScriptURL);
    if (url.protocol().lower() == appResourceProtocol)
        return true;
    return false;
}

bool WebFrameImpl::canConstructFunctions() const
{
    // Allow "new Function()" when eval not restricted.
    if (currentEvalLevel() >= KJS::EvalAny)
        return true;
    return false;
}

bool WebFrameImpl::isSynchronousRequestAllowed(const WebCore::KURL& url) const
{
    // Special rules for application content
    WebCore::KURL frameUrl = m_pFrame->document()->url();
    if (frameUrl.protocol().lower() == appResourceProtocol) {
        // Don't allow synchronous requests while full eval is allowed.
        if (currentEvalLevel() >= KJS::EvalAny) {
            // unless it's an application resource
            if (url.protocol().lower() != appResourceProtocol)
                return false;
        }
    }
    return true;
}

KJS::JSObject* WebFrameImpl::getRootPackage()
{
    WebScriptProxyVariant* const rootPackageVariant = m_pWebHost->m_pVTable->getRootPackage(m_pWebHost);
    ASSERT(rootPackageVariant);
    KJS::JSLock lock;
    KJS::JSValue* const rootPackageValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(rootPackageVariant);
    rootPackageVariant->m_pVTable->release(rootPackageVariant);
    ASSERT(rootPackageValue->isObject());
    return rootPackageValue->getObject();
}

KJS::JSObject* WebFrameImpl::getWKMethods()
{
    WebScriptProxyVariant* wkMethodsVariant = m_pWebHost->m_pVTable->getWKMethods(m_pWebHost);
    ASSERT(wkMethodsVariant);
    KJS::JSLock lock;
    KJS::JSValue* wkMethodsValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(wkMethodsVariant);
    wkMethodsVariant->m_pVTable->release(wkMethodsVariant);
    ASSERT(wkMethodsValue->isObject());
    return wkMethodsValue->getObject();
}

WebCore::ClipboardApolloHelper* WebFrameImpl::createDraggingClipboard()
{
    return createClipboard(true);
}

WebCore::ClipboardApolloHelper* WebFrameImpl::createGeneralClipboard()
{
    return createClipboard(false);
}

WebCore::Pasteboard* WebFrameImpl::getGeneralPasteboard()
{
    return m_generalPasteboard.get();
}

void WebFrameImpl::setGeneralPasteboard(WebCore::Pasteboard* pasteboard)
{
    m_generalPasteboard.set(pasteboard);
}

WebPopupWindow* WebFrameImpl::createWebPopupMenu(WebCore::PopupMenuClient* client, float computedFontSize)
{
    WebHost* const host = webHost();
    ASSERT(host);
    ASSERT(client);
    WebPopupWindowClientImpl* const webMenuClient = WebPopupWindowClientImpl::construct(host, client);
    return host->m_pVTable->createPopupWindow(host, webMenuClient->getWebPopupWindowClient(), computedFontSize);
}

WebCore::ClipboardApolloHelper* WebFrameImpl::createClipboard(bool forDragging)
{
    KJS::JSObject* wkMethods = getWKMethods();
    ASSERT(wkMethods);

    WebCore::JSDOMWindow* jsDOMWindow = frame()->page()->mainFrame()->scriptProxy()->globalObject();
    KJS::ExecState* exec = jsDOMWindow->globalExec();

    KJS::JSLock lock;

    KJS::JSObject* createClipboard = wkMethods->get(exec, KJS::Identifier("createClipboard"))->toObject(exec);

    KJS::List args;
    args.append(KJS::jsBoolean(true)); // writable
    args.append(KJS::jsBoolean(forDragging));
    if (!forDragging) {
        KJS::JSObject* rootPackage = getRootPackage();
        ASSERT(rootPackage);
        KJS::JSObject* flash = rootPackage->get(exec, KJS::Identifier("flash"))->toObject(exec);
        KJS::JSObject* desktop = flash->get(exec, KJS::Identifier("desktop"))->toObject(exec);
        KJS::JSObject* Clipboard = desktop->get(exec, KJS::Identifier("Clipboard"))->toObject(exec);
        KJS::JSObject* generalClipboard = Clipboard->get(exec, KJS::Identifier("generalClipboard"))->toObject(exec);
        args.append(generalClipboard);
    }

    KJS::JSObject* jsClipboardObject = createClipboard->callAsFunction(exec, wkMethods, args)->toObject(exec);
    return new WebCore::ClipboardApolloHelper(jsClipboardObject, exec);
}

WTF::PassRefPtr<WebCore::IWidgetApolloImpl> WebFrameImpl::createNPPluginImpl(WebCore::PluginView* ownerPluginView)
{
#ifdef PLUGINS_ENABLED
#if PLATFORM(DARWIN)
    return WebNPPluginWidgetImplMac::construct(ownerPluginView, frame()->view()->getApolloImpl().get(), webViewImpl());
#elif PLATFORM(WIN_OS)
    return WidgetApolloPluginWinImpl::construct(ownerPluginView, frame()->view()->getApolloImpl().get(), webViewImpl());
#elif PLATFORM(UNIX_OS)
    return WidgetApolloPluginUnixImpl::construct(ownerPluginView, frame()->view()->getApolloImpl().get(), webViewImpl());
#else
    return 0;
#endif
#else
    return 0;
#endif
}

void WebFrameImpl::updateLocation( const WebCore::KURL& newLocation )
{
    ASSERT(!m_dead);
    WebCore::CString const urlAsUTF8( newLocation.prettyURL().utf8() );
    ASSERT(m_pFrame);
    ASSERT(m_pFrame->loader());
    const char* const rawURLStrAsUTF8 = urlAsUTF8.data();
    if (!m_pFrame || !m_pFrame->loader() || m_pFrame->loader()->creatingInitialEmptyDocument())
        return;
    if (m_pWebHost)
    {
        m_pWebHost->m_pVTable->updateLocation(m_pWebHost, reinterpret_cast<const unsigned char*>(rawURLStrAsUTF8));
    }
}

void WebFrameImpl::clearResourceHandleRedirect()
{
    m_resourceHandleClientRedirect = NULL;
    m_didRedirectDataToPlugin = false;
}
WebCore::ResourceHandleClient* WebFrameImpl::resourceHandleClientRedirect() const
{
    return m_resourceHandleClientRedirect;
}
void WebFrameImpl::addPluginRootObject(WTF::RefPtr<KJS::Bindings::RootObject>& root)
{
    m_rootObjects.append(root);
}

void WebFrameImpl::cleanupPluginRootObjects()
{
    KJS::JSLock lock;

    unsigned count = m_rootObjects.size();
    for (unsigned i = 0; i < count; i++)
        m_rootObjects[i]->invalidate(); // XXX - used to be removeAllNativeReferences, perhaps instead the destructing of the object is enough

    m_rootObjects.clear();
}

namespace
{
    enum InvokedBy { InvokedBySelf, InvokedByParent, InvokedByOpener, InvokedByOther };
    static InvokedBy invokedBy(KJS::ExecState* exec, WebCore::Frame* frame)
    {
        if (frame) {
            WebCore::Frame* invokingFrame = static_cast<WebCore::JSDOMWindow*>(exec->dynamicGlobalObject())->impl()->frame();
            if (invokingFrame) {
                if (frame == invokingFrame)
                    return InvokedBySelf;
                if (frame->tree()->parent() == invokingFrame)
                    return InvokedByParent;
                if (frame->loader()->opener() == invokingFrame)
                    return InvokedByOpener;
            }
        }
        return InvokedByOther;
    }
}
KJS::JSValue* WebFrameImpl::getChildDoor(KJS::ExecState* exec) const
{
    switch (invokedBy(exec, m_pFrame)) {
    case InvokedBySelf:
        return m_childDoor ? m_childDoor.get() : KJS::jsUndefined();
    case InvokedByParent:
    case InvokedByOpener:
        return m_childDoorProxy ? m_childDoorProxy.get() : KJS::jsUndefined() ;
    default:
        ;
    }
    return KJS::jsUndefined();
}

void WebFrameImpl::setChildDoor(KJS::ExecState* exec, KJS::JSValue* value)
{
    switch (invokedBy(exec, m_pFrame)) {
    case InvokedBySelf:
        if (!value || value->isUndefined())
            m_childDoor = m_childDoorProxy = KJS::jsUndefined();
        else {
            m_childDoor = value;
            WebScriptProxyVariant* variant = WebCore::ApolloScriptBridging::getApolloVariantForJSValue(exec, value);
            ASSERT(variant);
            WebScriptProxyVariant* resultVariant = m_pWebHost->m_pVTable->makeDoor(m_pWebHost, variant);
            // the resultVariant can be null if we are destroying the WebView.
            if (resultVariant) {
                m_childDoorProxy = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(resultVariant);
                resultVariant->m_pVTable->release(resultVariant);
            }
            else {
                variant->m_pVTable->release(variant);
                m_childDoor = m_childDoorProxy = KJS::jsUndefined();
            }
        }
    default:
        ;
    }
}

KJS::JSValue* WebFrameImpl::getParentDoor(KJS::ExecState* exec) const
{
    switch (invokedBy(exec, m_pFrame)) {
    case InvokedBySelf:
        return m_parentDoorProxy ? m_parentDoorProxy.get() : KJS::jsUndefined();
    case InvokedByParent:
    case InvokedByOpener:
        return m_parentDoor ? m_parentDoor.get() : KJS::jsUndefined();
    default:
        ;
    }
    return KJS::jsUndefined();
}

void WebFrameImpl::setParentDoor(KJS::ExecState* exec, KJS::JSValue* value)
{
    switch (invokedBy(exec, m_pFrame)) {
    case InvokedByParent:
    case InvokedByOpener:
        if (!value || value->isUndefined())
            m_parentDoor = m_parentDoorProxy = KJS::jsUndefined();
        else {
            m_parentDoor = value;
            WebScriptProxyVariant* variant = WebCore::ApolloScriptBridging::getApolloVariantForJSValue(exec, value);
            ASSERT(variant);
            WebScriptProxyVariant* resultVariant = m_pWebHost->m_pVTable->makeDoor(m_pWebHost, variant);
            // the resultVariant can be null if we are destroying the WebView.
            if (resultVariant) {
                m_parentDoorProxy = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(resultVariant);
                resultVariant->m_pVTable->release(resultVariant);
            }
            else {
                variant->m_pVTable->release(variant);
                m_parentDoor = m_parentDoorProxy = KJS::jsUndefined();
            }
        }
    default:
        ;
    }
}

KJS::JSValue* WebFrameImpl::getHtmlLoader(KJS::ExecState* exec) const
{
    WebCore::Frame* invokingFrame = static_cast<WebCore::JSDOMWindow*>(exec->dynamicGlobalObject())->impl()->frame();
    if (!invokingFrame)
        return false;

    // Only app content can access the HTMLLoader
    WebCore::KURL invokingFrameUrl = invokingFrame->document()->url();
    if (invokingFrameUrl.protocol().lower() != appResourceProtocol)
        return KJS::jsUndefined();

    WebScriptProxyVariant* resultVariant = m_pWebHost->m_pVTable->getHtmlControl(m_pWebHost);
    if (!resultVariant)
        return KJS::jsUndefined();
    KJS::JSValue* const htmlLoaderValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(resultVariant);
    resultVariant->m_pVTable->release(resultVariant);
    return htmlLoaderValue;
}

WebCore::FrameLoaderClient::AllowXHR WebFrameImpl::allowCrossDomainXMLHttpRequest(const WebCore::KURL& url) const
{
    ASSERT(m_pFrame);
    if (!m_pFrame)
        return AllowXHRDefault;

    // Allow application content to access anything
    WebCore::KURL frameUrl = m_pFrame->document()->url();
    if (frameUrl.protocol().lower() == appResourceProtocol)
        return AllowXHRAlways;

    // If the parent frame's an application frame
    WebCore::Frame* parent = m_pFrame->tree()->parent();
    if (parent && WebCore::KURL(parent->document()->url()).protocol().lower() == appResourceProtocol) {
        // and the frame element has the magic attribute
        if (frameHasAllowCrossDomainXHRAttr(m_pFrame)) {
            // and the url is not local
            if (!m_pFrame->loader()->shouldTreatURLAsLocal(url))
                return AllowXHRAlways;
        }
    }
    
    //file is allowed to load file
    if(m_pFrame->loader()->shouldTreatURLAsLocal(frameUrl) && m_pFrame->loader()->shouldTreatURLAsLocal(url))
    	return AllowXHRDefault;

    // Don't allow non-app content to access files, app, or app-storage
    // This is to some extent redundant with what the caller checks.
    if (m_pFrame->loader()->shouldTreatURLAsLocal(url))
        return AllowXHRNever;
    if (url.protocol().lower() == appResourceProtocol)
        return AllowXHRNever;

    return AllowXHRDefault;
}

bool WebFrameImpl::canFrameBeClosedOrFocusedFrom(KJS::ExecState *exec) const
{
    if (!m_pFrame)
        return false;

    WebCore::Frame* invokingFrame = static_cast<WebCore::JSDOMWindow*>(exec->dynamicGlobalObject())->impl()->frame();
    if (!invokingFrame)
        return false;

    // Application content can do whatever it wants to
    WebCore::KURL invokingFrameUrl = invokingFrame->document()->url();
    if (invokingFrameUrl.protocol().lower() == appResourceProtocol)
        return true;

    // App content can't be closed by non-app content
    if (hasApplicationRestrictions())
        return false;

    // The Firefox/Safari behavior:
    //   Windows can be closed as long as they were opened via script (have an opener)
    if (m_pFrame->loader()->opener())
        return true;

    return false;
}

bool WebFrameImpl::canExecuteScriptURL() const
{
    // App content can't execute JavaScript links.
    if (!m_pFrame)
        return false;

    WebCore::Frame *frame = m_pFrame;

    WebCore::KURL frameUrl = frame->loader()->url();

    // A messy detail.  If this is an "empty" frame (perhaps loaded using document.open, document.write),
    // then check the parent or owner.
    if (hasApplicationRestrictions())
        return false;

    return true;
}

bool WebFrameImpl::isDebuggingEnabled() const
{
	ASSERT(WebKitApollo::g_HostFunctions);
	ASSERT(WebKitApollo::g_HostFunctions->isDebuggingEnabled);

	return WebKitApollo::g_HostFunctions->isDebuggingEnabled();
}

void WebFrameImpl::logWarning(WebCore::String msg) const 
{
	ASSERT(WebKitApollo::g_HostFunctions);
	ASSERT(WebKitApollo::g_HostFunctions->logWarning);

	WebKitApollo::g_HostFunctions->logWarning(msg.webString());
}

}
