/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WebDragClient.h"
#include "WebFrameImpl.h"
#include <ClipboardApollo.h>
#include <Frame.h>
#include <FrameView.h>
#include <Page.h>
#include <FrameLoaderClientApollo.h>
#include <apollo/proxy_obj.h>
#include <kjs/value.h>
#include <kjs_binding.h>
#include <kjs_proxy.h>

namespace WebKitApollo {

WebDragClient::WebDragClient()
{
}

WebDragClient::~WebDragClient()
{
}

void WebDragClient::willPerformDragDestinationAction(WebCore::DragDestinationAction, WebCore::DragData*)
{
}

void WebDragClient::willPerformDragSourceAction(WebCore::DragSourceAction, const WebCore::IntPoint&, WebCore::Clipboard*)
{
}

WebCore::DragDestinationAction WebDragClient::actionMaskForDrag(WebCore::DragData*)
{
    return (WebCore::DragDestinationAction) (WebCore::DragDestinationActionAny & ~WebCore::DragDestinationActionLoad);
}

WebCore::DragSourceAction WebDragClient::dragSourceActionMaskForPoint(const WebCore::IntPoint& windowPoint)
{
    return WebCore::DragSourceActionAny;
}

void WebDragClient::startDrag(WebCore::DragImageRef dragImage, const WebCore::IntPoint& dragImageOrigin, const WebCore::IntPoint& eventPos, WebCore::Clipboard* clipboard, WebCore::Frame* frame, bool linkDrag)
{
    m_frameProtector = frame;
    m_viewProtector = m_frameProtector->view();

    KJS::JSLock lock;

    WebCore::FrameLoaderClientApollo* clientApollo = WebCore::FrameLoaderClientApollo::clientApollo(frame);

    WebCore::JSDOMWindow* const globalObj(frame->page()->mainFrame()->scriptProxy()->globalObject());
    KJS::ExecState* exec = globalObj->globalExec();
    KJS::JSObject* wkMethods = clientApollo->getWKMethods();
    KJS::JSObject* doDrag = wkMethods->get(exec, KJS::Identifier("doDrag"))->toObject(exec);

    KJS::JSObject* jsClipboard = static_cast<WebCore::ClipboardApollo*>(clipboard)->dataTransfer();

    jsClipboard->put(exec, KJS::Identifier("effectAllowed"), KJS::jsString(clipboard->effectAllowed()));

    WebCore::IntSize dragImageOffset(dragImageOrigin - eventPos);

    WebScriptProxyVariant* dragImageVariant = NULL;
    KJS::JSValue* dragImageValue = KJS::jsNull();
    if (dragImage) {
        dragImageVariant = dragImage->m_pVTable->getBitmapVariant(dragImage);
        dragImageValue = WebCore::ApolloScriptBridging::jsValueFromBridgingVariant(dragImageVariant);
        if (linkDrag)
            dragImageOffset -= WebCore::IntSize(0, dragImage->m_pVTable->getHeight(dragImage));
    }

    KJS::List args;
    args.append(jsClipboard);
    args.append(dragImageValue);
    args.append(KJS::jsNumber(dragImageOffset.width()));
    args.append(KJS::jsNumber(dragImageOffset.height()));
    doDrag->callAsFunction(exec, wkMethods, args);

    // Decrement drag image variant reference count.
    if (dragImageVariant)
        dragImageVariant->m_pVTable->release(dragImageVariant);
}

void WebDragClient::dragEnded()
{
    m_frameProtector = 0;
    m_viewProtector = 0;
}

WebCore::DragImageRef WebDragClient::createDragImageForLink(WebCore::KURL& url, const WebCore::String& label, WebCore::Frame* frame)
{
    KJS::JSLock lock;

    WebCore::FrameLoaderClientApollo* clientApollo = WebCore::FrameLoaderClientApollo::clientApollo(frame);
    WebHost* webHost = clientApollo->webHost();

    WebCore::JSDOMWindow* const globalObj(frame->page()->mainFrame()->scriptProxy()->globalObject());
    KJS::ExecState* exec = globalObj->globalExec();
    KJS::JSObject* wkMethods = clientApollo->getWKMethods();
    KJS::JSObject* dragImageForLink = wkMethods->get(exec, KJS::Identifier("dragImageForLink"))->toObject(exec);

    KJS::List args;
    args.append(KJS::jsString(url));
    args.append(KJS::jsString(label));

    KJS::JSValue* bitmapValue = dragImageForLink->callAsFunction(exec, wkMethods, args);
    WebScriptProxyVariant* bitmapVariant = WebCore::ApolloScriptBridging::getApolloVariantForJSValue(exec, bitmapValue);
    WebBitmap* bitmap = webHost->m_pVTable->createBitmapFromBitmapDataObject(webHost, bitmapVariant);
    if (bitmapVariant)
        bitmapVariant->m_pVTable->release(bitmapVariant);
    return bitmap;
}

void WebDragClient::dragControllerDestroyed()
{
}

}
