/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WebChromeClient.h"
#include <FloatRect.h>
#include <IntRect.h>
#include <wtf/Assertions.h>
#include <CString.h>
#include <PlatformString.h>
#include <TextEncoding.h>
#include <WebKitApollo/WebKit.h>
#include <apollo/proxy_obj.h>
#include "WebViewImpl.h"
#include <Frame.h>
#include <FrameLoader.h>
#include <FrameLoadRequest.h>
#include "Page.h"

namespace WebKitApollo
{

extern WebKitAPIHostFunctions* g_HostFunctions;

WebChromeClient::WebChromeClient(WebWindow* const pWindow, WebHost* const pHost)
	: m_pWindow(pWindow)
	, m_pHost(pHost)
    , m_closeTimer(this, &WebChromeClient::onCloseTimer)
{
    SetDefaultWindowFeatures();
}

WebChromeClient::~WebChromeClient()
{
}

void WebChromeClient::chromeDestroyed()
{
}

void WebChromeClient::SetDefaultWindowFeatures()
{
    m_features.x = 0;
    m_features.xSet = false;
    m_features.y = 0;
    m_features.ySet = false;
    m_features.width = 0;
    m_features.widthSet = false;
    m_features.height = 0;
    m_features.heightSet = false;
    m_features.menuBarVisible = true;
    m_features.statusBarVisible = true;
    m_features.toolBarVisible = true;
    m_features.locationBarVisible = true;
    m_features.scrollbarsVisible = true;
    m_features.resizable = true;
    m_features.fullscreen = false;
    m_features.dialog = false;
}

void WebChromeClient::setWindowRect(const WebCore::FloatRect& windowRect)
{
    ASSERT(m_pWindow);
    ASSERT(m_pWindow->m_pVTable);
    WebFloatRect const newWindowRect = {   windowRect.x(),
                                           windowRect.location().y(),
                                           windowRect.location().x() + windowRect.width(),
                                           windowRect.location().y() + windowRect.height() };
    m_pWindow->m_pVTable->setWindowRect( m_pWindow, &newWindowRect );
}

WebCore::FloatRect WebChromeClient::windowRect()
{
    ASSERT(m_pWindow);
    ASSERT(m_pWindow->m_pVTable);
    ASSERT(m_pWindow->m_pVTable->getWindowRect);
    WebFloatRect windowRect = { 0, 0, 0, 0 };
    m_pWindow->m_pVTable->getWindowRect(m_pWindow, &windowRect);

    ASSERT(windowRect.m_left <= windowRect.m_right);
    ASSERT(windowRect.m_top <= windowRect.m_bottom);

    float const windowWidth = windowRect.m_right - windowRect.m_left;
    float const windowHeight = windowRect.m_bottom - windowRect.m_top;
    
    WebCore::FloatRect const frameGeometry(windowRect.m_left, windowRect.m_top, windowWidth, windowHeight);
	return frameGeometry;
}


WebCore::FloatRect WebChromeClient::pageRect()
{
    ASSERT(m_pWindow);
    ASSERT(m_pWindow->m_pVTable);
    ASSERT(m_pWindow->m_pVTable->getPageRect);
    WebFloatRect pageRect = { 0, 0, 0, 0 };
    m_pWindow->m_pVTable->getPageRect(m_pWindow, &pageRect);

    ASSERT(pageRect.m_left <= pageRect.m_right);
    ASSERT(pageRect.m_top <= pageRect.m_bottom);

    float const pageWidth = pageRect.m_right - pageRect.m_left;
    float const pageHeight = pageRect.m_bottom - pageRect.m_top;
    
    WebCore::FloatRect const pageGeometry(pageRect.m_left, pageRect.m_top, pageWidth, pageHeight);
	return pageGeometry;
}


float WebChromeClient::scaleFactor()
{
	return 1;
}


void WebChromeClient::focus()
{
    ASSERT(m_pWindow);
    m_pWindow->m_pVTable->focus(m_pWindow);
}

void WebChromeClient::unfocus()
{
    ASSERT(m_pWindow);
    m_pWindow->m_pVTable->unfocus(m_pWindow);
}


bool WebChromeClient::canTakeFocus(WebCore::FocusDirection)
{
    ASSERT(m_pWindow);
    return (m_pWindow->m_pVTable->canTakeFocusFromWebKit(m_pWindow) != 0);
}

void WebChromeClient::takeFocus(WebCore::FocusDirection)
{
    ASSERT(m_pWindow);
    m_pWindow->m_pVTable->takeFocusFromWebKit(m_pWindow);
}

WebCore::Page* WebChromeClient::createWindow(WebCore::Frame* frame, const WebCore::FrameLoadRequest& loadRequest, const WebCore::WindowFeatures& features)
{
	if (m_pHost) {
        ASSERT(m_pHost->m_pVTable);
        ASSERT(m_pHost->m_pVTable->createNewWindow);

		// Copy features into binary-stable structure.
		WebHostCreateWindowArgs createArgs;
		memset(&createArgs, 0, sizeof createArgs);
		createArgs.windowArgsSize = sizeof(WebHostCreateWindowArgs);
		createArgs.x = features.xSet ? int(features.x) : WebHostCreateWindowArgs::USE_DEFAULT;
		createArgs.y = features.ySet ? int(features.y) : WebHostCreateWindowArgs::USE_DEFAULT;
		createArgs.width = features.widthSet ? int(features.width) : WebHostCreateWindowArgs::USE_DEFAULT;
		createArgs.height = features.heightSet ? int(features.height) : WebHostCreateWindowArgs::USE_DEFAULT;
		createArgs.menuBarVisible = features.menuBarVisible;
		createArgs.statusBarVisible = features.statusBarVisible;
		createArgs.toolBarVisible = features.toolBarVisible;
		createArgs.locationBarVisible = features.locationBarVisible;
		createArgs.scrollBarsVisible = features.scrollbarsVisible;
		createArgs.resizable = features.resizable;
		createArgs.fullscreen = features.fullscreen;

        WebView *webView = m_pHost->m_pVTable->createNewWindow(m_pHost, &createArgs);
        if (webView != NULL)
        {
            WebViewImpl* const webViewImpl = WebViewImpl::getImpl(webView);
            if (!loadRequest.isEmpty())
                webViewImpl->mainFrame()->loader()->load(loadRequest.resourceRequest());

			WebCore::Page *page = webViewImpl->page();

            static_cast<WebChromeClient*>(page->chrome()->client())->m_features = features;
			return page;       
        }
    }
	return NULL;
}

void WebChromeClient::show()
{
	// Not needed for Apollo
}


bool WebChromeClient::canRunModal()
{
	return false;
}

void WebChromeClient::runModal()
{
}


void WebChromeClient::setToolbarsVisible(bool bVisible)
{
}

bool WebChromeClient::toolbarsVisible()
{
    return m_features.toolBarVisible;
}


void WebChromeClient::setStatusbarVisible(bool bVisible)
{
}

bool WebChromeClient::statusbarVisible()
{
    return m_features.statusBarVisible;
}


void WebChromeClient::setScrollbarsVisible(bool bVisible)
{
}

bool WebChromeClient::scrollbarsVisible()
{
    return m_features.scrollbarsVisible;
}


void WebChromeClient::setMenubarVisible(bool bVisible)
{
}

bool WebChromeClient::menubarVisible()
{
    return m_features.menuBarVisible;
}


void WebChromeClient::setResizable(bool bResizable)
{
}


void WebChromeClient::addMessageToConsole(const WebCore::String& message, unsigned int lineNumber, const WebCore::String& sourceID)
{
}


bool WebChromeClient::canRunBeforeUnloadConfirmPanel()
{
	return false;
}

bool WebChromeClient::runBeforeUnloadConfirmPanel(const WebCore::String& message, WebCore::Frame* frame)
{
	return false;
}


void WebChromeClient::closeWindowSoon()
{
    m_closeTimer.startOneShot(0);
}


void WebChromeClient::runJavaScriptAlert(WebCore::Frame*, const WebCore::String& message)
{
    ASSERT(m_pHost);
    WebCore::CString messageAsUTF8(message.utf8());
    m_pHost->m_pVTable->runJSAlert(m_pHost, messageAsUTF8.data());
}

bool WebChromeClient::runJavaScriptConfirm(WebCore::Frame*, const WebCore::String& message)
{
    ASSERT(m_pHost);
	WebCore::CString messageAsUTF8(message.utf8());
	return m_pHost->m_pVTable->runJSConfirm(m_pHost, messageAsUTF8.data()) != 0;
}

bool WebChromeClient::runJavaScriptPrompt(WebCore::Frame*, const WebCore::String& prompt, const WebCore::String& defaultValue, WebCore::String& resultStr)
{
    ASSERT(m_pHost);
    WebCore::CString promptAsUTF8(prompt.utf8());
    WebCore::CString defaultValueAsUTF8(defaultValue.utf8());
    char* pResultUTF8 = 0;
    bool result = m_pHost->m_pVTable->runJSPrompt(m_pHost, promptAsUTF8.data(), defaultValueAsUTF8.data(), &pResultUTF8) != 0;
    if (result) {
        ASSERT(pResultUTF8);
        const WebCore::TextEncoding& utf8Encoding = WebCore::UTF8Encoding();
        size_t const resultByteLen = strlen(pResultUTF8);
        resultStr = utf8Encoding.decode(pResultUTF8, resultByteLen);

        ASSERT(g_HostFunctions);
        ASSERT(g_HostFunctions->freeBytes);
        g_HostFunctions->freeBytes( pResultUTF8 );
    }

	return result;
}


void WebChromeClient::setStatusbarText(const WebCore::String& statusText)
{
	if (m_pHost) {
		ASSERT(sizeof(uint16_t) == sizeof(UChar));
		const uint16_t* const statusUTF16 = reinterpret_cast<const uint16_t*>(statusText.characters());
		unsigned long const numStatusCodeUnits = statusText.length();
        m_pHost->m_pVTable->setStatusText(m_pHost, statusUTF16, numStatusCodeUnits);
    }
}

bool WebChromeClient::shouldInterruptJavaScript()
{
	return false;
}

bool WebChromeClient::tabsToLinks() const
{
	return true;
}

WebCore::IntRect WebChromeClient::windowResizerRect() const
{
	return WebCore::IntRect();
}

void WebChromeClient::addToDirtyRegion(const WebCore::IntRect&)
{
}

void WebChromeClient::scrollBackingStore(int dx, int dy, const WebCore::IntRect& scrollViewRect, const WebCore::IntRect& clipRect)
{
}

void WebChromeClient::updateBackingStore()
{
}

void WebChromeClient::mouseDidMoveOverElement(const WebCore::HitTestResult&, unsigned)
{
}

void WebChromeClient::setToolTip(const WebCore::String&)
{
}

void WebChromeClient::print(WebCore::Frame*)
{
}

void WebChromeClient::uncaughtJavaScriptException(const KJS::ExecState* const exec, KJS::JSValue* const pExceptionValue)
{
    if(m_pHost)
    {
        WebScriptProxyVariant* const pExceptionVariant = WebCore::ApolloScriptBridging::getApolloVariantForJSValue(exec, pExceptionValue);
        ASSERT(pExceptionVariant);
        if (pExceptionVariant)
        {
            ASSERT(m_pHost);
            ASSERT(m_pHost->m_pVTable);
            ASSERT(m_pHost->m_pVTable->uncaughtJSException);
            m_pHost->m_pVTable->uncaughtJSException(m_pHost, pExceptionVariant);
            ASSERT(pExceptionVariant->m_pVTable);
            ASSERT(pExceptionVariant->m_pVTable->release);
            pExceptionVariant->m_pVTable->release(pExceptionVariant);
        }
    }
}

void WebChromeClient::onCloseTimer(WebCore::Timer<WebChromeClient>* pTimer)
{
    ASSERT(m_pHost);
    ASSERT(m_pHost->m_pVTable);
    ASSERT(m_pHost->m_pVTable->closeWindow);
    ASSERT(pTimer == &m_closeTimer);
    (void)pTimer;
    m_pHost->m_pVTable->closeWindow(m_pHost);
}

bool WebChromeClient::processingUserGesture()
{
    ASSERT(m_pWindow);
    ASSERT(m_pWindow->m_pVTable);
    ASSERT(m_pWindow->m_pVTable->processingUserGesture);
    return m_pWindow->m_pVTable->processingUserGesture(m_pWindow) != 0;
}

void WebChromeClient::startWindowedPluginUserGesture()
{
    ASSERT(m_pWindow);
    ASSERT(m_pWindow->m_pVTable);
    ASSERT(m_pWindow->m_pVTable->startProcessingWindowedPluginUserGesture);
    m_pWindow->m_pVTable->startProcessingWindowedPluginUserGesture(m_pWindow);
}

void WebChromeClient::endWindowedPluginUserGesture()
{
    ASSERT(m_pWindow);
    ASSERT(m_pWindow->m_pVTable);
    ASSERT(m_pWindow->m_pVTable->endProcessingWindowedPluginUserGesture);
    m_pWindow->m_pVTable->endProcessingWindowedPluginUserGesture(m_pWindow);
}

}
