/*
 * Copyright (C) 2006, 2007 Apple Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "PluginView.h"

#include "Document.h"
#include "DocumentLoader.h"
#include "Element.h"
#include "EventNames.h"
#include "FrameLoader.h"
#include "FrameLoadRequest.h"
#include "FrameTree.h"
#include "FrameView.h"
#include "GraphicsContext.h"
#include "Image.h"
#include "HTMLNames.h"
#include "HTMLPlugInElement.h"
#include "KeyboardEvent.h"
#include "MouseEvent.h"
#include "NotImplemented.h"
#include "Page.h"
#include "PlatformMouseEvent.h"
#include "PlatformKeyboardEvent.h"
#include "kjs_binding.h"
#include "kjs_proxy.h"
#include "PluginDebug.h"
#include "PluginPackage.h"
#include "PluginStream.h"
#include "npruntime_impl.h"
#include "npapi.h"
#include "runtime_root.h"
#include "Settings.h"
#include "kjs/JSLock.h"
#include "kjs/value.h"

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <X11/keysymdef.h>
#include <qt/KeyboardCodes.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#if PLATFORM(APOLLO)
#define XEMBED_SUPPORTED 1
#include <IScrollViewApolloImpl.h>
#include "FrameLoaderClientApollo.h"
#include <WebKitApollo/WebKit.h>
namespace WebKitApollo { extern WebKitAPIHostFunctions* g_HostFunctions; }
#endif

#define PRINT_WINDOW_SIZES 0

using KJS::ExecState;
using KJS::Interpreter;
using KJS::JSLock;
using KJS::JSObject;
using KJS::JSValue;
using KJS::UString;

namespace WebCore {

using namespace EventNames;
using namespace HTMLNames;

void printWidgetHierarchy(GtkWidget *widget);

class PluginRequestUnix {
public:
    PluginRequestUnix(const FrameLoadRequest& frameLoadRequest, bool sendNotification, void* notifyData)
        : m_frameLoadRequest(frameLoadRequest)
        , m_notifyData(notifyData)
        , m_sendNotification(sendNotification) { }
public:
    const FrameLoadRequest& frameLoadRequest() const { return m_frameLoadRequest; }
    void* notifyData() const { return m_notifyData; }
    bool sendNotification() const { return m_sendNotification; }
private:
    FrameLoadRequest m_frameLoadRequest;
    void* m_notifyData;
    bool m_sendNotification;
    // TODO_AIRLinux: user gesture
};

static GtkWidget* GetChildWindow(GtkWidget* parentWindow)
{
    if(!GTK_IS_CONTAINER(parentWindow))
        return NULL;

    GtkWidget *child;
    GList *children = gtk_container_get_children(GTK_CONTAINER(parentWindow));
    if(children)
    {
        GList *firstChild = children;
        if(firstChild)
        {
            child = (GtkWidget*)(firstChild->data);
            g_list_free(children);
            return child;
        }
    }
    return NULL;
} 


void PluginView::updateWindow() const
{
    if(!getApolloImpl()->isInVisibleViewHierarchy())
        return;

    FrameView* frameView = m_parentFrame->view();

    IntRect oldWindowRect = m_windowRect;
    IntRect oldClipRect = m_clipRect;

    m_windowRect = IntRect(frameView->contentsToWindow(frameGeometry().location()), frameGeometry().size());

#if PLATFORM(APOLLO)
    if(m_window)
    {
        WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();

        IntPoint p = impl->viewportToWindow( m_windowRect.location() );
        m_windowRect = IntRect(p, m_windowRect.size());
    }
#endif

    m_clipRect = windowClipRect();

    if(!m_isWindowed || !m_isVisible)
        return;

    if (m_window) {
	setCallingPlugin(true);

#if PRINT_WINDOW_SIZES
        printf("m_windowRect    %d   %d   %d   %d\n", m_windowRect.x(), m_windowRect.y(), m_windowRect.width(), m_windowRect.height());
        printf("m_clipRect    %d   %d   %d   %d\n", m_clipRect.x(), m_clipRect.y(), m_clipRect.width(), m_clipRect.height());
#endif

        // Get the child window (drawing area inside the gtk plug)
        GtkWidget* gtkChild = GetChildWindow(m_window);
        if(gtkChild)
            gtkChild = GetChildWindow(gtkChild);

        if(gtkChild)
        {
            int leftClip = m_clipRect.x() - m_windowRect.x();
            int topClip = m_clipRect.y() - m_windowRect.y();

#if PRINT_WINDOW_SIZES
            printf("flash window   %d %d   %d   %d\n",  -leftClip, -topClip, m_windowRect.width(), m_windowRect.height());
#endif
            gdk_window_move_resize(gtkChild->window, -leftClip, -topClip, m_windowRect.width(), m_windowRect.height());
        }

       /* We must set this window's size requisition.  Otherwise when a
           resize is queued (when gtk_widget_queue_resize is called) the
           window will snap to its default requisition of 0x0.  If we omit
           this call, Frames and Dialogs shrink to degenerate 1x1 windows
           when their resizable property changes. */
        // TODO_AIRLinux : gtk_widget_set_uposition is deprecated. Change this.
        if(GTK_IS_WIDGET(m_window))
        {
            gtk_widget_set_size_request(m_window, m_clipRect.width(), m_clipRect.height());
            if(m_clipRect.width() && m_clipRect.height())
                gtk_widget_set_uposition(m_window, m_clipRect.x(), m_clipRect.y());
        }

	setCallingPlugin(false);
    }
}

void PluginView::setFocus()
{
    if (m_window)
        gtk_widget_grab_focus(m_window);

    Widget::setFocus();
}

void PluginView::show()
{
#if PLATFORM(APOLLO)
    init();
    createWindow();
#endif

    // TODO_AIRLinux: Here check for windowed plugin
    // Compare it with PluginViewWin.cpp's implementation

#if PLATFORM(APOLLO)
    if(m_isEnabled)
#endif
    { 
        if (m_window && GTK_IS_WIDGET(m_window))
            gtk_widget_show(m_window);
    }

    m_isVisible = true;
    Widget::show();
}

void PluginView::hide()
{
    m_isVisible = false;

    if (m_window)
        gtk_widget_hide(m_window);

    Widget::hide();
}

static bool isUnixUserGesture(int message)
{
    const int KeyPress = 2;
    const int KeyRelease = 3;

    switch (message) {
        case KeyPress:
        case KeyRelease:
        case ButtonPress:
        case ButtonRelease:
            return true;
        default:
            return false;
    }
}

void PluginView::paintMissingPluginIcon(GraphicsContext* context, const IntRect& rect)
{
    static Image* nullPluginImage;
    if (!nullPluginImage)
        nullPluginImage = Image::loadPlatformResource("nullPlugin");

    IntRect imageRect(frameGeometry().x(), frameGeometry().y(), nullPluginImage->width(), nullPluginImage->height());

    int xOffset = (frameGeometry().width() - imageRect.width()) / 2;
    int yOffset = (frameGeometry().height() - imageRect.height()) / 2;

    imageRect.move(xOffset, yOffset);

    if (!rect.intersects(imageRect))
        return;

    context->save();
    context->clip(windowClipRect());
    context->drawImage(nullPluginImage, imageRect.location());
    context->restore();
}

#if PLATFORM(APOLLO)
void PluginView::enableOrSuppress(bool enable)
{
    enable = enable && getApolloImpl()->isInVisibleViewHierarchy();

    if(enable == m_isEnabled)
        return;

    m_isEnabled = enable;

    // This prevents the widget from showing up again when the view is made opaque again.
    // Bug 2259148
/*
    // we don't need to do anything if the window is hidden
    if(!m_isVisible)
        return;
*/

    if(m_isEnabled)
    {
        show(); // it's imperative we call this as this is where we do the reparenting

        if(!m_isWindowed && getApolloImpl()->getParent())
            invalidate();
    }
    else
    {
        hide();

        // hide() set visible to false
        // XXX - TODO: for the future restructure this so the guts of show() and hide() are done in this function
        // m_isVisible = true;
    }
}
#endif


IntPoint PluginView::clientToWindow(const IntPoint &p) const
{
    // TODO_AIRLinux : Fix the commented lines for Unix

    /*
    // translate from ClientRect to WindowRect
    HWND hwnd = m_parentFrame->view()->containingWindow();

    RECT windowScreenRect;

    // screen coords of window rect
    GetWindowRect(hwnd, &windowScreenRect);

    POINT pt = { 0, 0 };

    ClientToScreen(hwnd, &pt);

    return IntPoint(p.x() + (pt.x - windowScreenRect.left),
                    p.y() + (pt.y - windowScreenRect.top) );
    */
    //return IntPoint(0, 0);  // TODO_AIRLinux - Remove this line
    return p;
}

IntPoint PluginView::viewportToWindow(const IntPoint &pIn) const
{
    IntPoint p = pIn;

    WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();
    p = impl->viewportToWindow( p );

    return m_isWindowed ? p : clientToWindow(p);
}

void PluginView::paint(GraphicsContext* context, const IntRect& rect)
{
    if (!m_isStarted) {
        // Draw the "missing plugin" image
        paintMissingPluginIcon(context, rect);
        return;
    }

    if (m_isWindowed || context->paintingDisabled() || !m_isVisible || !m_isEnabled)
        return;

    IntPoint p = m_parentFrame->view()->contentsToWindow(frameGeometry().location());

#if PLATFORM(APOLLO)
    WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();

    // get point in HTML drawable space
    IntPoint htmlDrawablePoint = impl->viewportToWindow(p, true);
    IntRect rectInWindow(htmlDrawablePoint, frameGeometry().size());
#else
    IntRect rectInWindow(p, frameGeometry().size());
#endif

    GdkDrawable* gdkdrawable = context->getDrawable(rectInWindow);

    m_npWindow.type = NPWindowTypeDrawable;
    m_npWindow.window = (void*)GDK_WINDOW_XWINDOW(gdkdrawable);

    setNPWindowRect(frameGeometry());

    NPEvent npEvent_paint;
    memset(&npEvent_paint, 0, sizeof(NPEvent));
    npEvent_paint.type = GraphicsExpose;
    npEvent_paint.xgraphicsexpose.display = GDK_WINDOW_XDISPLAY(gdkdrawable);
    npEvent_paint.xgraphicsexpose.drawable = GDK_WINDOW_XID(gdkdrawable);

    npEvent_paint.xgraphicsexpose.x = m_windowRect.x() > 0 ? 0 : (0 - m_windowRect.x());
    npEvent_paint.xgraphicsexpose.y = m_windowRect.y() > 0 ? 0 : (0 - m_windowRect.y());

    if((m_clipRect.x() > 0) && (m_clipRect.y() > 0))
    {
        int extraWidth = m_clipRect.x() - (m_windowRect.x() > 0 ?  m_windowRect.x() : 0);
        int extraHeight = m_clipRect.y() - (m_windowRect.y() > 0 ?  m_windowRect.y() : 0);

        npEvent_paint.xgraphicsexpose.width = m_clipRect.width() + extraWidth;
        npEvent_paint.xgraphicsexpose.height = m_clipRect.height() + extraHeight;
    }
    else
    {
        npEvent_paint.xgraphicsexpose.width = m_windowRect.width();
        npEvent_paint.xgraphicsexpose.height = m_windowRect.height();
    }

/*
    fprintf(stderr, "frameGeometry    %d   %d   %d   %d\n", frameGeometry().x(), frameGeometry().y(), frameGeometry().width(), frameGeometry().height());
    fprintf(stderr, "m_windowRect    %d   %d   %d   %d\n", m_windowRect.x(), m_windowRect.y(), m_windowRect.width(), m_windowRect.height());
    fprintf(stderr, "m_clipRect    %d   %d   %d   %d\n", m_clipRect.x(), m_clipRect.y(), m_clipRect.width(), m_clipRect.height());
    fprintf(stderr, "=== expose    %d   %d   %d   %d\n", npEvent_paint.xgraphicsexpose.x, npEvent_paint.xgraphicsexpose.y, npEvent_paint.xgraphicsexpose.width, npEvent_paint.xgraphicsexpose.height);
    fprintf(stderr, "----------------------------------------\n");
*/

    dispatchNPEvent(npEvent_paint);

    context->releaseDrawable(rectInWindow);
}

bool PluginView::dispatchNPEvent(NPEvent& npEvent)
{
    if (!m_plugin->pluginFuncs()->event)
        return true;

    bool shouldPop = false;

    if (m_plugin->pluginFuncs()->version < NPVERS_HAS_POPUPS_ENABLED_STATE && isUnixUserGesture(npEvent.type)) {
        pushPopupsEnabledState(true);
        shouldPop = true;
    }    

    KJS::JSLock::DropAllLocks dropAllLocks;
    setCallingPlugin(true);
    bool result = m_plugin->pluginFuncs()->event(m_instance, &npEvent);
    setCallingPlugin(false);

    if (shouldPop) 
        popPopupsEnabledState();

    return result;
}

void PluginView::handleKeyboardEvent(KeyboardEvent* event)
{
    GtkWidget *cwindow = m_parentFrame->view()->containingWindow();
    if(!cwindow)
        return;

   /*
    * TODO_AIRLinux : See if we can fix this
    *  
    * Reason for hardcoding KeyPress and KeyRelease values below:
    *   Following is the inclusion hierarchy of headers:
    *   ../../../../../Content/html/WebCore/plugins/PluginStream.h
    *      ../../../../../Content/html/WebCore/bridge/npruntime_internal.h
    *        ../../../../../Content/html/WebCore/bridge/npapi.h  (includes X11/X.h that defines KeyPress and KeyRelease)
    *   npruntime_internal.h after including npapi.h undefines the KeyPress and KeyRelease macros 
    *   (see npruntime_internal.h for comments)
    */
    const int KeyPress = 2;
    const int KeyRelease = 3;

    unsigned xkeysym = 0;
    switch(event->keyCode())
    {
        case VK_BACK:		xkeysym = XK_BackSpace; break;
        case VK_TAB:		xkeysym = XK_Tab; break;
        case VK_CLEAR:		xkeysym = XK_Clear; break;
        case VK_RETURN:		xkeysym = XK_Return; break;
        case VK_SHIFT:		xkeysym = XK_Shift_Lock; break;
        case VK_CONTROL:	xkeysym = XK_Control_L; break;
        case VK_MENU:		xkeysym = XK_Menu; break;
        case VK_PAUSE:		xkeysym = XK_Pause; break;
        case VK_CAPITAL:	xkeysym = XK_Caps_Lock; break;
        case VK_KANA:		xkeysym = XK_Kana_Lock; break;
        //case VK_HANGUL:	xkeysym = XK_Hangul; break;
        //case VK_HANJA:	xkeysym = XK_Hangul_Hanja; break;
        case VK_KANJI:		xkeysym = XK_Kanji; break;
        case VK_ESCAPE:		xkeysym = XK_Escape; break;
        case VK_MODECHANGE:	xkeysym = XK_Mode_switch; break;
        case VK_SPACE:		xkeysym = XK_space; break;
        case VK_PRIOR:		xkeysym =XK_Prior; break;
        case VK_NEXT:		xkeysym =XK_Next; break;
        case VK_END:		xkeysym =XK_End; break;
        case VK_HOME:		xkeysym =XK_Home; break;
        case VK_LEFT:		xkeysym =XK_Left; break;
        case VK_UP:		xkeysym =XK_Up; break;
        case VK_RIGHT:		xkeysym =XK_Right; break;
        case VK_DOWN:		xkeysym =XK_Down; break;
        case VK_PRINT:		xkeysym =XK_Print; break;
        case VK_INSERT:		xkeysym =XK_Insert; break;
        case VK_DELETE:		xkeysym =XK_Delete; break;
        case VK_NUMPAD0:	xkeysym =XK_KP_0; break;
        case VK_NUMPAD1:	xkeysym =XK_KP_1; break;
        case VK_NUMPAD2:	xkeysym =XK_KP_2; break;
        case VK_NUMPAD3:	xkeysym =XK_KP_3; break;
        case VK_NUMPAD4:	xkeysym =XK_KP_4; break;
        case VK_NUMPAD5:	xkeysym =XK_KP_5; break;
        case VK_NUMPAD6:	xkeysym =XK_KP_6; break;
        case VK_NUMPAD7:	xkeysym =XK_KP_7; break;
        case VK_NUMPAD8:	xkeysym =XK_KP_8; break;
        case VK_NUMPAD9:	xkeysym =XK_KP_9; break;
        case VK_MULTIPLY:	xkeysym =XK_KP_Multiply; break;
        case VK_ADD:		xkeysym =XK_KP_Add; break;
        case VK_SEPARATOR:	xkeysym =XK_KP_Separator; break;
        case VK_SUBTRACT:	xkeysym =XK_KP_Subtract; break;
        case VK_DECIMAL:	xkeysym =XK_KP_Decimal; break;
        case VK_DIVIDE:		xkeysym =XK_KP_Divide; break;
        case VK_F1:		xkeysym = XK_F1; break;
        case VK_F2:		xkeysym = XK_F2; break;
        case VK_F3:		xkeysym = XK_F3; break;
        case VK_F4:		xkeysym = XK_F4; break;
        case VK_F5:		xkeysym = XK_F5; break;
        case VK_F6:		xkeysym = XK_F6; break;
        case VK_F7:		xkeysym = XK_F7; break;
        case VK_F8:		xkeysym = XK_F8; break;
        case VK_F9:		xkeysym = XK_F9; break;
        case VK_F10:		xkeysym = XK_F10; break;
        case VK_F11:		xkeysym = XK_F11; break;
        case VK_F12:		xkeysym = XK_F12; break;
        case VK_F13:		xkeysym = XK_F13; break;
        case VK_F14:		xkeysym = XK_F14; break;
        case VK_F15:		xkeysym = XK_F15; break;
        case VK_F16:		xkeysym = XK_F16; break;
        case VK_F17:		xkeysym = XK_F17; break;
        case VK_F18:		xkeysym = XK_F18; break;
        case VK_F19:		xkeysym = XK_F19; break;
        case VK_F20:		xkeysym = XK_F20; break;
        case VK_F21:		xkeysym = XK_F21; break;
        case VK_F22:		xkeysym = XK_F22; break;
        case VK_F23:		xkeysym = XK_F23; break;
        case VK_F24:		xkeysym = XK_F24; break;
        case VK_NUMLOCK:	xkeysym =XK_Num_Lock; break;
        case VK_SCROLL:		xkeysym =XK_Scroll_Lock; break;
        case VK_LSHIFT:		xkeysym =XK_Shift_L; break;
        case VK_RSHIFT:		xkeysym =XK_Shift_R; break;
        case VK_LCONTROL:	xkeysym =XK_Control_L; break;
        case VK_RCONTROL:	xkeysym =XK_Control_R; break;
        case VK_OEM_COMMA:	xkeysym = XK_comma; break;
        case VK_OEM_MINUS:	xkeysym = XK_minus; break;
        case VK_OEM_PERIOD:	xkeysym = XK_period; break;
        case VK_OEM_PLUS:	xkeysym = XK_plus; break;
        case VK_OEM_1:		xkeysym = XK_semicolon; break;
        case VK_OEM_2:		xkeysym = XK_slash; break;
	case VK_OEM_3:		xkeysym = XK_asciitilde; break;
        case VK_OEM_4:		xkeysym = XK_bracketleft; break;
        case VK_OEM_5:		xkeysym = XK_backslash; break;
        case VK_OEM_6:		xkeysym = XK_bracketright; break;
        case VK_OEM_7:		xkeysym = XK_quotedbl; break;
        default:		xkeysym = event->keyCode(); break;
    }

    NPEvent npEvent;
    memset(&npEvent, 0, sizeof(NPEvent));

    npEvent.xkey.keycode = XKeysymToKeycode(GDK_WINDOW_XDISPLAY(cwindow->window), xkeysym);

    if(event->shiftKey())
        npEvent.xkey.state |= ShiftMask;
    if(event->ctrlKey())
        npEvent.xkey.state |= ControlMask;
    if(event->altKey())
        npEvent.xkey.state |= Mod3Mask;
    if(event->metaKey())
        npEvent.xkey.state |= Mod1Mask;
    if(event->keyEvent()->currentCapsLockState())
        npEvent.xkey.state |= LockMask;

    npEvent.xkey.display = GDK_WINDOW_XDISPLAY(cwindow->window);
    npEvent.xkey.window = GDK_WINDOW_XID(cwindow->window);

    if (event->type() == EventNames::keydownEvent) {
        npEvent.type = KeyPress;
        npEvent.xkey.type = KeyPress;
    } else if (event->type() == EventNames::keyupEvent) {
        npEvent.type = KeyRelease;
        npEvent.xkey.type = KeyRelease;
    }

    startUserGesture();

    KJS::JSLock::DropAllLocks dropLocks;
    if (!dispatchNPEvent(npEvent))
        event->setDefaultHandled();
}

void PluginView::handleMouseEvent(MouseEvent* event)
{
    NPEvent npEvent;
    memset(&npEvent, 0, sizeof(NPEvent));

    IntPoint p = m_parentFrame->view()->contentsToWindow(IntPoint(event->pageX(), event->pageY()));

#if PLATFORM(APOLLO)
    if(m_window)
    {
        WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();

        p = impl->viewportToWindow( p );
    }
#endif

    if (event->type() == EventNames::mousemoveEvent) {
        npEvent.type = MotionNotify;
        npEvent.xmotion.x = p.x() - frameGeometry().x();
        npEvent.xmotion.y = p.y() - frameGeometry().y();
    }
    else if (event->type() == EventNames::mousedownEvent) {
        npEvent.type = ButtonPress;
        npEvent.xbutton.x = p.x() - frameGeometry().x();
        npEvent.xbutton.y = p.y() - frameGeometry().y();
        npEvent.xbutton.button = event->button() + 1;
        npEvent.xbutton.time = gtk_get_current_event_time ();
	if(event->button() == 0)	//left
             m_parentFrame->document()->setFocusedNode(m_element);
    } else if (event->type() == EventNames::mouseupEvent) {
        npEvent.type = ButtonRelease;
        npEvent.xbutton.x = p.x() - frameGeometry().x();
        npEvent.xbutton.y = p.y() - frameGeometry().y();
        npEvent.xbutton.button = event->button() + 1;
        npEvent.xbutton.time = gtk_get_current_event_time ();
    } else 
        return;

    startUserGesture();

    KJS::JSLock::DropAllLocks dropLocks;
    if (!dispatchNPEvent(npEvent))
        event->setDefaultHandled();
}

void PluginView::attachToWindow()
{
    if (m_attachedToWindow)
        return;

    m_attachedToWindow = true;

    // TODO_AIRLinux - Fix this for Unix
/*
    if (m_isVisible && m_window)
        ShowWindow(m_window, SW_SHOWNA);
*/
}

void PluginView::detachFromWindow()
{
    if (!m_attachedToWindow)
        return;

    // TODO_AIRLinux - Fix this for Unix
/*
    if (m_isVisible && m_window)
        ShowWindow(m_window, SW_HIDE);
*/
    m_attachedToWindow = false;
}

void PluginView::setNPWindowRect(const IntRect& rect)
{
    if (!m_isStarted || !m_isVisible || (m_isEnabled && !getApolloImpl()->isInVisibleViewHierarchy()))
        return;

    if(m_isWindowed && !(m_npWindow.window))
        return;

    IntPoint p = m_parentFrame->view()->contentsToWindow(rect.location());

#if PLATFORM(APOLLO)
    if(m_window)
    {
        WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();

        p = impl->viewportToWindow( p );
    }
#endif

    if(m_isWindowed)
    {
        m_npWindow.x = p.x();
        m_npWindow.y = p.y();
    }
    else
    {
        m_npWindow.x = 0;
        m_npWindow.y = 0;
    }

    m_npWindow.width = rect.width();
    m_npWindow.height = rect.height();

    m_npWindow.clipRect.left = 0;
    m_npWindow.clipRect.top = 0;
    m_npWindow.clipRect.right = rect.width();
    m_npWindow.clipRect.bottom = rect.height();

    if (m_plugin->pluginFuncs()->setwindow) {
        KJS::JSLock::DropAllLocks dropAllLocks;
	setCallingPlugin(true);
        m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
	setCallingPlugin(false);
    }
}

void PluginView::stop()
{
    if (!m_isStarted)
        return;
    
    HashSet<RefPtr<PluginStream> > streams = m_streams;
    HashSet<RefPtr<PluginStream> >::iterator end = streams.end();
    for (HashSet<RefPtr<PluginStream> >::iterator it = streams.begin(); it != end; ++it) {
        (*it)->stop();
        disconnectStream((*it).get());
    }

    ASSERT(m_streams.isEmpty());

    m_isStarted = false;

    KJS::JSLock::DropAllLocks dropLocks;

#if 0	// In flash player code (unixnsplayer.cpp), we try to do gtk_widget_show even if the widget is NULL
	// So don't do it as it results in a warning. And anyways, we are destroying the instance in the next step.
    // Clear the window
    m_npWindow.window = 0;
    if (m_plugin->pluginFuncs()->setwindow)
        m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
#endif

    // Destroy the plugin
    setCallingPlugin(true);
    NPError npErr = m_plugin->pluginFuncs()->destroy(m_instance, 0);
    setCallingPlugin(false);
    LOG_NPERROR(npErr); (void)npErr;

    m_instance->pdata = 0;
}

#if ENABLE(NETSCAPE_PLUGIN_API)
const char* PluginView::userAgentStatic()
{
    return 0;
}
#endif

const char* PluginView::userAgent()
{
    if (m_userAgent.isNull())
        m_userAgent = m_parentFrame->loader()->userAgent(m_url).utf8();
    return m_userAgent.data();
}

NPError PluginView::handlePostReadFile(Vector<char>& buffer, uint32 len, const char* buf)
{
    String filename(buf, len);

    if (filename.startsWith("file:///"))
        filename = filename.substring(8);

    // TODO_AIRLinux : Fix the implementation below for Unix

    // Get file info
    /*
    WIN32_FILE_ATTRIBUTE_DATA attrs;
    if (GetFileAttributesExW(filename.charactersWithNullTermination(), GetFileExInfoStandard, &attrs) == 0)
        return NPERR_FILE_NOT_FOUND;

    if (attrs.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
        return NPERR_FILE_NOT_FOUND;

    HANDLE fileHandle = CreateFileW(filename.charactersWithNullTermination(), FILE_READ_DATA, FILE_SHARE_READ, 0, OPEN_EXISTING, 0, 0);

    if (fileHandle == INVALID_HANDLE_VALUE)
        return NPERR_FILE_NOT_FOUND;

    buffer.resize(attrs.nFileSizeLow);

    DWORD bytesRead;
    int retval = ReadFile(fileHandle, buffer.data(), attrs.nFileSizeLow, &bytesRead, 0);

    CloseHandle(fileHandle);

    if (retval == 0 || bytesRead != attrs.nFileSizeLow)
        return NPERR_FILE_NOT_FOUND;
    */

    return NPERR_NO_ERROR;
}

NPError PluginView::getValueStatic(NPNVariable variable, void* value)
{
    return NPERR_GENERIC_ERROR;
}

NPError PluginView::getValue(NPNVariable variable, void* value)
{
    switch (variable) {
        case NPNVSupportsXEmbedBool: {
            *(NPBool*)value = true;
            return NPERR_NO_ERROR;
        }

        case NPNVSupportsWindowless: {
            *(unsigned int*)value = 1;  /* true */
            return NPERR_NO_ERROR;
        }

        case NPNVToolkit: {
            *(unsigned int*)value = 2; /* NPNVGtk2 */
            return NPERR_NO_ERROR;
        }

        case NPNVWindowNPObject: {
            NPObject* windowScriptObject = m_parentFrame->windowScriptNPObject();

            // Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
            if (windowScriptObject)
                _NPN_RetainObject(windowScriptObject);

            void** v = (void**)value;
            *v = windowScriptObject;
            
            return NPERR_NO_ERROR;
        }

        case NPNVPluginElementNPObject: {
            NPObject* pluginScriptObject = 0;

            if (m_element->hasTagName(appletTag) || m_element->hasTagName(embedTag) || m_element->hasTagName(objectTag))
                pluginScriptObject = static_cast<HTMLPlugInElement*>(m_element)->getNPObject();

            // Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
            if (pluginScriptObject)
                _NPN_RetainObject(pluginScriptObject);

            void** v = (void**)value;
            *v = pluginScriptObject;

            return NPERR_NO_ERROR;
        }
        default:
            return NPERR_GENERIC_ERROR;
    }
}

void PluginView::invalidateRect(NPRect* rect)
{
    if (!rect) {
        invalidate();
        return;
    }

    IntRect r(rect->left, rect->top, rect->right - rect->left, rect->bottom - rect->top);

    if (m_isWindowed) {
        GdkRectangle invalidRect = {r.x(), r.y(), r.width(), r.height()};
        gdk_window_invalidate_rect(m_window->window, &invalidRect, false);
    } else {
        if (m_plugin->quirks().contains(PluginQuirkThrottleInvalidate)) {
            m_invalidRects.append(r);
            if (!m_invalidateTimer.isActive())
                m_invalidateTimer.startOneShot(0.001);
        } else
            Widget::invalidateRect(r);
    }
}

void PluginView::invalidateRegion(NPRegion region)
{
    if (m_isWindowed)
        return;

    XRectangle r;
    XClipBox(region, &r);
    NPRect rect = {r.x, r.y, r.x + r.width, r.y + r.height};

    invalidateRect(&rect);
}

void PluginView::forceRedraw()
{
    if (m_isWindowed)
        gdk_window_invalidate_region(m_window->window, NULL, false);
    else
        Widget::invalidate();
}

PluginView::~PluginView()
{
    stop();

    deleteAllValues(m_requests);

    freeStringArray(m_paramNames, m_paramCount);
    freeStringArray(m_paramValues, m_paramCount);

    if (m_window && GTK_IS_WIDGET(m_window))
        gtk_widget_destroy(m_window);

    m_window = 0;

    m_parentFrame->cleanupScriptObjectsForPlugin(this);

    if (m_plugin && !m_plugin->quirks().contains(PluginQuirkDontUnloadPlugin))
        m_plugin->unload();
}

void PluginView::startUserGesture()
{
    bool const userInput = m_plugin->quirks().contains(PluginQuirkDetectUserInput);
    if (userInput) {
       m_doingUserGesture = true;
       m_userGestureTimer.stop();
       m_userGestureTimer.startOneShot(0.25);
    }    
}

void PluginView::startUserGestureForWindowedPlugins()
{
    startUserGesture();
}

void PluginView::userGestureTimerFired(Timer<PluginView>*)
{
    m_doingUserGesture = false;
}

static gboolean
flash_canvas_event_handler (GtkWidget *widget, GdkEventButton* event, gpointer data)
{
    if(data)
    {
        switch (event->type)
        {
            case GDK_KEY_PRESS: 
            case GDK_KEY_RELEASE: 
            case GDK_BUTTON_PRESS: 
            case GDK_BUTTON_RELEASE: 
                ((PluginView*)data)->startUserGestureForWindowedPlugins();
                break;
            default:
                break;
        }
    }
    return false;
}

gboolean
plug_removed_cb (GtkWidget *widget, gpointer data)
{
    return TRUE;
}

static gboolean
flash_canvas_added_to_plug (GtkContainer *container, GtkWidget *widget, gpointer data)
{
    GtkWidget *flashCanvas = widget;

    unsigned int eventfilter = GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK; 
    gtk_widget_add_events (flashCanvas, eventfilter);
    g_signal_connect(G_OBJECT(flashCanvas), "event", G_CALLBACK(flash_canvas_event_handler), (void*) data);

    return true;
}

static gboolean
plug_added_cb (GtkWidget *widget, gpointer data)
{
    if(((GtkSocket*)widget)->plug_widget)
        g_signal_connect(G_OBJECT(((GtkSocket*)widget)->plug_widget), "add", G_CALLBACK(flash_canvas_added_to_plug), (void*) data);
    return true;
}

void PluginView::createWindow()
{
    if(m_isEnabled)
        setNPWindowRect(frameGeometry());
}

void PluginView::init()
{
    if (m_haveInitialized)
        return;
    m_haveInitialized = true;

    if (!m_plugin) {
        ASSERT(m_status == PluginStatusCanNotFindPlugin);
        return;
    }

    if (!m_plugin->load()) {
        m_plugin = 0;
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

    if (!start()) {
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

    if (m_isWindowed) {

        // TODO_AirLinux - Remove this hardcoding of m_isVisible to true
        // This has been done because without showing the gtk_widget, its
        // corresponding  window is not getting created
        m_isVisible = true;

#if XEMBED_SUPPORTED
        GtkContainer *container = GTK_CONTAINER(m_parentFrame->view()->containingWindow());
        if(!container)
        {
            m_npWindow.type = NPWindowTypeWindow;
            m_npWindow.window = 0;
            return;
        }

        m_window = gtk_socket_new();

        gtk_container_add(container, m_window);

        // When a plug is removed/destroyed, its corresponding socket is then destroyed.
        // Another case is when we try to destroy the socket first. That results in
        // destruction of the plug also which in turn again tries to destroy the parent 
        // socket. This way, the socket is tried to destroy twice - resulting into 
        // random crashes. To avoid this, we handle the plug_removed signal here in 
        // which do not do anything (by returning true) thereby preventing the second time 
        // destruction of the socket.
        g_signal_connect(m_window, "plug_removed", G_CALLBACK(plug_removed_cb), NULL);
	g_signal_connect(m_window, "plug_added", G_CALLBACK(plug_added_cb), (void*)this);

        gtk_widget_realize(m_window);
        if (m_isVisible && GTK_IS_WIDGET(m_window))
            gtk_widget_show(m_window);

        g_object_set_data(G_OBJECT(m_window), "kWebPluginViewProperty", this);

        m_npWindow.type = NPWindowTypeWindow;
        m_npWindow.window = (void*)gtk_socket_get_id(GTK_SOCKET(m_window));

        GdkWindow *gdkWindow = ((GtkWidget*)container)->window;

        // ws_info
        m_npWindow.ws_info = (NPSetWindowCallbackStruct *)malloc(sizeof(NPSetWindowCallbackStruct));
        NPSetWindowCallbackStruct *ws = (NPSetWindowCallbackStruct *)m_npWindow.ws_info;

        // fill in window info structure
        ws->type = 0;
        ws->depth = gdk_window_get_visual(gdkWindow)->depth;
        ws->display = GDK_WINDOW_XDISPLAY(gdkWindow);
        ws->visual = GDK_VISUAL_XVISUAL(gdk_window_get_visual(gdkWindow));
        ws->colormap = GDK_COLORMAP_XCOLORMAP(gdk_window_get_colormap(gdkWindow));

#else
        GtkContainer *container = GTK_CONTAINER(m_parentFrame->view()->containingWindow());
        if(!container)
           return;

        m_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

        gtk_widget_set_parent_window(m_window, (container->widget).window);

        if (m_isVisible)
            gtk_window_present(GTK_WINDOW(m_window));

        g_object_set_data(G_OBJECT(m_window), "kWebPluginViewProperty", this);

        m_npWindow.type = NPWindowTypeWindow;
        m_npWindow.window = (void*)GDK_WINDOW_XWINDOW(m_window->window);
#endif

    } else {
        m_npWindow.type = NPWindowTypeDrawable;
        m_npWindow.window = 0;

        m_npWindow.ws_info = (NPSetWindowCallbackStruct *)malloc(sizeof(NPSetWindowCallbackStruct));
        NPSetWindowCallbackStruct *ws = (NPSetWindowCallbackStruct *)m_npWindow.ws_info;

        // fill in window info structure
        GtkWidget *w = m_parentFrame->view()->containingWindow();
        if(w)
        {
            GdkWindow *gdkWindow = w->window;
            ws->type = 0;
            ws->depth = gdk_window_get_visual(gdkWindow)->depth;
            ws->display = GDK_WINDOW_XDISPLAY(gdkWindow);
            ws->visual = GDK_VISUAL_XVISUAL(gdk_window_get_visual(gdkWindow));
            ws->colormap = GDK_COLORMAP_XCOLORMAP(gdk_window_get_colormap(gdkWindow));
         }
         else
         {
            ws->type = 0;
            ws->depth = 24;
            ws->display = XOpenDisplay(0);
            ws->visual = DefaultVisual(ws->display, DefaultScreen(ws->display));
            ws->colormap = DefaultColormap(ws->display, DefaultScreen(ws->display));
         }
    }

    if (!m_plugin->quirks().contains(PluginQuirkDeferFirstSetWindowCall))
        setNPWindowRect(frameGeometry());

    m_status = PluginStatusLoadedSuccessfully;
}

// Use this function to debug the widget hierarchy
void printWidgetHierarchy(GtkWidget *widget)
{
    static int indent = 0;
    fprintf (stderr, "%s (%p) \n", GTK_OBJECT_TYPE_NAME(widget), widget);
    if (GTK_IS_CONTAINER(widget))
    {
        GList *children = gtk_container_get_children(GTK_CONTAINER(widget));
        GList *current_child = NULL;
        indent++;
        if(!children)
            fprintf (stderr, "---------- This GTK container (%p) does not have any children\n", widget);
        for(current_child = children; current_child != NULL; current_child = current_child->next)
        {
            for(int i=0; i<indent*4; i++)
            {
                fprintf(stderr, "-");
            } 
            printWidgetHierarchy((GtkWidget*)(current_child->data));
            fprintf(stderr, "\n");
        }
        g_list_free(children);
        indent--;
    }
    else
       fprintf (stderr, "---------- This is not a GTK container\n");

}


} // namespace WebCore
