/*
 * Copyright (C) 2006, 2007 Apple Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "PluginView.h"

#include "FrameLoadRequest.h"
#include "PlatformString.h"
#include "KURL.h"
#include "RenderLayer.h"
#include "Element.h"
#include "Document.h"
#include "FrameView.h"
#include "Image.h"
#include "GraphicsContext.h"
#include "KeyboardEvent.h"
#include "EventNames.h"
#include "PluginPackage.h"
#include "c_instance.h"
#include "MouseEvent.h"
#include "PlatformMouseEvent.h"
#include "PluginDebug.h"
#include "value.h"
#include "kjs_proxy.h"
#include "kjs_binding.h"
#include "PluginDatabase.h"
#include "FrameLoader.h"
#include "FrameTree.h"
#include "Settings.h"
#include "runtime.h"
#include "NotImplemented.h"
#include "Page.h"
#include "FocusController.h"
#include "npruntime_impl.h"
#include "HTMLNames.h"
#include "HTMLPluginElement.h"
#include "JSDOMWindow.h"
#include <DeprecatedCarbonFunctions/QDDeprecated.h>
#include "runtime_root.h"
#include "Cursor.h"
#include "FrameLoaderClientApollo.h"
#include "PluginCarbonWindowForNSWindow.h"
#include "PluginCarbonWindow.h"
#include "FocusController.h"
#include "PlatformKeyboardEvent.h"
#include "SelectionController.h"

#if PLATFORM(APOLLO)
#include "IScrollViewApolloImpl.h"
#include <WebKitApollo/WebKit.h>
#include <WebKitApollo/WebWindow.h>
#include <WebKitApollo/WebBitmap.h>

namespace WebKitApollo { extern WebKitAPIHostFunctions* g_HostFunctions; }
#endif


namespace WebCore {


namespace {

IntRect contentsToViewport(IScrollViewApolloImpl* const scrollViewImpl, const IntRect& srcRect)
{
    ASSERT(scrollViewImpl);
    IntRect const result(scrollViewImpl->contentsToViewport(srcRect.location()), srcRect.size());
    return result;
}

static const int activeNullTimerPeriodInMilliSeconds = 20;
static const int inActiveNullTimerPeriodInMilliSeconds = 250;


template <typename T1, typename T2>
inline void setAndUpdateDirty(bool& dirty, T1 const newValue, T2& dest) {
    if ((dirty) || (static_cast<T2>(newValue) != dest)) {
        dirty = true;
        dest = newValue;
    }
}
class CGrafPtrPusherPopper
{
public:
    inline CGrafPtrPusherPopper(CGrafPtr const newCGrafPtr)
        : m_savePort(0)
    {
        QDDeprecated::GetPort(&m_savePort);
        QDDeprecated::SetPort(newCGrafPtr);
    }
    
    inline ~CGrafPtrPusherPopper()
    {
        QDDeprecated::SetPort(m_savePort);
    }

private:
    CGrafPtrPusherPopper(const CGrafPtrPusherPopper&);
    CGrafPtrPusherPopper& operator=(const CGrafPtrPusherPopper&);
    CGrafPtr m_savePort;
};

}

IntRect PluginView::getGeometryInCarbonWindow() const
{
    IntRect geometryInContent(frameGeometry());
    ASSERT(m_element);
    ASSERT(m_element->renderer());
    // Take our element and get the clip rect from the enclosing layer.
    RenderLayer* const layer = m_element->renderer()->enclosingLayer();
    ASSERT(layer);
    
    IntRect layerClipRect(layer->childrenClipRect());
    geometryInContent.intersect(layerClipRect);
    
    WTF::PassRefPtr<IWidgetApolloImpl> impl(getApolloImpl());
    
    IScrollViewApolloImpl* const parentScrollView = impl->getParent();
    
    IntPoint const locationInViewPort(parentScrollView->contentsToViewport(geometryInContent.location()));
    IntPoint const locationInWindow(parentScrollView->viewportToWindow(locationInViewPort));
   
    WebWindow* const webWindow = impl->getWindow();
    ASSERT(webWindow);
    NSWindow* const nsWindow = webWindow->m_pVTable->getPlatformWindow(webWindow);
    ASSERT(nsWindow);
    
    return pluginCarbonWindowRectForNSWindowRect(nsWindow, IntRect(locationInWindow, geometryInContent.size()));
}

void PluginView::updateWindow() const
{
    if(!getApolloImpl()->isInVisibleViewHierarchy())
        return;
    
    FrameView* frameView = m_parentFrame->view();
    
    IntRect oldWindowRect = m_windowRect;
    IntRect oldClipRect = m_clipRect;
    
    m_windowRect = IntRect(frameView->contentsToWindow(frameGeometry().location()), frameGeometry().size());
    
#if PLATFORM(APOLLO)
    // in Apollo, ScrollView::contentsToWindow actually gives you viewport coords
    IntPoint p = viewportToWindow( m_windowRect.location() );
    m_windowRect = IntRect(p, m_windowRect.size());
#endif
    
    m_clipRect = windowClipRect();
    m_clipRect.move(-m_windowRect.x(), -m_windowRect.y());    
}
    
void PluginView::setFocus()
{
    // Widget::setFocus calls WebViewImpl::setFocus, and when it sets the focus ends up calling
    // WebNPPluginWidgetImplMac::onFocus, which calls this function again <sigh>
    static bool guard = false;
    
    if(!guard)
    {
        guard = true;
        
        // Focus the plugin
        if (m_plugin) {
            EventRecord carbonEvent;
            carbonEvent.what = getFocusEvent;
            carbonEvent.message = 0;
            carbonEvent.when = TickCount();
            carbonEvent.where.v = 0;
            carbonEvent.where.h = 0;
            carbonEvent.modifiers = 0;
            KJS::JSLock::DropAllLocks dropLocks;

			setCallingPlugin(true);
            m_plugin->pluginFuncs()->event(m_instance, &carbonEvent);
			setCallingPlugin(false);
        }
        
        Widget::setFocus();
        
        guard = false;
    }
}
    
void PluginView::show()
{
    if (!m_plugin)
        return;
    
    const bool visibleStateChanged = (m_isEnabled && !m_isVisible /*&& m_attachedToWindow*/);
    m_isVisible = true;
    
    enableOrSuppress(m_isEnabled);

    if(getApolloImpl()->getParent() && visibleStateChanged)
        invalidate();
}

void PluginView::hide()
{
    if (!m_plugin)
        return;
    
    const bool visibleStateChanged = (m_isEnabled && m_isVisible/* && m_attachedToWindow*/);
    m_isVisible = false;
    
    enableOrSuppress(m_isEnabled);
    if(getApolloImpl()->getParent() && visibleStateChanged)
        invalidate();
}

void PluginView::onBlur()
{
    // Blur the plugin
    if (m_plugin) {
        EventRecord carbonEvent;
        carbonEvent.what = loseFocusEvent;
        carbonEvent.message = 0;
        carbonEvent.when = TickCount();
        carbonEvent.where.v = 0;
        carbonEvent.where.h = 0;
        carbonEvent.modifiers = 0;
        KJS::JSLock::DropAllLocks dropLocks;
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->event(m_instance, &carbonEvent);
        setCallingPlugin(false);
    }
}

IntPoint PluginView::getOriginOfHTMLControlInWebWindow() const
{
    WTF::PassRefPtr<IWidgetApolloImpl> impl(getApolloImpl());
    IScrollViewApolloImpl* rootScrollView = impl->getParent();
    ASSERT(rootScrollView);
    while (rootScrollView->getParent())
        rootScrollView = rootScrollView->getParent();
    ASSERT(!rootScrollView->getParent());
    
    IntPoint const originOfHTMLControlInWindow(rootScrollView->viewportToWindow(IntPoint(0,0)));
    return originOfHTMLControlInWindow;
}

IntPoint PluginView::getOriginOfHTMLControlInCarbonWindow() const
{
    IntPoint const originOfHTMLControlInWindow(getOriginOfHTMLControlInWebWindow());
    IntRect const onePixelRect(originOfHTMLControlInWindow, IntSize(1, 1));
    
    WTF::PassRefPtr<IWidgetApolloImpl> impl(getApolloImpl());
    WebWindow* const webWindow = impl->getWindow();
    ASSERT(webWindow);
    NSWindow* const nsWindow = webWindow->m_pVTable->getPlatformWindow(webWindow);
    ASSERT(nsWindow);
    
    IntRect const onePixelRectInCarbonWindow(pluginCarbonWindowRectForNSWindowRect(nsWindow, onePixelRect));
    return onePixelRectInCarbonWindow.location();
}

WindowRef PluginView::getCarbonWindow() const
{
    WebWindow* const webWindow = getApolloImpl()->getWindow();
    ASSERT(webWindow);
    NSWindow* const nsWindow = webWindow->m_pVTable->getPlatformWindow(webWindow);
    if (nsWindow)
        return pluginCarbonWindowForNSWindow(nsWindow);
    else
        return 0;
}

void PluginView::getCarbonMousePosition(::Point* const mousePoint)
{
    ASSERT(mousePoint);
    GetGlobalMouse(mousePoint);
    if (m_drawingModel == NPDrawingModelQuickDraw) {
        ASSERT(m_npWindow.window == &m_npPort);
        ASSERT(m_npPort.port);
        HIPoint hiMousePoint = { mousePoint->h, mousePoint->v };
        WindowRef const carbonWindow = getCarbonWindow();
        HIPointConvert(&hiMousePoint
                      , kHICoordSpaceScreenPixel  
                      , 0
                      , kHICoordSpaceWindow
                      , carbonWindow);
        IntPoint const htmlControlOriginInCarbonWindow(getOriginOfHTMLControlInCarbonWindow());
        mousePoint->h = static_cast<int>(hiMousePoint.x) - htmlControlOriginInCarbonWindow.x();
        mousePoint->v = static_cast<int>(hiMousePoint.y) - htmlControlOriginInCarbonWindow.y();
        
        CGrafPtrPusherPopper const savePort(m_npPort.port);
        QDDeprecated::SetOrigin(m_npPort.portx, m_npPort.porty);
        
    }
}

// similar to getGeometryInCarbonWindow
IntRect PluginView::clipDirtyRect(const IntRect& dirtyRect) const
{
    // Take our element and get the clip rect from the enclosing layer and frame view.
    
    WTF::RefPtr<IScrollViewApolloImpl> const impl(m_parentFrame->view()->getApolloImpl());
    
    IntRect const dirtyRectClippedToWidget(intersection(dirtyRect, frameGeometry()));
    IntRect const dirtyRectInWebWindow(impl->viewportToWindow(impl->contentsToViewport(dirtyRectClippedToWidget.location()), false), dirtyRectClippedToWidget.size());

    IntPoint const widgetLocationInWebWindow(impl->viewportToWindow(impl->contentsToViewport(frameGeometry().location()), false));

    IntPoint const locationOfHTMLLoaderInWebWindow(getOriginOfHTMLControlInWebWindow());
    IntRect const bitmapSurfaceRectInPluginCoords(IntPoint(locationOfHTMLLoaderInWebWindow.x() - widgetLocationInWebWindow.x(), locationOfHTMLLoaderInWebWindow.y() - widgetLocationInWebWindow.y()), IntSize(m_webBitmapWidth, m_webBitmapHeight));

    RenderLayer* layer = m_element->renderer()->enclosingLayer();
    FrameView* parentView = m_element->document()->view();
    ASSERT(parentView == m_parentFrame->view());
    IntRect layerClipRectInWebWindow(parentView->windowClipRectForLayer(layer, true));
    IntRect layerClipRectInPluginCoords(IntPoint(layerClipRectInWebWindow.x() - widgetLocationInWebWindow.x(), layerClipRectInWebWindow.y() - widgetLocationInWebWindow.y())
                                       , layerClipRectInWebWindow.size());

    IntRect result(IntPoint(dirtyRectInWebWindow.x() - widgetLocationInWebWindow.x(), dirtyRectInWebWindow.y() - widgetLocationInWebWindow.y())
                  , dirtyRectInWebWindow.size());
    
    result.intersect(bitmapSurfaceRectInPluginCoords);
    result.intersect(layerClipRectInPluginCoords);
    
    return result;
}

void PluginView::paintMissingPluginIcon(GraphicsContext* context, const IntRect& rect)
{
    static Image* nullPluginImage;
    if (!nullPluginImage)
        nullPluginImage = Image::loadPlatformResource("nullPlugin");

    IntRect imageRect(frameGeometry().x(), frameGeometry().y(), nullPluginImage->width(), nullPluginImage->height());

    int xOffset = (frameGeometry().width() - imageRect.width()) / 2;
    int yOffset = (frameGeometry().height() - imageRect.height()) / 2;

    imageRect.move(xOffset, yOffset);

    if (!rect.intersects(imageRect))
        return;

    context->drawImage(nullPluginImage, imageRect.location());
}

void PluginView::paint(GraphicsContext* context, const IntRect& rect)
{
    if ((!m_isEnabled) || (!(getApolloImpl()->isInVisibleViewHierarchy())))
        return;
    
    IntRect const geometry(frameGeometry());
    context->save();
    context->clip(geometry);
    
    if (!m_isStarted) {
        // Draw the "missing plugin" image
        paintMissingPluginIcon(context, rect);
    }
    else {
        setNPWindowRect(frameGeometry());
        
        IntRect clipRect = clipDirtyRect(rect);
        
        if (m_drawingModel == NPDrawingModelCoreGraphics) {
            CGRect cgClipRect = CGRectMake(clipRect.x(), clipRect.y(), clipRect.width(), clipRect.height());
            CGContextSaveGState(m_npCGContext.context);
            CGContextClipToRect(m_npCGContext.context, cgClipRect);
        }
        else  {
            ASSERT(m_drawingModel == NPDrawingModelQuickDraw);
            CGrafPtrPusherPopper const savePort(m_npPort.port);
            
            RgnHandle rgn = QDDeprecated::NewRgn();
            QDDeprecated::SetRectRgn(rgn, clipRect.x(), clipRect.y(), clipRect.right(), clipRect.bottom());
            QDDeprecated::SetClip (rgn);
            QDDeprecated::DisposeRgn(rgn);
        }
        
        
        EventRecord carbonEvent;
        carbonEvent.what = updateEvt;
        if (m_drawingModel == NPDrawingModelCoreGraphics) {
            ASSERT(getCarbonWindow() == m_npCGContext.window);
            carbonEvent.message = reinterpret_cast<unsigned long>(m_npCGContext.window);
        } 
        else {
            ASSERT(m_drawingModel == NPDrawingModelQuickDraw);
            carbonEvent.message = reinterpret_cast<unsigned long>(getCarbonWindow());
        }
            
        carbonEvent.when = TickCount();
        carbonEvent.where.v = 0;
        carbonEvent.where.h = 0;
        carbonEvent.modifiers = 0;
        
        KJS::JSLock::DropAllLocks dropLocks;
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->event(m_instance, &carbonEvent);
        setCallingPlugin(false);
        
        if (m_drawingModel == NPDrawingModelCoreGraphics) {
            CGContextRestoreGState(m_npCGContext.context);
#ifndef NDEBUG
			IntPoint p = m_parentFrame->view()->contentsToWindow(frameGeometry().location());
			WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();
			p = impl->viewportToWindow(p, true);
            IntRect clipRectInLoader(clipRect);
            clipRectInLoader.move(p.x(), p.y());
            ASSERT(clipRectInLoader.x() >= 0);
            ASSERT(clipRectInLoader.right() <= static_cast<int>(m_webBitmapWidth));
            ASSERT(clipRectInLoader.y() >= 0);
            ASSERT(clipRectInLoader.bottom() <= static_cast<int>(m_webBitmapHeight));
#endif
            
        }
        else {
            ASSERT(m_drawingModel == NPDrawingModelQuickDraw);
            // QuickDraw does not understand alpha, so we just saturate the alpha channel
            // after we draw.  The FlashPlayer plugin draws then entire bitmape everytime, so
            // I whack all the pixels (not just the update rect).
			IntPoint p = m_parentFrame->view()->contentsToWindow(frameGeometry().location());
			WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();
			p = impl->viewportToWindow(p, true);
            
			IntRect clipRectInLoader(clipRect);
            clipRectInLoader.move(p.x(), p.y());
			ASSERT(clipRectInLoader.x() >= 0);
            ASSERT(clipRectInLoader.right() <= static_cast<int>(m_webBitmapWidth));
            ASSERT(clipRectInLoader.y() >= 0);
            ASSERT(clipRectInLoader.bottom() <= static_cast<int>(m_webBitmapHeight));
            unsigned long const rowOffset = clipRectInLoader.x() * sizeof(uint32_t);
			unsigned char* const pixelBytesForPlugin = reinterpret_cast<unsigned char*>(m_webBitmapPixels) + (clipRectInLoader.y() * m_webBitmapStride) + rowOffset;
			unsigned long numRowsInPlugin = clipRectInLoader.height();
			unsigned long numColsInPlugin = clipRectInLoader.width();
			unsigned char* currRowFirstByte = pixelBytesForPlugin;
			for (unsigned long i = 0; i < numRowsInPlugin; ++i) {
				uint32_t* currPixel = reinterpret_cast<uint32_t*>(currRowFirstByte);
				for (unsigned long j = 0; j < numColsInPlugin; ++j) {
					(*currPixel) |= 0xFF000000;
					++currPixel;
				}
				currRowFirstByte += m_webBitmapStride;
			}
		}
    }
    
    context->restore();
}

void PluginView::nullEventTimerFired(Timer<PluginView>* timer)
{
    // If we are already in the plugin, then don't re-enter it.
    if (isCallingPlugin())
        return;
        
    EventRecord carbonEvent;
    carbonEvent.what = nullEvent;
    carbonEvent.message = 0;
    carbonEvent.when = TickCount();
    
    carbonEvent.where.v = -1;
    carbonEvent.where.h = -1;
    
    ASSERT(m_parentFrame);
    Page* const page = m_parentFrame->page();
    ASSERT(page);
    FocusController* const focusController = page->focusController();
    ASSERT(focusController);
    bool const pageHasFocus = focusController->isActive();
    if (pageHasFocus && getApolloImpl()->isInVisibleViewHierarchy()) {
        //Get the carbon window that really contains us.
        WindowRef const carbonWindowRef = getCarbonWindow();
        if (carbonWindowRef)
            getCarbonMousePosition(&carbonEvent.where);
    }
    carbonEvent.modifiers = GetCurrentKeyModifiers();
    if (!Button())
        carbonEvent.modifiers |= btnState;
    
    // Scope dropLocks to just the call to pluginFuncs()->event(...)
    {
        KJS::JSLock::DropAllLocks dropLocks;
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->event(m_instance, &carbonEvent);
        setCallingPlugin(false);
    }
}

namespace {
    struct WinKeyCodeToMacKeyCodeEntry {
        char const macKeyCode;
    };
    
    WinKeyCodeToMacKeyCodeEntry const windowKeyCodeToMacKeyCode[] = {
          { 0x0a } // 0
        , { 0x7F } // No mac key for windows key 1.
        , { 0x7F } // No mac key for windows key 2.
        , { 0x7F } // No mac key for windows key 3.
        , { 0x7F } // No mac key for windows key 4.
        , { 0x7F } // No mac key for windows key 5.
        , { 0x7F } // No mac key for windows key 6.
        , { 0x7F } // No mac key for windows key 7.
        , { 0x33 } // 8
        , { 0x30 } // 9
        , { 0x7F } // No mac key for windows key 10.
        , { 0x4b } // 11
        , { 0x7F } // No mac key for windows key 12.
        , { 0x24 } // 13
        , { 0x7F } // No mac key for windows key 14.
        , { 0x37 } // 15
        , { 0x38 } // 16
        , { 0x3b } // 17
        , { 0x3a } // 18
        , { 0x7F } // No mac key for windows key 19.
        , { 0x39 } // 20
        , { 0x7F } // No mac key for windows key 21.
        , { 0x7F } // No mac key for windows key 22.
        , { 0x7F } // No mac key for windows key 23.
        , { 0x7F } // No mac key for windows key 24.
        , { 0x7F } // No mac key for windows key 25.
        , { 0x7F } // No mac key for windows key 26.
        , { 0x35 } // 27
        , { 0x7F } // No mac key for windows key 28.
        , { 0x7F } // No mac key for windows key 29.
        , { 0x7F } // No mac key for windows key 30.
        , { 0x7F } // No mac key for windows key 31.
        , { 0x31 } // 32
        , { 0x74 } // 33
        , { 0x79 } // 34
        , { 0x77 } // 35
        , { 0x73 } // 36
        , { 0x7b } // 37
        , { 0x7e } // 38
        , { 0x7c } // 39
        , { 0x7d } // 40
        , { 0x7F } // No mac key for windows key 41.
        , { 0x7F } // No mac key for windows key 42.
        , { 0x7F } // No mac key for windows key 43.
        , { 0x7F } // No mac key for windows key 44.
        , { 0x72 } // 45
        , { 0x75 } // 46
        , { 0x7F } // No mac key for windows key 47.
        , { 0x1d } // 48
        , { 0x12 } // 49
        , { 0x13 } // 50
        , { 0x14 } // 51
        , { 0x15 } // 52
        , { 0x17 } // 53
        , { 0x16 } // 54
        , { 0x1a } // 55
        , { 0x1c } // 56
        , { 0x19 } // 57
        , { 0x7F } // No mac key for windows key 58.
        , { 0x7F } // No mac key for windows key 59.
        , { 0x7F } // No mac key for windows key 60.
        , { 0x7F } // No mac key for windows key 61.
        , { 0x7F } // No mac key for windows key 62.
        , { 0x7F } // No mac key for windows key 63.
        , { 0x7F } // No mac key for windows key 64.
        , { 0x00 } // 65
        , { 0x0b } // 66
        , { 0x08 } // 67
        , { 0x02 } // 68
        , { 0x0e } // 69
        , { 0x03 } // 70
        , { 0x05 } // 71
        , { 0x04 } // 72
        , { 0x22 } // 73
        , { 0x26 } // 74
        , { 0x28 } // 75
        , { 0x25 } // 76
        , { 0x2e } // 77
        , { 0x2d } // 78
        , { 0x1f } // 79
        , { 0x23 } // 80
        , { 0x0c } // 81
        , { 0x0f } // 82
        , { 0x01 } // 83
        , { 0x11 } // 84
        , { 0x20 } // 85
        , { 0x09 } // 86
        , { 0x0d } // 87
        , { 0x07 } // 88
        , { 0x10 } // 89
        , { 0x06 } // 90
        , { 0x7F } // No mac key for windows key 91.
        , { 0x7F } // No mac key for windows key 92.
        , { 0x7F } // No mac key for windows key 93.
        , { 0x7F } // No mac key for windows key 94.
        , { 0x7F } // No mac key for windows key 95.
        , { 0x52 } // 96
        , { 0x53 } // 97
        , { 0x54 } // 98
        , { 0x55 } // 99
        , { 0x56 } // 100
        , { 0x57 } // 101
        , { 0x58 } // 102
        , { 0x59 } // 103
        , { 0x5b } // 104
        , { 0x5c } // 105
        , { 0x43 } // 106
        , { 0x45 } // 107
        , { 0x7F } // No mac key for windows key 108.
        , { 0x4e } // 109
        , { 0x41 } // 110
        , { 0x7F } // No mac key for windows key 111.
        , { 0x7a } // 112
        , { 0x78 } // 113
        , { 0x63 } // 114
        , { 0x76 } // 115
        , { 0x60 } // 116
        , { 0x61 } // 117
        , { 0x62 } // 118
        , { 0x64 } // 119
        , { 0x65 } // 120
        , { 0x6d } // 121
        , { 0x67 } // 122
        , { 0x6f } // 123
        , { 0x69 } // 124
        , { 0x6b } // 125
        , { 0x71 } // 126
        , { 0x7F } // No mac key for windows key 127.
    };
    
    static const size_t numWindowKeyCodeToMacKeyCodeEntries = sizeof(windowKeyCodeToMacKeyCode) / sizeof(WinKeyCodeToMacKeyCodeEntry);
    
}

void PluginView::handleKeyboardEvent(KeyboardEvent* kbEvent)
{
    EventRecord carbonEvent;
    
    const PlatformKeyboardEvent* keyEvent = kbEvent->keyEvent();
    PlatformKeyboardEvent::Type type = keyEvent->type();
    
    // punt on keypress events for now, when we handle them revise TODO internationalization section below
    if( type !=  PlatformKeyboardEvent::KeyUp && type !=  PlatformKeyboardEvent::RawKeyDown)
        return;
    
    String const keyEventText = keyEvent->text();
    int const charCode(keyEventText.isEmpty() ? 0 : keyEventText[0] );
    
    // TODO internationalization!!
    if (charCode < 0x7F) {
        int windowsKeyCode = keyEvent->windowsVirtualKeyCode();
        if (windowsKeyCode >= static_cast<int>(numWindowKeyCodeToMacKeyCodeEntries))
            windowsKeyCode = 0x7F;
        
        ASSERT(windowsKeyCode < static_cast<int>(numWindowKeyCodeToMacKeyCodeEntries));
        unsigned const macKeyCode = windowKeyCodeToMacKeyCode[windowsKeyCode].macKeyCode;
    
        carbonEvent.message = (macKeyCode << 8) | charCode;
        carbonEvent.when = TickCount();
        carbonEvent.where.v = 0;
        carbonEvent.where.h = 0;
        
        // shiftKeyBit, controlKeyBit, optionKeyBit, and cmdKeyBit are defined
        // in /System/Library/Frameworks/Carbon.framework/Versions/A/Frameworks/HIToolbox.framework/Versions/A/Headers/Events.h
        UInt16 const shiftModifier = static_cast<UInt16>(kbEvent->shiftKey()) << shiftKeyBit;
        UInt16 const ctrlModified = static_cast<UInt16>(kbEvent->ctrlKey()) << controlKeyBit;
        UInt16 const altModifier = static_cast<UInt16>(kbEvent->altKey()) << optionKeyBit;
        UInt16 const metaModifier = static_cast<UInt16>(kbEvent->metaKey()) << cmdKeyBit;
        
        carbonEvent.modifiers = shiftModifier | ctrlModified | altModifier | metaModifier;

        carbonEvent.what = (type == PlatformKeyboardEvent::KeyUp) ? keyUp : keyDown;
        
        startUserGesture();
        
        KJS::JSLock::DropAllLocks dropLocks;
        setCallingPlugin(true);
        bool result = m_plugin->pluginFuncs()->event(m_instance, &carbonEvent);
        setCallingPlugin(false);
        
        if(result)
            kbEvent->setDefaultHandled();
    }
    else {
        // totally punt, non-ascii text being entered!!
    }
}

bool PluginView::handleInsertText(const String& text)
{
    ScriptCode script;
    ::TextEncoding encoding;
    OSStatus result;

    script = (ScriptCode) GetScriptManagerVariable(smKeyScript);
    result = UpgradeScriptInfoToTextEncoding(script, kTextLanguageDontCare, kTextRegionDontCare, NULL, &encoding);
    if (result != noErr)
        return false;

    RetainPtr<CFStringRef> str(AdoptCF, text.createCFString());
    CFIndex bufferSize;
    CFIndex length;
    length = CFStringGetBytes(str.get(), CFRangeMake(0, CFStringGetLength(str.get())), encoding, '?', false, NULL, 0, &bufferSize);
    if (length <= 0)
        return false;
    Vector<UInt8> buffer(bufferSize);
    length = CFStringGetBytes(str.get(), CFRangeMake(0, CFStringGetLength(str.get())), encoding, '?', false, buffer.data(), bufferSize, NULL);

    for (CFIndex i=0; i<bufferSize; i++) {
        EventRecord carbonEvent;
        carbonEvent.message = buffer[i];
        carbonEvent.when = TickCount();
        carbonEvent.where.v = 0;
        carbonEvent.where.h = 0;
        carbonEvent.modifiers = 0;
        carbonEvent.what = keyDown;

        startUserGesture();

        KJS::JSLock::DropAllLocks dropLocks;
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->event(m_instance, &carbonEvent);
        
        // on insertText, we don't get keyDown or keyUp events
        carbonEvent.when = TickCount();
        carbonEvent.what = keyUp;
        m_plugin->pluginFuncs()->event(m_instance, &carbonEvent);
        
        setCallingPlugin(false);
    }

    return true;
}

void PluginView::handleMouseEvent(MouseEvent* event)
{
    EventRecord carbonEvent;
    carbonEvent.message = 0;
    carbonEvent.when = TickCount();
    getCarbonMousePosition(&carbonEvent.where);

    
    EventModifiers const ctrlKeyMod = event->ctrlKey() ? controlKey : 0;
    EventModifiers const shiftKeyMod = event->shiftKey() ? shiftKey : 0;
    EventModifiers const cmdKeyMod = event->metaKey() ? cmdKey : 0;
    EventModifiers const optionKeyMod = event->altKey() ? optionKey : 0;
    EventModifiers const btnStateMod = ((event->buttonDown()) && (event->button() == LeftButton)) ? btnState : 0;
    carbonEvent.modifiers = ctrlKeyMod | shiftKeyMod | cmdKeyMod | optionKeyMod | btnStateMod;
    
    if (event->type() == EventNames::mousemoveEvent) {
        carbonEvent.what = nullEvent;
    }
    else if (event->type() == EventNames::mousedownEvent) {
        
        carbonEvent.what = mouseDown;
        ASSERT(event->buttonDown());
        switch (event->button()) {
            case LeftButton:
                break;
            case RightButton:
                // this is what the netscape PI is expecting
                carbonEvent.modifiers |= controlKey;
                break;
            case MiddleButton:
            default:
                return;
                break;
        }
        ASSERT(m_parentFrame);
        ASSERT(m_parentFrame->page());
        ASSERT(m_parentFrame->page()->focusController());
        ASSERT(m_element);
        // We need to make sure we get the focus when we get a mouse down.
        // This does not happen by default because we tell webcore that the default
        // has been handled down below ( because the player told us it handled the event ).
        // Focus the plugin
        if (Page* page = m_parentFrame->page())
            page->focusController()->setFocusedFrame(m_parentFrame);
        m_parentFrame->document()->setFocusedNode(m_element);
        
        ASSERT(m_parentFrame->selectionController());
        
        // Yuck! We need to kick the selection controller in the pants
        // if the plugin is the only thing on the doc that can be selected.
        // This is due to the early out
        // in FocusController::clearSelectionIfNeeded(Frame* oldFocusedFrame, Frame* newFocusedFrame, Node* newFocusedNode).
        // the line in question looks like this:
        // if (selectionStartNode == newFocusedNode || selectionStartNode->isDescendantOf(newFocusedNode) || selectionStartNode->shadowAncestorNode() == newFocusedNode)
        //     return;
        // If we are the only selectable node in the doc, then selectionStartNode will == newFocusedNode.
        // If the code in FocusController::clearSelectionIfNeeded changes we might be able to remove the hack below.
        // Rather than determine if this plugin is the only selectable content
        // in the doc, we'll just always clear the selection when we mouse down
        // on a plugin.
        m_parentFrame->selectionController()->clear();
        
    } else if (event->type() == EventNames::mouseupEvent) {
        carbonEvent.what = mouseUp;
        ASSERT(event->buttonDown());
        switch (event->button()) {
            case LeftButton:
                break;
            case RightButton:
                // this is what the netscape PI is expecting
                carbonEvent.modifiers |= controlKey;
                break;
            case MiddleButton:
            default:
                return;
                break;
        }
    } else 
        return;
    
    startUserGesture();
    
    const NPPluginFuncs* const pluginFuncs = m_plugin->pluginFuncs();
    KJS::JSLock::DropAllLocks dropLocks;
    setCallingPlugin(true);
    bool const eventHandledByPlugin = pluginFuncs->event(m_instance, &carbonEvent);
    setCallingPlugin(false);
    if (eventHandledByPlugin)
        event->setDefaultHandled();
}

#if PLATFORM(APOLLO)
void PluginView::enableOrSuppress(bool enable)
{
	if (!m_plugin)
		return;
    
	const bool suppressionStateChanged = (m_isEnabled != enable);
	m_isEnabled = enable;
    
	bool const enabled = m_isEnabled && m_isVisible /*&& m_attachedToWindow*/;
	if (!enabled) {
		m_nullEventTimer.stop();
	}
	else if (!(m_nullEventTimer.isActive())) {
		setNPWindowRect(frameGeometry());
		m_nullEventTimer.start(0, static_cast<double>(activeNullTimerPeriodInMilliSeconds) / static_cast<double>(1000));
	}
    
	if (getApolloImpl()->getParent() && suppressionStateChanged)
		invalidate();
}

IntPoint PluginView::clientToWindow(const IntPoint &p) const
{
    // XXX - this is probably wrong
    return p;
}
#endif

IntPoint PluginView::viewportToWindow(const IntPoint &pIn) const
{
    IntPoint p = pIn;
    
    WTF::RefPtr<IScrollViewApolloImpl> impl = m_parentFrame->view()->getApolloImpl();
    p = impl->viewportToWindow( p );
    
    return m_isWindowed ? p : clientToWindow(p);
}
    
void PluginView::attachToWindow()
{
    if (m_attachedToWindow)
        return;
    
    // XXX - do we need to do something here?
    
    m_attachedToWindow = true;
}

void PluginView::detachFromWindow()
{
    if (!m_attachedToWindow)
        return;
    
    // XXX - do we need to do something here?
    
    m_attachedToWindow = false;
}
    
void PluginView::setNPWindowRect(const IntRect& rect)
{
    if(rect != frameGeometry())
    {
        ASSERT(0); // not supported yet
    }
    
    if ((!m_isStarted) || (!m_isEnabled) || (!m_isVisible) /*|| (!m_attachedToWindow)*/ || (!(getApolloImpl()->isInVisibleViewHierarchy())))
        return;
    
    bool dirty = false;
    
    IntRect const geometryInCarbonWindow(getGeometryInCarbonWindow());
    IntPoint const locationInCarbonWindow(geometryInCarbonWindow.location());
 
    setAndUpdateDirty(dirty, locationInCarbonWindow.x(), m_npWindow.x);
    setAndUpdateDirty(dirty, locationInCarbonWindow.y(), m_npWindow.y);
    
    int const widthInCarbonWindow =  geometryInCarbonWindow.width();
    int const heightInCarbonWindow =  geometryInCarbonWindow.height();
        
    setAndUpdateDirty(dirty, widthInCarbonWindow, m_npWindow.width);
    setAndUpdateDirty(dirty, heightInCarbonWindow, m_npWindow.height);
    
    
    WebWindow* const webWindow = getApolloImpl()->getWindow();
    WebBitmap* const webBitmap = webWindow->m_pVTable->getBitmapSurface(webWindow);
    void* const bitmapPixels = webBitmap->m_pVTable->getPixelData(webBitmap);
    unsigned long const bitmapWidth = webBitmap->m_pVTable->getWidth(webBitmap);
    unsigned long const bitmapHeight = webBitmap->m_pVTable->getHeight(webBitmap);
    unsigned long const bitmapStride = webBitmap->m_pVTable->getStride(webBitmap);
    ASSERT((bitmapWidth * 4) <= bitmapStride);
    
    setAndUpdateDirty(dirty, m_npWindow.x, m_npWindow.clipRect.left);
    setAndUpdateDirty(dirty, m_npWindow.y, m_npWindow.clipRect.top);
    setAndUpdateDirty(dirty, m_npWindow.clipRect.left + widthInCarbonWindow, m_npWindow.clipRect.right);
    setAndUpdateDirty(dirty, m_npWindow.clipRect.top + heightInCarbonWindow, m_npWindow.clipRect.bottom);
    
    setAndUpdateDirty(dirty, bitmapPixels, m_webBitmapPixels);
    setAndUpdateDirty(dirty, bitmapWidth, m_webBitmapWidth);
    setAndUpdateDirty(dirty, bitmapHeight, m_webBitmapHeight);
    setAndUpdateDirty(dirty, bitmapStride, m_webBitmapStride);
    
    if (!dirty)
        return;
    
    // branch for QDraw
    IntPoint const originOfHTMLControlInWindow(getOriginOfHTMLControlInCarbonWindow());
    IntPoint const originOfPluginInHTMLControl(locationInCarbonWindow.x() - originOfHTMLControlInWindow.x(), locationInCarbonWindow.y() - originOfHTMLControlInWindow.y());
    WindowRef const carbonWindow = getCarbonWindow();
    if (m_drawingModel == NPDrawingModelCoreGraphics) {
        if (m_npCGContext.context)
            CGContextRelease(m_npCGContext.context);
        m_npCGContext.context = 0;

        CGColorSpaceRef const colorSpace = CGColorSpaceCreateDeviceRGB();
        m_npCGContext.context = CGBitmapContextCreate(bitmapPixels, bitmapWidth, bitmapHeight, 8, bitmapStride, colorSpace, kCGImageAlphaPremultipliedFirst | kCGBitmapByteOrder32Host);
        CGColorSpaceRelease(colorSpace);
        
        CGAffineTransform const transformToMakeCGContextUseWindowCoords = { 1, 0, 0, -1, originOfPluginInHTMLControl.x(), bitmapHeight - originOfPluginInHTMLControl.y()};
        CGContextConcatCTM(m_npCGContext.context, transformToMakeCGContextUseWindowCoords);
    
        if (m_npCGContext.window != carbonWindow)
            m_npCGContext.window = getCarbonWindow();
        if (m_npWindow.window != &m_npCGContext)
            m_npWindow.window = &m_npCGContext;
    } 
    else {
        ASSERT(m_drawingModel == NPDrawingModelQuickDraw);
        
        ::Rect boundsRect;
        QDDeprecated::SetRect(&boundsRect, 0, 0, bitmapWidth, bitmapHeight);
        
        GWorldPtr newPort = NULL;
        QDDeprecated::NewPortFromBytes(&newPort, &boundsRect, bitmapPixels, bitmapStride);
        
        if (m_npPort.port != NULL)
            QDDeprecated::DisposePortFromBytes(m_npPort.port);
        
        m_npPort.port = newPort;
        m_npPort.portx = -originOfPluginInHTMLControl.x();
        m_npPort.porty = -originOfPluginInHTMLControl.y();

        m_npWindow.window = &m_npPort;
        
        {
            ::HIPoint carbonWindowOrigin = { 0, 0 };
            HIPointConvert(&carbonWindowOrigin
                          , kHICoordSpaceWindow
                          , carbonWindow
                          , kHICoordSpaceScreenPixel
                          , 0);
                          
            CGrafPtrPusherPopper const savePort(newPort);
            //QDDeprecated::SetOrigin(static_cast<short>(carbonWindowOrigin.x + originOfHTMLControlInWindow.x())
            //                       , static_cast<short>(carbonWindowOrigin.y + originOfHTMLControlInWindow.y()));
        }
        
    }

    if (m_plugin->pluginFuncs()->setwindow) {
        KJS::JSLock::DropAllLocks dropAllLocks;
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
        setCallingPlugin(false);
    }
}

namespace {
class CurrentPluginViewScope {
public:
    CurrentPluginViewScope(PluginView* pluginView) : m_originalPluginView(currentPluginView())
    {
        s_currentPluginView = pluginView;
        ASSERT(pluginView == currentPluginView());
    }
    ~CurrentPluginViewScope()
    {
        ASSERT(s_currentPluginView);
        s_currentPluginView = m_originalPluginView;
        ASSERT(currentPluginView() == m_originalPluginView);
    }
    static inline PluginView* currentPluginView() { return s_currentPluginView; };
private:
    CurrentPluginViewScope(const CurrentPluginViewScope&);
    CurrentPluginViewScope& operator=(const CurrentPluginViewScope&);
    static PluginView* s_currentPluginView;
    PluginView* const m_originalPluginView;
};

PluginView* CurrentPluginViewScope::s_currentPluginView = 0;

}

void PluginView::stop()
{
    m_nullEventTimer.stop();
    
    if (!m_isStarted)
        return;
    
    HashSet<RefPtr<PluginStream> > streams = m_streams;
    HashSet<RefPtr<PluginStream> >::iterator end = streams.end();
    for (HashSet<RefPtr<PluginStream> >::iterator it = streams.begin(); it != end; ++it) {
        (*it)->stop();
        disconnectStream((*it).get());
    }

    ASSERT(m_streams.isEmpty());

    KJS::JSLock::DropAllLocks dropLocks;

    // Clear the window
    m_npWindow.window = 0;
    if (m_plugin->pluginFuncs()->setwindow && !m_plugin->quirks().contains(PluginQuirkDontSetNullWindowHandleOnDestroy)) {
        setCallingPlugin(true);
        m_plugin->pluginFuncs()->setwindow(m_instance, &m_npWindow);
        setCallingPlugin(false);
    }
    
    // Destroy the plugin
    NPSavedData* savedData = 0;
    setCallingPlugin(true);
    NPError npErr = m_plugin->pluginFuncs()->destroy(m_instance, &savedData);
    setCallingPlugin(false);
    LOG_NPERROR(npErr);
    
    if (savedData) {
        if (savedData->buf)
            NPN_MemFree(savedData->buf);
        NPN_MemFree(savedData);        
    }
    
    m_instance->pdata = 0;
}

#if ENABLE(NETSCAPE_PLUGIN_API)
const char* PluginView::userAgentStatic()
{
    return 0;
}
#endif

const char* PluginView::userAgent()
{
#if PLATFORM(APOLLO)
    if (m_plugin->quirks().contains(PluginQuirkWantsSpecialFlashUserAgent)) {
        unsigned long numUTF8BytesInUA = 0;
        const unsigned char* const uaUTF8Bytes = WebKitApollo::g_HostFunctions->getUserAgentForFlashNPP(&numUTF8BytesInUA);
        ASSERT(uaUTF8Bytes);
        ASSERT(uaUTF8Bytes[numUTF8BytesInUA] == '\0');
        return reinterpret_cast<const char*>(uaUTF8Bytes);
    }
    
    if (m_userAgent.isNull())
        m_userAgent = m_parentFrame->loader()->userAgent(m_url).utf8();
    
    return m_userAgent.data();
#endif    
}
    
namespace {
static const off_t maxPostFileSize = 0x2000000; // 500 Megabytes 
}

NPError PluginView::handlePostReadFile(Vector<char>& buffer, uint32 len, const char* buf)
{
    String filename(buf, len);
    
    if (filename.startsWith("file:///"))
        filename = filename.substring(8);
    CString const fileNameUTF8(filename.utf8());
    int const fd = open(fileNameUTF8.data(), O_RDONLY, 0);
    
    if (fd == -1)
        return NPERR_FILE_NOT_FOUND;
    
    // Get file info
    struct stat attrs;
    int const fstatRet = fstat(fd, &attrs);
    if (fstatRet != 0)
        return NPERR_FILE_NOT_FOUND;
    
    // Make sure file is not in fact a directory.
    if (attrs.st_mode & S_IFDIR)
        return NPERR_FILE_NOT_FOUND;
    
    // Make sure file is not too big.  
    if (attrs.st_size > maxPostFileSize)
        return NPERR_FILE_NOT_FOUND;
    size_t const fileSizeAsSizeT = static_cast<size_t>(attrs.st_size);
    buffer.resize(fileSizeAsSizeT);
    
    int const readRet = read(fd, buffer.data(), fileSizeAsSizeT);
    close(fd);

    if (readRet <= 0 || static_cast<size_t>(readRet) != fileSizeAsSizeT)
        return NPERR_FILE_NOT_FOUND;
    
    return NPERR_NO_ERROR;
}

NPError PluginView::getValueStatic(NPNVariable variable, void* value)
{
    return NPERR_GENERIC_ERROR;
}

NPError PluginView::getValue(NPNVariable variable, void* value)
{
    switch (variable) {
#if ENABLE(NETSCAPE_PLUGIN_API)
        case NPNVWindowNPObject: {
            if (m_isJavaScriptPaused)
                return NPERR_GENERIC_ERROR;

            NPObject* windowScriptObject = m_parentFrame->windowScriptNPObject();

            // Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
            if (windowScriptObject)
                _NPN_RetainObject(windowScriptObject);

            void** v = (void**)value;
            *v = windowScriptObject;
            
            return NPERR_NO_ERROR;
        }

        case NPNVPluginElementNPObject: {
            if (m_isJavaScriptPaused)
                return NPERR_GENERIC_ERROR;

            NPObject* pluginScriptObject = 0;

            if (m_element->hasTagName(HTMLNames::appletTag) || m_element->hasTagName(HTMLNames::embedTag) || m_element->hasTagName(HTMLNames::objectTag))
                pluginScriptObject = static_cast<HTMLPlugInElement*>(m_element)->getNPObject();

            // Return value is expected to be retained, as described here: <http://www.mozilla.org/projects/plugin/npruntime.html>
            if (pluginScriptObject)
                _NPN_RetainObject(pluginScriptObject);

            void** v = (void**)value;
            *v = pluginScriptObject;

            return NPERR_NO_ERROR;
        }
#endif
        case NPNVsupportsCoreGraphicsBool:
            *reinterpret_cast<NPBool*>(value) = TRUE;
            return NPERR_NO_ERROR;
        case NPNVsupportsQuickDrawBool:
            *reinterpret_cast<NPBool*>(value) = TRUE;
            return NPERR_NO_ERROR;
        case NPNVsupportsOpenGLBool:
            *reinterpret_cast<NPBool*>(value) = FALSE;
            return NPERR_NO_ERROR;
        case NPNVpluginDrawingModel:
            *reinterpret_cast<NPDrawingModel*>(value) = m_drawingModel;
        default:
            return NPERR_GENERIC_ERROR;
    }
}

void PluginView::invalidateRect(NPRect* rect)
{
    if (!rect) {
        invalidate();
        return;
    }
    
    if (m_isWindowed && m_window)
    {
        ASSERT(0); // need to implement, see windows
        return;
    }
    
    IntRect r(rect->left, rect->top, rect->right - rect->left, rect->bottom - rect->top);
    
    if (m_plugin->quirks().contains(PluginQuirkThrottleInvalidate)) {
        m_invalidRects.append(r);
        if (!m_invalidateTimer.isActive())
            m_invalidateTimer.startOneShot(0.001);
    } else
        Widget::invalidateRect(r);
}

void PluginView::invalidateRegion(NPRegion region)
{
    if (m_isWindowed)
        return;
    
    RgnHandle const qdRegion = reinterpret_cast<RgnHandle>(region);
    ::Rect r;
    QDDeprecated::GetRegionBounds(qdRegion, &r);
    ASSERT(r.left <= r.right);
    ASSERT(r.top <= r.bottom);
    WebCore::IntRect const wcRect(r.left, r.top, r.right - r.left, r.bottom - r.top);

    Widget::invalidateRect(wcRect);
}

void PluginView::forceRedraw()
{
    Widget::invalidate();
}

PluginView::~PluginView()
{
    stop();

    deleteAllValues(m_requests);

    freeStringArray(m_paramNames, m_paramCount);
    freeStringArray(m_paramValues, m_paramCount);

    m_parentFrame->cleanupScriptObjectsForPlugin(this);

    if (m_plugin && !m_plugin->quirks().contains(PluginQuirkDontUnloadPlugin))
        m_plugin->unload();
        
    if (m_npCGContext.context)
        CGContextRelease(m_npCGContext.context);
    if (m_npPort.port)
        QDDeprecated::DisposePortFromBytes(m_npPort.port);
}

void PluginView::startUserGesture()
{
    bool const userInput = m_plugin->quirks().contains(PluginQuirkDetectUserInput);
    if (userInput) {
        m_doingUserGesture = true;
		m_userGestureTimer.stop();
		m_userGestureTimer.startOneShot(0.25);
    }
}

void PluginView::userGestureTimerFired(Timer<PluginView>*)
{
    m_doingUserGesture = false;
}

void PluginView::init()
{
    if (m_haveInitialized)
        return;
    
    m_haveInitialized = true;

    if (!m_plugin) {
        ASSERT(m_status == PluginStatusCanNotFindPlugin);
        return;
    }

    if (!m_plugin->load(m_plugin->quirks().contains(PluginQuirkNeedsCarbonAPITrapsForQD))) {
        m_plugin = 0;
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

    if (!start()) {
        m_status = PluginStatusCanNotLoadPlugin;
        return;
    }

    if (!m_plugin->quirks().contains(PluginQuirkDeferFirstSetWindowCall))
        setNPWindowRect(frameGeometry());

    m_status = PluginStatusLoadedSuccessfully;
}

} // namespace WebCore
