/*
 * Copyright (C) 2006, 2007 Apple Inc.  All rights reserved.
 * Copyright (C) 2008 Collabora, Ltd.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "PluginPackage.h"

#include "CString.h"
#include "MIMETypeRegistry.h"
#include "PluginDatabase.h"
#include "PluginDebug.h"

#include "String.h"
#include "npruntime_impl.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <dlfcn.h>

namespace WebCore {

namespace {
    
    std::auto_ptr<char> cfStringToUTF8(const WTF::RetainPtr<CFStringRef>& cfString)
    {
        size_t const numCharacters = CFStringGetLength(cfString.get());
        size_t const numUTF8Bytes = CFStringGetMaximumSizeForEncoding(numCharacters, kCFStringEncodingUTF8) + 1; // +1 for null terminator.
        std::auto_ptr<char> result;
        if (numUTF8Bytes > 0) {
            result = std::auto_ptr<char>(new char[numUTF8Bytes]);
            ASSERT(result.get());
            Boolean const getCStrRet = CFStringGetCString(cfString.get(), result.get(), numUTF8Bytes, kCFStringEncodingUTF8);
            ASSERT(getCStrRet); (void)getCStrRet;
        }
        return result;
    }
    
    CFStringEncoding encodingForResource(Handle const resource)
    {
        short resRef = HomeResFile(resource);
        if (ResError() != noErr)
            return kCFStringEncodingMacRoman;
        
        // Get the FSRef for the current resource file
        FSRef fref;
        OSStatus error = FSGetForkCBInfo(resRef, 0, 0, 0, 0, &fref, 0);
        if (error != noErr)
            return kCFStringEncodingMacRoman;
        
        WTF::RetainPtr<CFURLRef> const URL(WTF::AdoptCF, CFURLCreateFromFSRef(kCFAllocatorDefault, &fref));
        if (!URL)
            return kCFStringEncodingMacRoman;
        
        WTF::RetainPtr<CFURLRef> const lprojURL(CFURLCreateCopyDeletingLastPathComponent(kCFAllocatorDefault, URL.get()));
        if (!lprojURL)
            return kCFStringEncodingMacRoman;
        
        // Get the lproj directory name
        WTF::RetainPtr<CFStringRef> const extension(WTF::AdoptCF, CFURLCopyPathExtension(lprojURL.get()));
        if ((!extension) || (CFStringCompare(extension.get(), CFSTR("lproj"), kCFCompareCaseInsensitive) != kCFCompareEqualTo))
            return kCFStringEncodingMacRoman;
        
        WTF::RetainPtr<CFURLRef> const lprojWithoutExtension(WTF::AdoptCF, CFURLCreateCopyDeletingPathExtension(kCFAllocatorDefault, lprojURL.get()));
        if (!lprojWithoutExtension)
            return kCFStringEncodingMacRoman;
        
        WTF::RetainPtr<CFStringRef> const lprojName(WTF::AdoptCF, CFURLCopyLastPathComponent(lprojWithoutExtension.get()));
        if (!lprojName)
            return kCFStringEncodingMacRoman;
        
        WTF::RetainPtr<CFStringRef> const locale(WTF::AdoptCF, CFLocaleCreateCanonicalLocaleIdentifierFromString(kCFAllocatorDefault, lprojName.get()));
        if (!locale)
            return kCFStringEncodingMacRoman;
        
        std::auto_ptr<char> const localUTF8(cfStringToUTF8(locale));
        
        LangCode lang;
        RegionCode region;
        error = LocaleStringToLangAndRegionCodes(localUTF8.get(), &lang, &region);
        if (error != noErr)
            return kCFStringEncodingMacRoman;
        
        TextEncoding encoding;
        error = UpgradeScriptInfoToTextEncoding(kTextScriptDontCare, lang, region, 0, &encoding);
        if (error != noErr)
            return kCFStringEncodingMacRoman;
        
        return encoding;
    }
    
    WTF::Vector<String> readResourceStringList(SInt16 stringListId)
    {
        WTF::Vector<String> result;
        Handle const stringListHandle = Get1Resource('STR#', stringListId);
        if (stringListHandle) {
            CFStringEncoding const encodingOfStringResource = encodingForResource(stringListHandle);
            const unsigned char* const stringListStart = reinterpret_cast<const unsigned char*>(*stringListHandle);
            SInt16 listLen = *reinterpret_cast<const SInt16*>(stringListStart);
            if (listLen > 0) {
                result.reserveCapacity(listLen);
                const unsigned char* currStringListPos = stringListStart + sizeof(SInt16);
                for ( unsigned char i = 0 ; i < listLen ; ++i ) {
                    WTF::RetainPtr<CFStringRef> const currString(WTF::AdoptCF, CFStringCreateWithPascalString(kCFAllocatorDefault, currStringListPos, encodingOfStringResource));
                    result.append(String(currString.get()));
                    unsigned char const currStringByteLength = *currStringListPos;
                    currStringListPos += currStringByteLength + 1; //+1 to skip over length byte.
                }
            }
        }
        return result;
    }
    
    static const SInt16 PluginNameOrDescriptionStringNumber = 126;
    static const SInt16 MIMEDescriptionStringNumber = 127;
    static const SInt16 MIMEListStringStringNumber = 128;
    
    class BundleResourcesScope
    {
        public:
            BundleResourcesScope(const WTF::RetainPtr<CFBundleRef>& bundle)
            : m_savedResources(-1)
            , m_module(bundle)
            , m_resources(-1)
            {
                m_savedResources = CurResFile();
                if (ResError() == noErr) {
                    m_resources = CFBundleOpenBundleResourceMap(m_module.get());
                    if (m_resources != -1) {
                        UseResFile(m_resources);
                        if (ResError() == noErr) {
                            ASSERT(isOk());
                        }
                        else {
                            UseResFile(m_savedResources);
                            CFBundleCloseBundleResourceMap(m_module.get(), m_resources);
                            m_resources = -1;
                        }
                    }
                    else {
                        ASSERT(!isOk());
                    }
                }
                else {
                    m_savedResources = -1;
                    ASSERT(!isOk());
                }
            }
            
            ~BundleResourcesScope()
            {
                if (m_savedResources != -1)
                    UseResFile(m_savedResources);
                if (m_resources != -1)
                    CFBundleCloseBundleResourceMap(m_module.get(), m_resources);
            }
            
            bool isOk() { return m_resources != -1; }
            
        private:
            BundleResourcesScope(const BundleResourcesScope&);
            BundleResourcesScope& operator=(const BundleResourcesScope&);
            
            SInt16 m_savedResources;
            WTF::RetainPtr<CFBundleRef> const m_module;
            SInt16 m_resources;
    };
}

    
int PluginPackage::compareFileVersion(const PlatformModuleVersion& compareVersion) const
{
    // return -1, 0, or 1 if plug-in version is less than, equal to, or greater than
    // the passed version
    if (m_moduleVersion != compareVersion)
        return m_moduleVersion > compareVersion ? 1 : -1;
    
    return 0;    
}

bool PluginPackage::isPluginBlacklisted()
{
    return false;
}

void PluginPackage::determineQuirks(const String& mimeType)
{
    // The flash plugin only requests windowless plugins if we return a mozilla user agent
    if (mimeType == "application/x-shockwave-flash") {
        m_quirks.add(PluginQuirkWantsMozillaUserAgent);
        m_quirks.add(PluginQuirkThrottleInvalidate);
#if PLATFORM(APOLLO)
        m_quirks.add(PluginQuirkNeedsCarbonAPITrapsForQD);
        m_quirks.add(PluginQuirkDontSetNullWindowHandleOnDestroy); // XXX - crashes, needs investigation
        m_quirks.add(PluginQuirkHasModalMessageLoop);
#endif
        m_quirks.add(PluginQuirkDetectUserInput);
#if PLATFORM(PPC)
        m_quirks.add(PluginQuirkDoNotAllowWindowedMode);
#endif
    }
    else
    {
        //ASSERT(0); // don't have the rest of the quirks
    }

}
    
bool PluginPackage::fetchInfo()
{
    // m_module may not be loaded yet
    CString pluginPathUTF8 = m_path.utf8();
    WTF::RetainPtr<CFURLRef> flashPluginURL(CFURLCreateFromFileSystemRepresentation(kCFAllocatorDefault, (const UInt8*)pluginPathUTF8.data(), pluginPathUTF8.length(), true));
    WTF::RetainPtr<CFBundleRef> const module(WTF::AdoptCF, CFBundleCreate(kCFAllocatorDefault, flashPluginURL.get()));
    
    BundleResourcesScope resources(module);
    if (!resources.isOk())
        return false;
    
    WTF::Vector<String> const nameDescriptionStrings(readResourceStringList(PluginNameOrDescriptionStringNumber));
    if (nameDescriptionStrings.size() != 2)
        return false;
    ASSERT(nameDescriptionStrings.size() == 2);
    
    m_description = nameDescriptionStrings[0];
    m_name = nameDescriptionStrings[1];
    if (m_name.isNull() || m_description.isNull())
        return false;

    // TODO - XXX - here's where we'd get the version info
    
    if (isPluginBlacklisted())
        return false;
    
    WTF::Vector<String> const mimeList(readResourceStringList(MIMEListStringStringNumber));
    if (mimeList.isEmpty() || ((mimeList.size() % 2) != 0))
        return false;
    WTF::Vector<String> const mimeDescriptionsList(readResourceStringList(MIMEDescriptionStringNumber));
    if (mimeList.size() != (mimeDescriptionsList.size() * 2 ))
        return false;
    
    for (unsigned i = 0; i < mimeDescriptionsList.size(); i++) {
        ASSERT(((i * 2) + 1) <  mimeList.size());
        String const mimeType(mimeList[i * 2]);
        String const extensionsString(mimeList[(i * 2) + 1]);
        WTF::Vector<String> extensions;
        extensionsString.split(',', extensions);
        m_mimeToExtensions.add(mimeType, extensions);
        
        String const mimeDescription(mimeDescriptionsList[i]);
        m_mimeToDescriptions.add(mimeType, mimeDescription);
        
        determineQuirks(mimeType);
    }
    
    return true;
}
    
bool PluginPackage::load(bool const trapCarbonAPI)
{
    if (m_freeLibraryTimer.isActive()) {
        ASSERT(m_module);
        m_freeLibraryTimer.stop();
    } else if (m_isLoaded) {
        if (m_quirks.contains(PluginQuirkDontAllowMultipleInstances))
            return false;
        m_loadCount++;
        return true;
    }
    else {
        CString pluginPathUTF8 = m_path.utf8();
        WTF::RetainPtr<CFURLRef> flashPluginURL(AdoptCF, CFURLCreateFromFileSystemRepresentation(kCFAllocatorDefault, (const UInt8*)pluginPathUTF8.data(), pluginPathUTF8.length(), false));
        m_module = CFBundleCreate(kCFAllocatorDefault, flashPluginURL.get());
        ASSERT(m_module);
        
        if (!CFBundleLoadExecutable(m_module)) {
            CFRelease(m_module);
            m_module = 0;
            return false;
        }
    }
    
    m_isLoaded = true;
    
    NP_GetEntryPointsFuncPtr NP_GetEntryPoints = 0;
    NP_InitializeFuncPtr NP_Initialize = 0;
    NPError npErr;
    
    NP_Initialize = (NP_InitializeFuncPtr)CFBundleGetFunctionPointerForName(m_module, CFSTR("NP_Initialize"));
    NP_GetEntryPoints = (NP_GetEntryPointsFuncPtr)CFBundleGetFunctionPointerForName(m_module, CFSTR("NP_GetEntryPoints"));
    m_NPP_Shutdown = (NPP_ShutdownProcPtr)CFBundleGetFunctionPointerForName(m_module, CFSTR("NP_Shutdown"));
    
    if (!NP_Initialize || !NP_GetEntryPoints || !m_NPP_Shutdown)
        goto abort;
    
    memset(&m_pluginFuncs, 0, sizeof(m_pluginFuncs));
    m_pluginFuncs.size = sizeof(m_pluginFuncs);
    
    npErr = NP_GetEntryPoints(&m_pluginFuncs);
    LOG_NPERROR(npErr);
    if (npErr != NPERR_NO_ERROR)
        goto abort;
    
    m_browserFuncs.size = sizeof (m_browserFuncs);
    m_browserFuncs.version = NP_VERSION_MINOR;
    m_browserFuncs.geturl = NPN_GetURL;
    m_browserFuncs.posturl = NPN_PostURL;
    m_browserFuncs.requestread = NPN_RequestRead;
    m_browserFuncs.newstream = NPN_NewStream;
    m_browserFuncs.write = NPN_Write;
    m_browserFuncs.destroystream = NPN_DestroyStream;
    m_browserFuncs.status = NPN_Status;
    m_browserFuncs.uagent = NPN_UserAgent;
    m_browserFuncs.memalloc = NPN_MemAlloc;
    m_browserFuncs.memfree = NPN_MemFree;
    m_browserFuncs.memflush = NPN_MemFlush;
    m_browserFuncs.reloadplugins = NPN_ReloadPlugins;
    m_browserFuncs.geturlnotify = NPN_GetURLNotify;
    m_browserFuncs.posturlnotify = NPN_PostURLNotify;
    m_browserFuncs.getvalue = NPN_GetValue;
    m_browserFuncs.setvalue = NPN_SetValue;
    m_browserFuncs.invalidaterect = NPN_InvalidateRect;
    m_browserFuncs.invalidateregion = NPN_InvalidateRegion;
    m_browserFuncs.forceredraw = NPN_ForceRedraw;
    m_browserFuncs.getJavaEnv = NPN_GetJavaEnv;
    m_browserFuncs.getJavaPeer = NPN_GetJavaPeer;
    m_browserFuncs.pushpopupsenabledstate = NPN_PushPopupsEnabledState;
    m_browserFuncs.poppopupsenabledstate = NPN_PopPopupsEnabledState;
    
    m_browserFuncs.releasevariantvalue = _NPN_ReleaseVariantValue;
    m_browserFuncs.getstringidentifier = _NPN_GetStringIdentifier;
    m_browserFuncs.getstringidentifiers = _NPN_GetStringIdentifiers;
    m_browserFuncs.getintidentifier = _NPN_GetIntIdentifier;
    m_browserFuncs.identifierisstring = _NPN_IdentifierIsString;
    m_browserFuncs.utf8fromidentifier = _NPN_UTF8FromIdentifier;
    m_browserFuncs.createobject = _NPN_CreateObject;
    m_browserFuncs.retainobject = _NPN_RetainObject;
    m_browserFuncs.releaseobject = _NPN_ReleaseObject;
    m_browserFuncs.invoke = _NPN_Invoke;
    m_browserFuncs.invokeDefault = _NPN_InvokeDefault;
    m_browserFuncs.evaluate = _NPN_Evaluate;
    m_browserFuncs.getproperty = _NPN_GetProperty;
    m_browserFuncs.setproperty = _NPN_SetProperty;
    m_browserFuncs.removeproperty = _NPN_RemoveProperty;
    m_browserFuncs.hasproperty = _NPN_HasProperty;
    m_browserFuncs.hasmethod = _NPN_HasMethod;
    m_browserFuncs.setexception = _NPN_SetException;
    m_browserFuncs.enumerate = _NPN_Enumerate;
    
    npErr = NP_Initialize(&m_browserFuncs);
    LOG_NPERROR(npErr);
    
    if (npErr != NPERR_NO_ERROR)
        goto abort;
    
    m_loadCount++;
    return true;
abort:
    unloadWithoutShutdown();
    return false;
}
    
unsigned PluginPackage::hash() const
{ 
    const unsigned hashCodes[] = {
        m_name.impl()->hash(),
        m_description.impl()->hash(),
        m_mimeToExtensions.size()
    };
    
    return StringImpl::computeHash(reinterpret_cast<const UChar*>(hashCodes), sizeof(hashCodes) / sizeof(UChar));    
}
    
bool PluginPackage::equal(const PluginPackage& a, const PluginPackage& b)
{
    if (a.m_name != b.m_name)
        return false;
    
    if (a.m_description != b.m_description)
        return false;
    
    if (a.m_mimeToExtensions.size() != b.m_mimeToExtensions.size())
        return false;
    
    MIMEToExtensionsMap::const_iterator::Keys end = a.m_mimeToExtensions.end().keys();
    for (MIMEToExtensionsMap::const_iterator::Keys it = a.m_mimeToExtensions.begin().keys(); it != end; ++it) {
        if (!b.m_mimeToExtensions.contains(*it))
            return false;
    }
    
    return true;
}

}
