/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"
#include "Widget.h"

#include "Cursor.h"
#include "GraphicsContext.h"
#include "IntRect.h"
#include "Font.h"
#include "IWidgetApolloImpl.h"
#include "IScrollViewApolloImpl.h"
#include "Frame.h"
#include "ScrollView.h"
#include "WidgetApolloDefaultImpl.h"

namespace WebCore {

	class WidgetPrivate
	{
	public:
		WidgetPrivate(const WTF::RefPtr<IWidgetApolloImpl>& impl)
			:	m_pApolloImpl( impl ),
				m_NeedShowOnAttach( false)
		{
		}

		WTF::RefPtr<IWidgetApolloImpl> m_pApolloImpl;
		bool m_NeedShowOnAttach;
		
	private:
		WidgetPrivate();
	};

Widget::Widget()
    : data(new WidgetPrivate( WidgetApolloDefaultImpl::construct()))
{
}

Widget::~Widget()
{
    if (getApolloImpl()) {
        if (getApolloImpl()->getParent())
            removeFromParent();
        ASSERT(!getApolloImpl()->getParent());
    }
	delete data;
}

void Widget::setClient(WidgetClient* c)
{
    ASSERT(data->m_pApolloImpl);
    data->m_pApolloImpl->setClient( c );
}

WidgetClient* Widget::client() const
{
    ASSERT(data->m_pApolloImpl);
    return data->m_pApolloImpl->client();
}

void Widget::paint(GraphicsContext* pGraphicsContext , const IntRect& dirtyRect )
{
    ASSERT(data->m_pApolloImpl);
    data->m_pApolloImpl->paint( pGraphicsContext, dirtyRect );
}

void Widget::invalidate()
{
    invalidateRect(IntRect(IntPoint(0, 0), size()));
}
void Widget::invalidateRect(const IntRect& dirtyRect)
{
    data->m_pApolloImpl->invalidateRect(dirtyRect);
}

IntRect Widget::frameGeometry() const
{
    ASSERT(data->m_pApolloImpl);
    return data->m_pApolloImpl->frameGeometry();
}

void Widget::setFocus()
{
    ASSERT(data->m_pApolloImpl);
    data->m_pApolloImpl->setFocus();
}

void Widget::setCursor( const Cursor& cursor )
{
    ASSERT(data->m_pApolloImpl);
    data->m_pApolloImpl->setCursor( cursor );
}

void Widget::show()
{
    if ( data->m_pApolloImpl ) {
        data->m_pApolloImpl->show();
    }
    else {
        data->m_NeedShowOnAttach = true;
    }
}

void Widget::removeFromParent()
{
    if (getApolloImpl()) {
        ASSERT(getApolloImpl());
        ASSERT(getApolloImpl()->getParent());
	    // if this widget has mouse capture we need to release it before removing it from the parent
	    if (getApolloImpl()->hasMouseCapture())
		    getApolloImpl()->releaseMouseCapture();

        ScrollView* const parentScrollView = getApolloImpl()->getParent()->scrollView();
        // parentScrollView will be null, even when getApolloImpl()->getParent() is non-null
        // due to order of destruction issues.
        if (parentScrollView)
            getApolloImpl()->getParent()->scrollView()->removeChild(this);
        else
            getApolloImpl()->setParent(0);
    }
}

void Widget::hide()
{
    if ( data->m_pApolloImpl ) {
        data->m_pApolloImpl->hide();
    }
    else {
        data->m_NeedShowOnAttach = false;
    }
}

void Widget::setFrameGeometry(const IntRect &rect)
{
    ASSERT(data->m_pApolloImpl);
    data->m_pApolloImpl->setFrameGeometry( rect );
}

#if ! PLATFORM(APOLLO)
IntPoint Widget::mapFromGlobal(const IntPoint &p) const
{
    ASSERT(data->m_pApolloImpl);
    return data->m_pApolloImpl->mapFromGlobal( p );
}
#endif

bool Widget::isEnabled() const
{
    ASSERT(data->m_pApolloImpl);
    return data->m_pApolloImpl->isEnabled();
}


void Widget::setEnabled( bool bEnabled )
{
    ASSERT(data->m_pApolloImpl);
    data->m_pApolloImpl->setEnabled( bEnabled );
}
void Widget::setIsSelected( bool bSelected )
{
    ASSERT(data->m_pApolloImpl);
    data->m_pApolloImpl->setIsSelected( bSelected );
}

void Widget::attachApolloImpl(WTF::PassRefPtr<IWidgetApolloImpl> pImpl)
{
    data->m_pApolloImpl = pImpl;
    if (data->m_NeedShowOnAttach) {
        data->m_pApolloImpl->show();
        data->m_NeedShowOnAttach = false;
    }
}

WTF::PassRefPtr<IWidgetApolloImpl> Widget::getApolloImpl() const
{
    return data->m_pApolloImpl;
}

#if PLATFORM(WIN_OS)
HWND Widget::containingWindow() const
{
    WebWindow* window = data->m_pApolloImpl->getWindow();

    if(window)
        return window->m_pVTable->getPlatformWindow( window );

    return 0;
}

IntPoint Widget::convertChildToSelf(const Widget* child, const IntPoint& point) const
{
    return IntPoint(point.x() + child->x(), point.y() + child->y());
}

IntPoint Widget::convertSelfToChild(const Widget* child, const IntPoint& point) const
{
    return IntPoint(point.x() - child->x(), point.y() - child->y());
}

#elif PLATFORM(UNIX_OS)

GtkWidget* Widget::containingWindow() const
{
    GtkWidget *cWin = NULL;
    WebWindow* window = data->m_pApolloImpl->getWindow();

    // On Unix, the top level window contains a VBox and that VBox in turn
    // contains a GtkLayout. All the AIR content (except platform menu bar) 
    // is a part of this GtkLayout. Therefore, we need to find the GtkLayout
    // widget here.
    if(window)
    {
        GtkContainer *container = &((window->m_pVTable->getPlatformWindow(window)->bin).container);
	
	if(!container)
		return NULL;

        GList *children = gtk_container_get_children(container);
        if(children)
        {

            GtkWidget *vbox = (GtkWidget*)(children->data);
            GList *vboxChildren = gtk_container_get_children(GTK_CONTAINER(vbox));
            GList *child;
            for(child = vboxChildren; child != NULL; child = child->next)
            {
                GtkWidget *widget = (GtkWidget*)(child->data);
                if(GTK_WIDGET_TYPE(widget) == GTK_TYPE_LAYOUT)
                {
                    cWin = widget;
                    break;
                }
            }
            if(vboxChildren)
                g_list_free(vboxChildren);
        }
        if(children)
            g_list_free(children);
    }

    return cWin;
}

IntPoint Widget::convertChildToSelf(const Widget* child, const IntPoint& point) const
{
    return IntPoint(point.x() + child->x(), point.y() + child->y());
}

IntPoint Widget::convertSelfToChild(const Widget* child, const IntPoint& point) const
{
    return IntPoint(point.x() - child->x(), point.y() - child->y());
}

void Widget::geometryChanged() const
{
}

#endif
}
