/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <config.h>
#include <wtf/Assertions.h>
#include <ScrollView.h>
#include <FloatRect.h>
#include <IntPoint.h>
#include <IntRect.h>
#include <PlatformWheelEvent.h>

#include "IScrollViewApolloImpl.h"


namespace WebCore {


ScrollView::ScrollView()
    :   m_data(0), m_canAttachImpl( true )
{
}

ScrollView::~ScrollView()
{
    while (m_children.size()) {
        HashSet<Widget*>::iterator const iterToFirstChild(m_children.begin());
        ASSERT(iterToFirstChild != m_children.end());
        Widget* const firstChild = *iterToFirstChild;
        ASSERT(firstChild);
        ASSERT(firstChild->getApolloImpl()->getParent() == getApolloImpl());
        ASSERT(firstChild->getApolloImpl()->getParent()->scrollView() == this);
        m_children.remove(iterToFirstChild);
        firstChild->getApolloImpl()->setParent(0);
    }
    ASSERT(m_children.size() == 0);
}

void ScrollView::updateContents(const IntRect& updateRect, bool now)
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    m_pApolloImpl->updateContents(updateRect, now);
}

void ScrollView::update()
{
    IntRect updateRect(contentsX(), contentsY(), visibleWidth(), visibleHeight());
    updateContents(updateRect);
}

int ScrollView::visibleWidth() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->visibleWidth();
}

int ScrollView::visibleHeight() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->visibleHeight();
}

FloatRect ScrollView::visibleContentRect() const
{
    m_canAttachImpl = false;
    float const width = visibleWidth();
    float const height = visibleHeight();
    IntSize const currScrollOffset( scrollOffset() );
    FloatPoint const scrollOffsetAsPoint( currScrollOffset.width(), currScrollOffset.height() );
    FloatRect const visibleContentFloatRect( scrollOffsetAsPoint, FloatSize( width, height ) );
    return visibleContentFloatRect;
}

FloatRect ScrollView::visibleContentRectConsideringExternalScrollers() const
{
    // external scrollers not supported for now
    return visibleContentRect();
}

void ScrollView::setContentsPos(int newX, int newY)
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    m_pApolloImpl->setContentsPos(  newX,
                                    newY );
}

void ScrollView::resizeContents(int w,int h)
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    m_pApolloImpl->resizeContents( w, h );
}

int ScrollView::contentsX() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->contentsX();
}

int ScrollView::contentsY() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->contentsY();
}

int ScrollView::contentsWidth() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->contentsWidth();
}

int ScrollView::contentsHeight() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->contentsHeight();
}

IntPoint ScrollView::windowToContents(const IntPoint& viewportPoint) const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->viewportToContents( viewportPoint );
}

IntPoint ScrollView::contentsToWindow(const IntPoint& contentsPoint) const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->contentsToViewport( contentsPoint );
}

IntSize ScrollView::scrollOffset() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->scrollOffset();
}

void ScrollView::scrollBy(int dx, int dy)
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    m_pApolloImpl->scrollBy( dx, dy );
}

void ScrollView::wheelEvent(PlatformWheelEvent& wheelEvent)
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);

	if(m_pApolloImpl->scrollBy( (-LINE_STEP * wheelEvent.deltaX()), (-LINE_STEP * wheelEvent.deltaY()) ))
        wheelEvent.accept();
}

WebCore::ScrollbarMode ScrollView::hScrollbarMode() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->getHScrollbarMode();
}

WebCore::ScrollbarMode ScrollView::vScrollbarMode() const
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    return m_pApolloImpl->getVScrollbarMode();
}

void ScrollView::suppressScrollbars(bool suppressed, bool repaintOnSuppress)
{
    m_canAttachImpl = false;
    if (m_pApolloImpl)
        m_pApolloImpl->suppressScrollbars(suppressed, repaintOnSuppress);
}

void ScrollView::setHScrollbarMode(ScrollbarMode newMode)
{
    m_canAttachImpl = false;
    if (m_pApolloImpl)
        m_pApolloImpl->setHScrollbarMode(newMode);
}

void ScrollView::setVScrollbarMode(ScrollbarMode newMode)
{
    m_canAttachImpl = false;
    if (m_pApolloImpl)
        m_pApolloImpl->setVScrollbarMode(newMode);
}

void ScrollView::setScrollbarsMode(ScrollbarMode newMode)
{
    m_canAttachImpl = false;
    setHScrollbarMode(newMode);
    setVScrollbarMode(newMode);
}

void ScrollView::setStaticBackground(bool flag)
{
    m_canAttachImpl = false;
    ASSERT(m_pApolloImpl);
    m_pApolloImpl->setStaticBackground(flag);
}

HashSet<Widget*>* ScrollView::children()
{
    return &m_children;
}

void ScrollView::attachApolloImpl(WTF::PassRefPtr<IScrollViewApolloImpl> pApolloImpl)
{
    ASSERT(pApolloImpl);
    ASSERT(m_canAttachImpl);
    m_pApolloImpl = pApolloImpl;
    Widget::attachApolloImpl(m_pApolloImpl);
    ASSERT(m_pApolloImpl);
    
    // When attaching the new scroll view impl we are really completing the construction
    // of the scroll view.  Some initial state needs to be set into the scroll view impl.
    // order is important here.  We must set the content size back to 0 before
    // setting the scrollbar mode to auto.
    setContentsPos(0,0);
    resizeContents(0,0);    
    setScrollbarsMode(ScrollbarAuto);
    setStaticBackground(false);
    suppressScrollbars(false, false);
}

WTF::PassRefPtr<IScrollViewApolloImpl> ScrollView::getApolloImpl() const
{
    m_canAttachImpl = false;
    return m_pApolloImpl;
}

void ScrollView::addChild(Widget* pChildWidget)
{
    m_canAttachImpl = false;
    
    ASSERT(pChildWidget);
    WTF::PassRefPtr<IWidgetApolloImpl> const childWidgetImpl = pChildWidget->getApolloImpl();
    ASSERT(childWidgetImpl);
    
    WTF::PassRefPtr<IScrollViewApolloImpl> const scrollViewImpl = getApolloImpl();
    ASSERT(scrollViewImpl);
    
    if (childWidgetImpl->getParent() != scrollViewImpl) {
        ASSERT(childWidgetImpl->getParent() == 0);
        ASSERT(!m_children.contains(pChildWidget));
        m_children.add(pChildWidget);
        pChildWidget->getApolloImpl()->setParent(scrollViewImpl.get());
    }
    ASSERT(m_children.contains(pChildWidget));
}

void ScrollView::removeChild(Widget* pChildWidget)
{
    m_canAttachImpl = false;
    ASSERT(pChildWidget);
    ASSERT(m_children.contains(pChildWidget));
    ASSERT(pChildWidget->getApolloImpl());
    ASSERT(getApolloImpl());
    ASSERT(pChildWidget->getApolloImpl()->getParent() == getApolloImpl());
    ASSERT(m_children.contains(pChildWidget));
    m_children.remove(pChildWidget);
    ASSERT(!m_children.contains(pChildWidget));
    pChildWidget->getApolloImpl()->setParent(0);
    ASSERT(pChildWidget->getApolloImpl()->getParent() == 0);
    
}

void ScrollView::scrollRectIntoViewRecursively(const IntRect& rect)
{
    m_canAttachImpl = false;
    m_pApolloImpl->scrollRectIntoViewRecursively(rect);
}

bool ScrollView::inWindow() const
{
    m_canAttachImpl = false;
    return m_pApolloImpl->inWindow();
}

PlatformScrollbar* ScrollView::scrollbarUnderMouse(const PlatformMouseEvent& mouseEvent)
{
    m_canAttachImpl = false;
    return m_pApolloImpl->scrollbarUnderMouse(mouseEvent);
}

}
