/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <config.h>
#include "PlatformScrollBar.h"
#include "Cursor.h"
#include <EventListener.h>
#include <Event.h>
#include <MouseEvent.h>
#include <Frame.h>
#include <FrameView.h>
#include <FrameLoader.h>
#include <FrameLoaderClient.h>
#include <EventNames.h>
#include <GraphicsContext.h>
#include <IWidgetApolloImpl.h>
#include <IScrollViewApolloImpl.h>
#include "FrameLoaderClientApollo.h"

// To debug the position of mouse events, you can uncomment the following line.
//#define DEBUG_MOUSE_EVENTS
#if defined(DEBUG_MOUSE_EVENTS)
#include <stdio.h>
#endif

namespace WebCore
{
//=============================================================================

static const int s_DefaultScrollBarWidth = 15;
static const int s_DefaultScrollBarHeight = 15;
static const int s_DefaultScrollBarButtonSize = 15;
static const int s_MinThumbSize = 6;

//=============================================================================

PlatformScrollbar::PlatformScrollbar( ScrollbarClient* client
                                    , ScrollbarOrientation orientation
                                    , ScrollbarControlSize size
                                    , WebCore::Frame* const frame ) 
    : Scrollbar(client, orientation, size)
    , m_scrollTimer(this, &PlatformScrollbar::scrollTimerFired)
    , m_frameView(0)
    , m_mousedownFirstButton(false)
    , m_mouseoverFirstButton(false)
    , m_mousedownSecondButton(false)
    , m_mouseoverSecondButton(false)
    , m_mousedownFirstSlackSpace(false)
    , m_mouseoverFirstSlackSpace(false)
    , m_mousedownSecondSlackSpace(false)
    , m_mouseoverSecondSlackSpace(false)
    , m_mousedownThumb(false)
    , m_mouseoverThumb(false)
    , m_sizeOfThumbAreaInPixels(0.0f)
    , m_thumbClickOffset(0.0f)
    , m_thumbValueInPixels(0.0f)
    , m_thumbMinValueinPixles(0.0f)
    , m_visibleToTotalRatio(0.0f)
    , m_isEnabled(true)
{
    FrameLoaderClientApollo* const pFrameLoaderClientApollo = FrameLoaderClientApollo::clientApollo(frame);
    ASSERT(pFrameLoaderClientApollo);
    
    pFrameLoaderClientApollo->attachScrollbarWidgetImpl( this );
    
    m_frameView = frame->view();
    ASSERT(m_frameView);
}

PlatformScrollbar::~PlatformScrollbar()
{
}

const static double s_scrollTimerInitialInterval = .4;
const static double s_scrollTimerSecondaryInterval = .1;

int PlatformScrollbar::width() const 
{
    if (m_orientation == HorizontalScrollbar)
        return m_scrollbarRect.width();
    return s_DefaultScrollBarWidth;
}
int PlatformScrollbar::height() const
{
    if (m_orientation == VerticalScrollbar)
        return m_scrollbarRect.height();
    return s_DefaultScrollBarHeight;
}

void PlatformScrollbar::setEnabled(bool isEnabled) 
{
    m_isEnabled = isEnabled;
}

void PlatformScrollbar::hitTestImplHelper(  const IntRect& rectToTest,
                                            const IntPoint& position,
                                            bool const isMouseDownEvent,
                                            bool* isOver,
                                            bool* isDown,
                                            bool* preventDefault )
{
    if ( rectToTest.contains( position ) )
    {
        (*isOver) = true;
        if ( isMouseDownEvent )//evnt->type() == EventNames::mousedownEvent )
        {
            (*isDown) = true;

            // prevent the default mouse down behavior in order to 
            // avoid selecting text while the scrollbar thumb is 
            // being dragged
            // (Bug #1500918 Top level HTML: dragging scroll bar 
            //  handle has side effect of selecting HTML text)
            //
            *preventDefault = true;
            //evnt->preventDefault();

            // set the mouse capture for this widget.
            getApolloImpl()->setMouseCapture();
        }
    }
    else
    {
        (*isOver) = false;
    }
}

void PlatformScrollbar::hitTestImpl(    PlatformScrollbar::MouseEventType type,
                                        const IntPoint& pos,
                                        bool* preventDefault )
{
    ASSERT( preventDefault );
    *preventDefault = false;
    if ( type == PlatformScrollbar::MouseEventType_Up )
    {
        m_scrollTimer.stop();

        m_mousedownFirstButton = false;
        m_mousedownSecondButton = false;
        m_mousedownFirstSlackSpace = false;
        m_mousedownSecondSlackSpace = false;
        m_mousedownThumb = false;

        // release the mouse capture for this widget, if we have capture
        bool const hasCapture = getApolloImpl()->hasMouseCapture();
        if (hasCapture)
            getApolloImpl()->releaseMouseCapture();
    }
    
    bool const isMouseDown = type == PlatformScrollbar::MouseEventType_Down;
    hitTestImplHelper( m_firstButtonRect
                     , pos
                     , isMouseDown
                     , &m_mouseoverFirstButton
                     , &m_mousedownFirstButton
                     , preventDefault );
    hitTestImplHelper( m_secondButtonRect
                     , pos
                     , isMouseDown
                     , &m_mouseoverSecondButton
                     , &m_mousedownSecondButton
                     , preventDefault );
    hitTestImplHelper( m_thumbRect
                     , pos
                     , isMouseDown
                     , &m_mouseoverThumb
                     , &m_mousedownThumb
                     , preventDefault );
    hitTestImplHelper( m_firstSlackSpaceRect
                     , pos
                     , isMouseDown
                     , &m_mouseoverFirstSlackSpace
                     , &m_mousedownFirstSlackSpace
                     , preventDefault );
    hitTestImplHelper( m_secondSlackSpaceRect
                     , pos
                     , isMouseDown
                     , &m_mouseoverSecondSlackSpace
                     , &m_mousedownSecondSlackSpace
                     , preventDefault );

    if (m_mousedownThumb)
    {
        float eventValue = (m_orientation == HorizontalScrollbar) ? (pos.x() - m_scrollbarRect.x()) : (pos.y() - m_scrollbarRect.y());
        if ( isMouseDown )
        {
            m_thumbClickOffset = (m_orientation == HorizontalScrollbar) ? (pos.x() - m_thumbRect.x()) : (pos.y() - m_thumbRect.y());
        }
        m_thumbValueInPixels = eventValue - m_thumbClickOffset - m_thumbMinValueinPixles;
    }
    
    scrollIfNeeded();
    
    if ( m_scrollbarRect.contains( pos ) )
    {
        setCursor( pointerCursor() );
    }
}

void PlatformScrollbar::hitTest(MouseEvent* evnt)
{
    ASSERT(evnt);

    PlatformScrollbar::MouseEventType eventType;
    if (evnt->type() == EventNames::mousedownEvent) {
        eventType = PlatformScrollbar::MouseEventType_Down;
    }
    else if (evnt->type() == EventNames::mouseupEvent) {
        eventType = PlatformScrollbar::MouseEventType_Up;
    }
    else if (evnt->type() == EventNames::mousemoveEvent) {
        eventType = PlatformScrollbar::MouseEventType_Move;
    }
    else if (evnt->type() == EventNames::mouseoutEvent) {
        eventType = PlatformScrollbar::MouseEventType_Move;
    }
    else if (evnt->type() == EventNames::mouseoverEvent) {
        eventType = PlatformScrollbar::MouseEventType_Move;
    }
    else if (evnt->type() == EventNames::mousewheelEvent) {
        eventType = PlatformScrollbar::MouseEventType_Move;
    }
    else {
        ASSERT(false);
        eventType = PlatformScrollbar::MouseEventType_Move;
    }

    IntPoint const position(evnt->pageX(), evnt->pageY());
    bool preventDefault = false;
    hitTestImpl(eventType, position, &preventDefault);
    if (preventDefault) {
        evnt->preventDefault();
    }
}

void PlatformScrollbar::scrollIfNeeded()
{
    if (m_mousedownThumb) {
        float newRealScrollValue = (m_thumbValueInPixels / m_sizeOfThumbAreaInPixels) * m_totalSize;
        setValue(newRealScrollValue);
        return;
    }

    // figure out which direction we should scroll.
    //
    ScrollDirection direction = ScrollUp;
    if (m_mousedownFirstButton || m_mousedownFirstSlackSpace ) {
        if ( m_orientation == HorizontalScrollbar )
            direction = ScrollLeft;
        else
            direction = ScrollUp;
    }
    if (m_mousedownSecondButton || m_mousedownSecondSlackSpace)
    {
        if (m_orientation == HorizontalScrollbar)
            direction = ScrollRight;
        else
            direction = ScrollDown;
    }

    // Figure out the scroll granularity.
    //
    ScrollGranularity granularity = ScrollByLine;
    if (m_mousedownFirstSlackSpace || m_mousedownSecondSlackSpace)
        granularity = ScrollByPage;
        
    if (m_mousedownFirstButton || m_mousedownSecondButton)
        granularity = ScrollByLine;


    if  (  (m_mousedownFirstButton && m_mouseoverFirstButton) 
        || (m_mousedownSecondButton && m_mouseoverSecondButton) 
        || (m_mousedownFirstSlackSpace && m_mouseoverFirstSlackSpace) 
        || (m_mousedownSecondSlackSpace && m_mouseoverSecondSlackSpace)) {       
        scroll( direction, granularity );
        if (!m_scrollTimer.isActive())
            m_scrollTimer.startOneShot( s_scrollTimerInitialInterval );
    }

    // this call needs to take care of invalidating the correct rects so that repaints happen.
    //
    Scrollbar::client()->scrollbarIsDirty( this );
}

void PlatformScrollbar::scrollTimerFired( Timer<PlatformScrollbar>* )
{
    scrollIfNeeded();
}

void PlatformScrollbar::paint(GraphicsContext* gCtx, const IntRect& damageRect) 
{

#if 0
    Color scrollAreaColor(0xff, 0xff, 0xff);
    Color scrollAreaBorderColor(0x00, 0x00, 0x00, 0x28);

    Color buttonColor(0xff, 0xff, 0xff);
    Color buttonHoverColor(0x00, 0x00, 0x00, 0x28);

    Color buttonBorderColor(0x00, 0x00, 0x00, 0x28);
    Color buttonBorderHoverColor(0x00, 0x00, 0x00, 0x60);

    Color arrowColor = Color(0x00, 0x00, 0x00);
    Color arrowHoverColor = Color(0x21, 0xa5, 0x21);

    Color thumbColor(0x00, 0x00, 0x00, 0x28);
    Color thumbBorderColor(0x00, 0x00, 0x00, 0x28);
    
    gCtx->setStrokeThickness(1);
    gCtx->setStrokeStyle(SolidStroke);
    
    // Draw the scroll area
    //
    gCtx->setStrokeColor(scrollAreaBorderColor);
    gCtx->setFillColor(scrollAreaColor);
    gCtx->drawRect(m_scrollbarRect);

    // Draw the thumb.
    //
    gCtx->setStrokeColor(thumbBorderColor);
    gCtx->setFillColor(thumbColor);
    gCtx->drawRect(m_thumbRect);

    // Draw the first button.
    //
    if (m_mouseoverFirstButton) {
        gCtx->setStrokeColor(buttonBorderHoverColor);
        gCtx->setFillColor(buttonHoverColor);
    }
    else {
        gCtx->setStrokeColor(buttonBorderColor);
        gCtx->setFillColor(buttonColor);
    }
    gCtx->drawRect(m_firstButtonRect);

    // Draw the arrow for the first button
    //
    if (m_mouseoverFirstButton) {
        gCtx->setStrokeColor(arrowHoverColor);
        gCtx->setFillColor(arrowHoverColor);
    }
    else {
        gCtx->setStrokeColor(arrowColor);
        gCtx->setFillColor(arrowColor);
    }
    gCtx->drawConvexPolygon(3, m_firstArrow, true);

    // Draw the second button
    //
    if (m_mouseoverSecondButton) {
        gCtx->setStrokeColor(buttonBorderHoverColor);
        gCtx->setFillColor(buttonHoverColor);
    }
    else {
        gCtx->setStrokeColor(buttonBorderColor);
        gCtx->setFillColor(buttonColor);
    }
    gCtx->drawRect(m_secondButtonRect);

    // Draw the arrow for the second button.
    //
    if (m_mouseoverSecondButton) {
        gCtx->setStrokeColor(arrowHoverColor);
        gCtx->setFillColor(arrowHoverColor);
    }
    else {
        gCtx->setStrokeColor(arrowColor);
        gCtx->setFillColor(arrowColor );
    }
    gCtx->drawConvexPolygon(3, m_secondArrow, true);
#endif
    static Image* imageHorz = Image::loadPlatformResource( "scrollBarStatesHorz" );
    static Image* imageVert = Image::loadPlatformResource( "scrollBarStatesVert" );

    // set up for vertical 
    Image* image = imageVert;
    IntSize disabledOffset( 48, 0 );
    IntSize downOffset( 32, 0 );
    IntSize overOffset( 16, 0 );

    IntRect firstButtonSrcRect( 0, 0, 15, 15 );
    IntRect firstSlackSpaceSrcRect( 0, 29, 15, 3 );
    IntRect secondSlackSpaceSrcRect( 0, 29, 15, 3 );
    IntRect secondButtonSrcRect( 0, 29, 15, 15 );

    IntRect thumbMiddleSrcRect( 0, 22, 15, 3 );
    IntRect thumbTopSrcRect( 0, 16, 15, 1 );
    IntRect thumbBottomSrcRect( 0, 27, 15, 1 );

    IntRect thumbTopDestRect = m_thumbRect;
    thumbTopDestRect.setHeight( 1 );

    IntRect thumbBottomDestRect = m_thumbRect;
    thumbBottomDestRect.setY( thumbBottomDestRect.bottom() - 1 );
    thumbBottomDestRect.setHeight( 1 );

    IntRect thumbGripSrcRect( 0, 18, 15, 3 );
    IntRect thumbGripDestRect = m_thumbRect;
    thumbGripDestRect.setY( m_thumbRect.y() + ((thumbGripDestRect.height() / 2) - ( 3 / 2 )) );
    thumbGripDestRect.setHeight( 3 );

    
    // adjust for horizontal 
    if ( m_orientation == HorizontalScrollbar )
    {
        image = imageHorz;
        disabledOffset = IntSize( 0, 48 );
        downOffset = IntSize( 0, 32 );
        overOffset = IntSize( 0, 16 );

        firstButtonSrcRect = IntRect( 0, 0, 15, 15 );
        firstSlackSpaceSrcRect = IntRect( 29, 0, 3, 15 );
        secondSlackSpaceSrcRect = IntRect( 29, 0, 3, 15 );
        secondButtonSrcRect = IntRect( 29, 0, 15, 15 );

        thumbMiddleSrcRect = IntRect( 22, 0, 3, 15 );
        thumbTopSrcRect = IntRect( 16, 0, 1, 15 );
        thumbBottomSrcRect = IntRect( 27, 0, 1, 15 );

        thumbTopDestRect = m_thumbRect;
        thumbTopDestRect.setWidth( 1 );

        thumbBottomDestRect = m_thumbRect;
        thumbBottomDestRect.setX( thumbBottomDestRect.right() - 1 );
        thumbBottomDestRect.setWidth( 1 );

        thumbGripSrcRect = IntRect( 18, 0, 3, 15 );
        thumbGripDestRect = m_thumbRect;
        thumbGripDestRect.setX( m_thumbRect.x() + ((thumbGripDestRect.width() / 2) - ( 3 / 2 )) );
        thumbGripDestRect.setWidth( 3 );
    }

    // first button
    if (!m_isEnabled)
        firstButtonSrcRect.move( disabledOffset );
    else if (m_mousedownFirstButton) 
        firstButtonSrcRect.move( downOffset );
    else if (m_mouseoverFirstButton) 
        firstButtonSrcRect.move( overOffset );
    gCtx->drawImage( image, m_firstButtonRect, firstButtonSrcRect ); 

    // first slack space
    if (!m_isEnabled)
        firstSlackSpaceSrcRect.move( disabledOffset );
    else if (m_mousedownFirstSlackSpace) 
        firstSlackSpaceSrcRect.move( downOffset );
    else if (m_mouseoverFirstSlackSpace) 
        firstSlackSpaceSrcRect.move( overOffset );

    if(m_firstSlackSpaceRect.width() && m_firstSlackSpaceRect.height())
        gCtx->drawImage( image, m_firstSlackSpaceRect, firstSlackSpaceSrcRect ); 

    // second slack space
    if (!m_isEnabled)
        secondSlackSpaceSrcRect.move( disabledOffset );
    else if (m_mousedownSecondSlackSpace) 
        secondSlackSpaceSrcRect.move( downOffset );
    else if (m_mouseoverSecondSlackSpace) 
        secondSlackSpaceSrcRect.move( overOffset );

    if(m_secondSlackSpaceRect.width() && m_secondSlackSpaceRect.height())
        gCtx->drawImage( image, m_secondSlackSpaceRect, secondSlackSpaceSrcRect ); 

    // second botton
    if (!m_isEnabled)
        secondButtonSrcRect.move( disabledOffset );
    else if (m_mousedownSecondButton) 
        secondButtonSrcRect.move( downOffset );
    else if (m_mouseoverSecondButton) 
        secondButtonSrcRect.move( overOffset );
    gCtx->drawImage( image, m_secondButtonRect, secondButtonSrcRect ); 

    int horzSlackAreaSize = ((m_secondSlackSpaceRect.x() + m_secondSlackSpaceRect.width()) - m_firstSlackSpaceRect.x());
    int vertSlackAreaSize = ((m_secondSlackSpaceRect.y() + m_secondSlackSpaceRect.height()) - m_firstSlackSpaceRect.y());

    if  (  (m_orientation == HorizontalScrollbar && (horzSlackAreaSize < s_MinThumbSize)) 
        || (m_orientation == VerticalScrollbar && (vertSlackAreaSize < s_MinThumbSize))
        )
    {
        // don't draw crazy small thumbs
    }
    else
    {
        // thumb middle
        if (!m_isEnabled)
            thumbMiddleSrcRect.move( disabledOffset );
        else if ( m_mousedownThumb ) 
            thumbMiddleSrcRect.move( downOffset );
        else if ( m_mouseoverThumb )
            thumbMiddleSrcRect.move( overOffset );
        gCtx->drawImage( image, m_thumbRect, thumbMiddleSrcRect );

        // thumb top
        if (!m_isEnabled)
            thumbTopSrcRect.move( disabledOffset );
        else if ( m_mousedownThumb ) 
            thumbTopSrcRect.move( downOffset );
        else if ( m_mouseoverThumb )
            thumbTopSrcRect.move( overOffset );
        gCtx->drawImage( image, thumbTopDestRect, thumbTopSrcRect );

        // thumb bottom
        if (!m_isEnabled)
            thumbBottomSrcRect.move( disabledOffset );
        else if ( m_mousedownThumb ) 
            thumbBottomSrcRect.move( downOffset );
        else if ( m_mouseoverThumb )
            thumbBottomSrcRect.move( overOffset );
        gCtx->drawImage( image, thumbBottomDestRect, thumbBottomSrcRect );

        if ( m_thumbRect.height() >= 14 )
        {
            if (!m_isEnabled)
                thumbGripSrcRect.move( disabledOffset );
            else if ( m_mousedownThumb ) 
                thumbGripSrcRect.move( downOffset );
            else if ( m_mouseoverThumb )
                thumbGripSrcRect.move( overOffset );
            gCtx->drawImage( image, thumbGripDestRect, thumbGripSrcRect );

        }
    }
}

void PlatformScrollbar::updateThumbProportion() 
{
    if ( m_totalSize )
        m_visibleToTotalRatio = static_cast<float>(m_visibleSize) / static_cast<float>(m_totalSize);

    // Added to fix bug [1547119] - Scrollbar is drawn too far down the page with Apple's XMLHttpRequest Object Demo.
    if ( m_visibleToTotalRatio > 1.0 )
        m_visibleToTotalRatio = 1.0f;
}

void PlatformScrollbar::updateThumbPosition() 
{
}


void PlatformScrollbar::setRect(const IntRect& rect)
{

    // The rect that is passed in is relative to the parent of the rect.
    //
    m_scrollbarRect = rect;
    
//    printf("PlatformScrollbar::setRect: %10p (x:%d y:%d, height:%d, width:%d)\n", this, rect.x(), rect.y(), rect.height(), rect.width());
    
    // calc the positions of the buttons.
    //
    float buttonWidth = s_DefaultScrollBarButtonSize;
    float buttonHeight = s_DefaultScrollBarButtonSize;
    float secondButtonX = 0;
    float secondButtonY = 0;
    if (m_orientation == HorizontalScrollbar) {
        // calc the thumb position.
        //
        m_sizeOfThumbAreaInPixels = m_scrollbarRect.width() - (2.0f *  buttonWidth);
        if (m_sizeOfThumbAreaInPixels <= 0) {
            // shrink the buttons 
            //
            buttonWidth = (m_scrollbarRect.width() / 2.0f); 
            m_sizeOfThumbAreaInPixels = 0;
        }
        m_thumbMinValueinPixles = buttonWidth - 1; //make thumbMinValue zero based.

        buttonHeight = height();
        secondButtonX = m_scrollbarRect.x() + m_scrollbarRect.width() - buttonWidth;
        secondButtonY = m_scrollbarRect.y();

        float leftOfThumb = m_scrollbarRect.x() + buttonWidth;
        if ( m_totalSize )
        {
            leftOfThumb += ((static_cast<float>(m_currentPos)/static_cast<float>(m_totalSize)) * m_sizeOfThumbAreaInPixels);
        }
        float topOfThumb = m_scrollbarRect.y();
        float widthOfThumb = m_sizeOfThumbAreaInPixels * m_visibleToTotalRatio;
        m_thumbRect = IntRect(leftOfThumb, topOfThumb, static_cast<float>(ceil(widthOfThumb)), buttonHeight);
        if ( m_visibleToTotalRatio != 0 && m_thumbRect.width() <= s_MinThumbSize )
        {
            m_thumbRect.setWidth( s_MinThumbSize );
            if ( m_thumbRect.x() + m_thumbRect.width() > secondButtonX )
            {
                m_thumbRect.setX( secondButtonX - m_thumbRect.width() );
            }
        }

#if 0 // may need this for styled scrollbars one day
        // Calc the position of the arrows.
        //
        const float arrowThickness = buttonHeight / 2.0;
        const float arrowLength = buttonWidth / 2.0;
        float yArrowTopEdge = m_scrollbarRect.y() + (buttonHeight - arrowThickness) / 2.0;
        float yArrowBottomEdge = yArrowTopEdge + arrowThickness;
        float yArrowTip = yArrowTopEdge + arrowThickness / 2.0;

        float xArrow1Left = m_scrollbarRect.x() + ((buttonWidth - arrowLength) / 2.0);
        float xArrow1Right = xArrow1Left + arrowLength;

        float xArrow2Left = secondButtonX + ((buttonWidth - arrowLength) / 2.0);
        float xArrow2Right = xArrow2Left + arrowLength;

        m_firstArrow[0] = FloatPoint(xArrow1Left, yArrowTip);
        m_firstArrow[1] = FloatPoint(xArrow1Right, yArrowBottomEdge);
        m_firstArrow[2] = FloatPoint(xArrow1Right, yArrowTopEdge);

        m_secondArrow[0] = FloatPoint(xArrow2Left, yArrowTopEdge);
        m_secondArrow[1] = FloatPoint(xArrow2Left, yArrowBottomEdge);
        m_secondArrow[2] = FloatPoint(xArrow2Right, yArrowTip);
#endif
    }
    else {
        ASSERT(m_orientation == VerticalScrollbar);

        // calc the thumb position.
        //
        m_sizeOfThumbAreaInPixels = m_scrollbarRect.height() - (2.0f *  buttonHeight);
        if (m_sizeOfThumbAreaInPixels <= 0) {
            // shrink the buttons 
            //
            buttonHeight = (m_scrollbarRect.height() / 2); 
            m_sizeOfThumbAreaInPixels = 0;
        }
        m_thumbMinValueinPixles = buttonHeight - 1; //make thumbMinValue zero based. 

        buttonWidth = width();
        secondButtonX = m_scrollbarRect.x();
        secondButtonY = m_scrollbarRect.y() + m_scrollbarRect.height() - buttonHeight;

        float leftOfThumb = m_scrollbarRect.x();
        float topOfThumb = m_scrollbarRect.y() + buttonHeight;
        if ( m_totalSize )
        {
            topOfThumb += ((static_cast<float>(m_currentPos)/static_cast<float>(m_totalSize)) * m_sizeOfThumbAreaInPixels);
        }
        float heightOfThumb = m_sizeOfThumbAreaInPixels * m_visibleToTotalRatio;
        m_thumbRect = IntRect( leftOfThumb, topOfThumb, buttonWidth, static_cast<float>(ceil( heightOfThumb )) );
        if ( m_visibleToTotalRatio != 0 && m_thumbRect.height() <= s_MinThumbSize )
        {
            m_thumbRect.setHeight( s_MinThumbSize );
            if ( m_thumbRect.y() + m_thumbRect.height() > secondButtonY )
            {
                m_thumbRect.setY( secondButtonY - m_thumbRect.height() );
            }
        }

#if 0 // may need this for styled scrollbars one day

        // Calc the position of the arrows.
        //
        const float arrowThickness = buttonWidth / 2.0;
        const float arrowLength = buttonHeight / 2.0;
        float xArrowLeftEdge = m_scrollbarRect.x() + (buttonWidth - arrowThickness) / 2.0;
        float xArrowRightEdge = xArrowLeftEdge + arrowThickness;
        float xArrowTip = xArrowLeftEdge + arrowThickness / 2.0;

        float yArrow1Top = m_scrollbarRect.y() + ((buttonHeight - arrowLength) / 2.0);
        float yArrow1Bottom = yArrow1Top + arrowLength;

        float yArrow2Top = secondButtonY + ((buttonHeight - arrowLength) / 2.0);
        float yArrow2Bottom = yArrow2Top + arrowLength;

        m_firstArrow[0] = FloatPoint(xArrowLeftEdge, yArrow1Bottom);
        m_firstArrow[1] = FloatPoint(xArrowRightEdge, yArrow1Bottom);
        m_firstArrow[2] = FloatPoint(xArrowTip, yArrow1Top);

        m_secondArrow[0] = FloatPoint(xArrowLeftEdge, yArrow2Top);
        m_secondArrow[1] = FloatPoint(xArrowRightEdge, yArrow2Top);
        m_secondArrow[2] = FloatPoint(xArrowTip, yArrow2Bottom);
#endif
    }

    m_firstButtonRect = IntRect(m_scrollbarRect.x(), m_scrollbarRect.y(), buttonWidth, buttonHeight);
    m_secondButtonRect = IntRect(secondButtonX, secondButtonY, buttonWidth, buttonHeight);

    // calc the slack space rects.
    //
    if (m_orientation == HorizontalScrollbar) {
        float firstSlackX = m_firstButtonRect.x() + m_firstButtonRect.width();
        float firstSlackY = m_firstButtonRect.y();
        float firstSlackWidth = m_thumbRect.x() - firstSlackX;
        float firstSlackHeight = m_firstButtonRect.height();

        float secondSlackX = m_thumbRect.x() + m_thumbRect.width();
        float secondSlackY = firstSlackY;
        float secondSlackWidth = m_secondButtonRect.x() - secondSlackX;
        float secondSlackHeight = firstSlackHeight;

        m_firstSlackSpaceRect = IntRect(firstSlackX, firstSlackY, firstSlackWidth, firstSlackHeight);
        m_secondSlackSpaceRect = IntRect(secondSlackX, secondSlackY, secondSlackWidth, secondSlackHeight);
    }
    else {
        ASSERT(m_orientation == VerticalScrollbar);

        float firstSlackX = m_firstButtonRect.x();
        float firstSlackY = m_firstButtonRect.y() + m_firstButtonRect.height();
        float firstSlackWidth = m_firstButtonRect.width();
        float firstSlackHeight = m_thumbRect.y() - firstSlackY;

        float secondSlackX = firstSlackX;
        float secondSlackY = m_thumbRect.y() + m_thumbRect.height();
        float secondSlackWidth = firstSlackWidth;
        float secondSlackHeight = m_secondButtonRect.y() - secondSlackY;

        m_firstSlackSpaceRect = IntRect(firstSlackX, firstSlackY, firstSlackWidth, firstSlackHeight);
        m_secondSlackSpaceRect = IntRect(secondSlackX, secondSlackY, secondSlackWidth, secondSlackHeight);
    }
}

WebCore::PlatformMouseEvent PlatformScrollbar::normalizeMouseEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    // Calculate the amount of translation required in canvas coordinates.
    //
    WTF::PassRefPtr<IWidgetApolloImpl> impl = getApolloImpl();
    IntSize offset( 0, 0 );
    if (impl) {
        IScrollViewApolloImpl* currentScrollView = impl->getParent();
        ASSERT(currentScrollView);
        offset += currentScrollView->scrollOffset();
    }
    
    WebCore::IntPoint const transformedPoint(mouseEvent.pos() + offset);    
    return WebCore::PlatformMouseEvent( transformedPoint,
                                        WebCore::IntPoint( mouseEvent.globalX(), mouseEvent.globalY() ),
                                        mouseEvent.button(),
                                        mouseEvent.eventType(),
                                        mouseEvent.clickCount(),
                                        mouseEvent.shiftKey(),
                                        mouseEvent.ctrlKey(),
                                        mouseEvent.altKey(),
                                        mouseEvent.metaKey(),
                                        mouseEvent.timestamp() );

        
}

void PlatformScrollbar::logMouseEvent( const char * text, const WebCore::PlatformMouseEvent* origEvent, const WebCore::PlatformMouseEvent* transEvent)
{
    ASSERT(text);
    ASSERT(origEvent);
    printf("%8p ", this );
    printf("(%4d, %4d)", origEvent->pos().x(), origEvent->pos().y());
    if ( transEvent )
    {
        printf("=>(%4d, %4d)", transEvent->pos().x(), transEvent->pos().y());
    }
    printf(" - %s\n", text);
}

bool PlatformScrollbar::isDisconnected()
{
    bool isDisconnected = true;
    WTF::PassRefPtr<IWidgetApolloImpl> impl = getApolloImpl();
    if (impl && impl->getParent())
        isDisconnected = false;
    return isDisconnected;
}

bool PlatformScrollbar::handleMousePressEventNormalized(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (isDisconnected())
        return false;

#if defined(DEBUG_MOUSE_EVENTS)
    logMouseEvent("PlatformScrollbar::handleMousePressEventNormalized", &mouseEvent);
#endif
    bool preventDefault = false;
    hitTestImpl(PlatformScrollbar::MouseEventType_Down, mouseEvent.pos(), &preventDefault);
    return preventDefault;
}

bool PlatformScrollbar::handleMouseMoveEvent(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (isDisconnected())
        return false;

    WebCore::PlatformMouseEvent normalizedMouseEvent = normalizeMouseEvent( mouseEvent );
#if defined(DEBUG_MOUSE_EVENTS)
    logMouseEvent("PlatformScrollbar::handleMouseMoveEvent", &mouseEvent, &normalizedMouseEvent);
#endif
    return handleMouseMoveEventNormalized(normalizedMouseEvent);
}

bool PlatformScrollbar::handleMouseMoveEventNormalized(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (isDisconnected())
        return false;

#if defined(DEBUG_MOUSE_EVENTS)
    logMouseEvent( "PlatformScrollbar::handleMouseMoveEventNormalized", &mouseEvent);
#endif
    bool preventDefault = false;
    hitTestImpl(PlatformScrollbar::MouseEventType_Move, mouseEvent.pos(), &preventDefault);
    return preventDefault;
}

bool PlatformScrollbar::handleMouseReleaseEventNormalized(const WebCore::PlatformMouseEvent& mouseEvent)
{
    if (isDisconnected())
        return false;

#if defined(DEBUG_MOUSE_EVENTS)
    logMouseEvent( "PlatformScrollbar::handleMouseReleaseEventNormalized", &mouseEvent);
#endif
    bool preventDefault = false;
    hitTestImpl(PlatformScrollbar::MouseEventType_Up, mouseEvent.pos(), &preventDefault);
    return preventDefault;
}

bool PlatformScrollbar::handleKeyboardEvent( const WebCore::PlatformKeyboardEvent& keyEvent )
{
    return false;
}

bool PlatformScrollbar::handleInsertText(const String&)
{
    return false;
}

}
