/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef IScrollViewImpl_h
#define IScrollViewImpl_h

#include <wtf/Platform.h>

#include "ScrollTypes.h"
#include "IWidgetApolloImpl.h"

#if PLATFORM(APOLLO)
namespace WebCore
{
    class IContextMenuStrings;
    
    class IScrollViewApolloImpl : virtual public IWidgetApolloImpl
    {
    protected:
        virtual ~IScrollViewApolloImpl() {}
        inline IScrollViewApolloImpl(){}
        void scrollRectIntoView(const WebCore::IntRect& rect);
        WebCore::IntRect visibleRect() const;
    public:
        virtual int visibleWidth() const = 0;
        virtual int visibleHeight() const = 0;
        virtual int contentsWidth() const = 0;
        virtual int contentsHeight() const = 0;
        virtual int contentsX() const = 0;
        virtual int contentsY() const = 0;
        virtual IntSize scrollOffset() const = 0;
        virtual bool scrollBy(int dx, int dy) = 0;
        virtual void scrollRectIntoViewRecursively(const IntRect&) = 0;
        
        virtual void setContentsPos(int x, int y) = 0;

        virtual void setVScrollbarMode(ScrollbarMode) = 0;
        virtual void setHScrollbarMode(ScrollbarMode) = 0;

        virtual void suppressScrollbars(bool suppressed, bool repaintOnUnsuppress) = 0;
        
        virtual ScrollbarMode getVScrollbarMode() const = 0;
        virtual ScrollbarMode getHScrollbarMode() const = 0;

        virtual void resizeContents(int w, int h) = 0;
        virtual void updateContents(const IntRect&, bool now = false) = 0;

        virtual IntPoint contentsToViewport(const IntPoint&) = 0;
        virtual IntPoint viewportToContents(const IntPoint&) = 0;
        virtual IntPoint windowToViewport(const IntPoint&) = 0;
        virtual IntPoint viewportToWindow(const IntPoint&, bool htmlWindowCoords = false) = 0;

        virtual void setStaticBackground(bool) = 0;
		virtual WebPlatformWindow getPlatformWindow() = 0;
        virtual bool inWindow() const = 0;
        virtual PlatformScrollbar* scrollbarUnderMouse(const PlatformMouseEvent&) = 0;
        virtual PlatformScrollbar* getVScrollbar() = 0;
        virtual PlatformScrollbar* getHScrollbar() = 0;
        virtual ScrollView* scrollView() const = 0;
    private:
        static IntSize computeScrollOffsetToMakeRectVisible(const IntRect& rect, const IntRect& visibleRect);
    };

    inline IntSize IScrollViewApolloImpl::computeScrollOffsetToMakeRectVisible(const IntRect& rect, const IntRect& visibleRect)
    {
        IntPoint const topLeft(rect.location());
        IntPoint const bottomRight(rect.right(), rect.bottom());
        IntPoint const visTopLeft(visibleRect.location());
        IntPoint const visBottomRight(visibleRect.right(), visibleRect.bottom());
        
        IntSize newScrollOffset(visibleRect.x(), visibleRect.y());

        //see if top edge is not visible
        if (topLeft.y() < visTopLeft.y()) {
            //Top edge is not visible, so set new scroll offset to top edge.
            newScrollOffset.setHeight(topLeft.y());
        }
        else {
            //top edge is visible, see if bottom edge is not visible
            if ( bottomRight.y() > visBottomRight.y() ) {
                //bottom edge is not visible
                if (visibleRect.height() < rect.height()) {
                    //can't fit both edges in visible area, see if request rect fits in visible rect
                    //we prefer the top edge to be visible, but put it at the top of the visible area
                    newScrollOffset.setHeight(topLeft.y());
                }
                else {
                    int const newScrollY = bottomRight.y() - visibleRect.height();
                    //ASSERT that the new scroll offset is ok,
                    //and that the top edge is still visible.
                    ASSERT(newScrollY >= 0);
                    ASSERT(newScrollY <= topLeft.y());
                    newScrollOffset.setHeight(newScrollY);
                }
            }
        }

        //see if left edge is not visible
        if (topLeft.x() < visTopLeft.x()) {
            //Left edge is not visible, so set new scroll offset to left edge.
            newScrollOffset.setWidth(topLeft.x());
        }
        else {
            //left edge is visible, see if right edge is not visible
            if ( bottomRight.x() > visBottomRight.x() ) {
                //right edge is not visible, see if request rect fits in visible rect
                if (visibleRect.width() < rect.width()) {
                    //can't fit both edges in visible area.
                    //we prefer the left edge to be visible, but put it at the left of the visible area
                    newScrollOffset.setWidth(topLeft.x());
                }
                else {
                    int const newScrollX = bottomRight.x() - visibleRect.width();
                    //ASSERT that the new scroll offset is ok,
                    //and that the left edge is still visible.
                    ASSERT(newScrollX >= 0);
                    ASSERT(newScrollX <= topLeft.x());
                    newScrollOffset.setWidth(newScrollX);
                }
            }
        }
        return newScrollOffset;
    }

    inline void IScrollViewApolloImpl::scrollRectIntoView(const WebCore::IntRect& rect)
    {
        IntRect const visRect(visibleRect());
        IntSize const newScrollOffset(IScrollViewApolloImpl::computeScrollOffsetToMakeRectVisible(rect, visRect));
        setContentsPos(newScrollOffset.width(), newScrollOffset.height());
        ASSERT(visibleRect().intersects(rect) || visibleRect().isEmpty()); //assert that new visible rect intersects requested rect.
    }

    inline IntRect IScrollViewApolloImpl::visibleRect() const
    {
        IntRect visRect(0, 0, visibleWidth(), visibleHeight());
        ASSERT(visRect.x() == 0);
        ASSERT(visRect.y() == 0);
        visRect.move(scrollOffset());
        return visRect;
    }
}
#endif //PLATFORM(APOLLO)

#endif
