#include "config.h"
#include "ClipboardApolloHelper.h"

#include "Clipboard.h"

namespace WebCore {

const String ClipboardApolloHelper::TEXT_TYPE("text/plain");
const String ClipboardApolloHelper::HTML_TYPE("text/html");
const String ClipboardApolloHelper::URI_LIST_TYPE("text/uri-list");
const String ClipboardApolloHelper::BITMAP_TYPE("image/x-vnd.adobe.air.bitmap");
const String ClipboardApolloHelper::FILE_LIST_TYPE("application/x-vnd.adobe.air.file-list");

ClipboardApolloHelper::ClipboardApolloHelper(KJS::JSObject* dataTransfer, KJS::ExecState* execState)
    : m_dataTransfer(dataTransfer)
    , m_exec(execState)
{
}

ClipboardApolloHelper::~ClipboardApolloHelper()
{
}

WTF::HashSet<String> ClipboardApolloHelper::types()
{
    KJS::JSLock lock;

    KJS::JSObject* types = m_dataTransfer->get(m_exec, KJS::Identifier("types"))->toObject(m_exec);
    uint32_t length = types->get(m_exec, KJS::Identifier("length"))->toUInt32(m_exec);
    WTF::HashSet<String> result;

    for (uint32_t i=0; i<length; i++) {
        String s(types->get(m_exec, i)->toString(m_exec));
        result.add(s);
    }

    if (m_exec->hadException())
        m_exec->clearException();

    return result;
}

KJS::JSValue* ClipboardApolloHelper::getData(const String& type, bool& success)
{
    KJS::JSLock lock;

    KJS::JSObject* getDataFunction = m_dataTransfer->get(m_exec, KJS::Identifier("getData"))->toObject(m_exec);
    KJS::List args;
    args.append(KJS::jsString(type));
    KJS::JSValue* data = getDataFunction->callAsFunction(m_exec, m_dataTransfer, args);

    success = !m_exec->hadException();
    m_exec->clearException();

    return data;
}

bool ClipboardApolloHelper::setData(const String& type, KJS::JSValue* data)
{
    KJS::JSLock lock;
    KJS::JSObject* setDataFunction = m_dataTransfer->get(m_exec, KJS::Identifier("setData"))->toObject(m_exec);
    KJS::List args;
    args.append(KJS::jsString(type));
    args.append(data);
    setDataFunction->callAsFunction(m_exec, m_dataTransfer, args);

    bool success = !m_exec->hadException();
    m_exec->clearException();

    return success;
}

void ClipboardApolloHelper::clearData(const String& type)
{
    KJS::JSLock lock;

    KJS::JSObject* clearDataFunction = m_dataTransfer->get(m_exec, KJS::Identifier("clearData"))->toObject(m_exec);
    KJS::List args;
    args.append(KJS::jsString(type));
    clearDataFunction->callAsFunction(m_exec, m_dataTransfer, args);
    if (m_exec->hadException())
        m_exec->clearException();
}

void ClipboardApolloHelper::clearAllData()
{
    KJS::JSLock lock;

    KJS::JSObject* clearAllDataFunction = m_dataTransfer->get(m_exec, KJS::Identifier("clearAllData"))->toObject(m_exec);
    KJS::List args;
    clearAllDataFunction->callAsFunction(m_exec, m_dataTransfer, args);
    if (m_exec->hadException())
        m_exec->clearException();
}

DragOperation ClipboardApolloHelper::dragOperationAllowed()
{
    KJS::JSValue* effectAllowedValue = m_dataTransfer->get(m_exec, KJS::Identifier("effectAllowed"));
    if (!effectAllowedValue->isString())
        return DragOperationNone;
    String effectAllowedString(effectAllowedValue->toString(m_exec));
    return Clipboard::dragOpFromIEOp(effectAllowedString);
}

void ClipboardApolloHelper::setDropEffect(const String& dropEffect)
{
    KJS::JSLock lock;
    m_dataTransfer->put(m_exec, KJS::Identifier("dropEffect"), KJS::jsString(dropEffect));
    if (m_exec->hadException())
        m_exec->clearException();
}

void ClipboardApolloHelper::setDragOperation(DragOperation operation)
{
    setDropEffect(Clipboard::IEOpFromDragOp(operation));
}

void ClipboardApolloHelper::setPropagationStopped(bool stopped)
{
    KJS::JSLock lock;
    m_dataTransfer->put(m_exec, KJS::Identifier("propagationStopped"), KJS::jsBoolean(stopped));
    if (m_exec->hadException())
        m_exec->clearException();
}

String ClipboardApolloHelper::getDataString(const String& type, bool& success)
{
    KJS::JSValue* data = getData(type, success);
    if (!success)
        return String();

    if (!data->isString()) {
        success = false;
        return String();
    }

    KJS::JSLock lock;

    String result(data->toString(m_exec));
    success = !m_exec->hadException();
    m_exec->clearException();
    return result;
}

bool ClipboardApolloHelper::setData(const String& type, const String& data)
{
    KJS::JSLock lock;
    return setData(type, KJS::jsString(data));
}

KJS::JSObject* ClipboardApolloHelper::dataTransfer() const
{
    return m_dataTransfer;
}

KJS::ExecState* ClipboardApolloHelper::execState() const
{
    return m_exec;
}

}
