/*
 * Copyright (C) 2007 Adobe Systems Incorporated.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Adobe Systems Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ADOBE SYSTEMS INCORPORATED "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ADOBE SYSTEMS INCORPORATED
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <config.h>
#include <EventHandler.h>

#include <ClipboardApollo.h>
#include <Frame.h>
#include <FrameLoader.h>
#include <FrameLoaderClientApollo.h>
#include <FrameView.h>
#include <IScrollViewApolloImpl.h>
#include <MouseEventWithHitTestResults.h>
#include <PlatformScrollBar.h>
#include "RenderWidget.h"
#include "PlatformKeyboardEvent.h"

namespace WebCore {

inline PlatformMouseEvent transformMouseEvent( const PlatformMouseEvent& incomingEvent
                                             , ScrollView* const parent
                                             , Widget* const childWidget)
{
    ASSERT(parent);
    ASSERT(childWidget);
    IntPoint const incomingEventCanvasLocation(parent->windowToContents(incomingEvent.pos()));
    IntPoint const childWidgetLocation(childWidget->frameGeometry().location());
    IntPoint const mouseLocationInChildWidget(IntPoint(0, 0) + (incomingEventCanvasLocation - childWidgetLocation));
    PlatformMouseEvent const transformedMouseEvent( mouseLocationInChildWidget
                                                  , IntPoint(incomingEvent.globalX(), incomingEvent.globalY())
                                                  , incomingEvent.button()
                                                  , incomingEvent.eventType()
                                                  , incomingEvent.clickCount()
                                                  , incomingEvent.shiftKey()
                                                  , incomingEvent.ctrlKey()
                                                  , incomingEvent.altKey()
                                                  , incomingEvent.metaKey()
                                                  , incomingEvent.timestamp());
    return transformedMouseEvent;
}
                                        

bool EventHandler::passMousePressEventToSubframe(MouseEventWithHitTestResults& mouseEvent, Frame* subframe)
{
    bool result = false;
    ASSERT(subframe);
    ASSERT(m_frame);
    ASSERT(mouseEvent.event().eventType() == MouseEventPressed);
    
    FrameView* const subframeView = subframe->view();
    if (subframeView) {
        WTF::PassRefPtr<IScrollViewApolloImpl> const scrollViewImpl(subframeView->getApolloImpl());
        ASSERT(scrollViewImpl);
        PlatformMouseEvent const transformedMouseEvent(transformMouseEvent(mouseEvent.event(), m_frame->view(), subframeView));
        ASSERT(transformedMouseEvent.eventType() == MouseEventPressed);
        result = scrollViewImpl->handleMousePressEvent(transformedMouseEvent);

        // XXX - this is truly bizarre!  sometimes the subFrame won't say it handled the mouse event but it WILL
        // call focusDocumentView!  This means that when we get out of here, the super frame will say ok, I'll ALSO
        // call focusDocumentView in handleMousePressEvent and cause the subframe to lose focus...argh!  
        // What were the webkit ppl thinking?  So the hack I'm seeing them do is say that the subframe always handles
        // the event, which sounds wrong.
        result = true;
    }

    return result;
}

bool EventHandler::passMouseMoveEventToSubframe(MouseEventWithHitTestResults& mouseEvent, Frame* subframe, HitTestResult*)
{
    bool result = false;
    ASSERT(subframe);
    ASSERT(m_frame);
    ASSERT(mouseEvent.event().eventType() == MouseEventMoved);
    
    FrameView* const subframeView = subframe->view();
    if (subframeView) {
        WTF::PassRefPtr<IScrollViewApolloImpl> const scrollViewImpl(subframeView->getApolloImpl());
        ASSERT(scrollViewImpl);
        PlatformMouseEvent const transformedMouseEvent(transformMouseEvent(mouseEvent.event(), m_frame->view(), subframeView));
        ASSERT(transformedMouseEvent.eventType() == MouseEventMoved);
        result = scrollViewImpl->handleMouseMoveEvent(transformedMouseEvent);

        // XXX - this is truly bizarre!  sometimes the subFrame won't say it handled the mouse event but it WILL
        // call focusDocumentView!  This means that when we get out of here, the super frame will say ok, I'll ALSO
        // call focusDocumentView in handleMousePressEvent and cause the subframe to lose focus...argh!  
        // What were the webkit ppl thinking?  So the hack I'm seeing them do is say that the subframe always handles
        // the event, which sounds wrong.
        result = true;
   }

    return result;
}

bool EventHandler::passMouseReleaseEventToSubframe(MouseEventWithHitTestResults& mouseEvent, Frame* subframe)
{
    bool result = false;
    ASSERT(subframe);
    ASSERT(m_frame);
    ASSERT(mouseEvent.event().eventType() == MouseEventReleased);
    
    FrameView* const subframeView = subframe->view();
    if (subframeView) {
        WTF::PassRefPtr<IScrollViewApolloImpl> const scrollViewImpl(subframeView->getApolloImpl());
        ASSERT(scrollViewImpl);
        PlatformMouseEvent const transformedMouseEvent(transformMouseEvent(mouseEvent.event(), m_frame->view(), subframeView));
        ASSERT(transformedMouseEvent.eventType() == MouseEventReleased);
        result = scrollViewImpl->handleMouseReleaseEvent(transformedMouseEvent);

        // XXX - this is truly bizarre!  sometimes the subFrame won't say it handled the mouse event but it WILL
        // call focusDocumentView!  This means that when we get out of here, the super frame will say ok, I'll ALSO
        // call focusDocumentView in handleMousePressEvent and cause the subframe to lose focus...argh!  
        // What were the webkit ppl thinking?  So the hack I'm seeing them do is say that the subframe always handles
        // the event, which sounds wrong.
        result = true;
    }

    return result;
}

bool EventHandler::passSubframeEventToSubframe(MouseEventWithHitTestResults&, Frame* subframe, HitTestResult*)
{
    return false;
}

bool EventHandler::passMousePressEventToScrollbar(MouseEventWithHitTestResults& mouseEvent, PlatformScrollbar* scrollbar)
{
    ASSERT(scrollbar);
    
    // need to normalize then thunk
    WebCore::PlatformMouseEvent normalizedMouseEvent = scrollbar->normalizeMouseEvent( mouseEvent.event() );
    return scrollbar->handleMousePressEventNormalized(normalizedMouseEvent);
}

bool EventHandler::passWidgetMouseDownEventToWidget(const MouseEventWithHitTestResults& event)
{
    // Figure out which view to send the event to.
    RenderObject* const target = event.targetNode() ? event.targetNode()->renderer() : 0;
    if (!target || !target->isWidget())
        return false;
    
    RenderWidget* const targetRenderWidget = static_cast<RenderWidget*>(target);
    Widget* const targetWidget = targetRenderWidget->widget();
    if (!targetWidget) {
        LOG_ERROR("hit a RenderWidget without a corresponding Widget, means a frame is half-constructed");
        return true;
    }
    
    ASSERT(!targetWidget->isFrameView());
    
    WTF::PassRefPtr<IWidgetApolloImpl> const widgetImpl(targetWidget->getApolloImpl());
    ASSERT(widgetImpl);

    // bail if the widget does not need to see the mouse down from us.
    if (!widgetImpl->widgetNeedsMouseDownFromEventHandler())
        return false;

    if (event.event().clickCount() <= 1) {
        targetWidget->setFocus();
    }
    
    
    PlatformMouseEvent const transformedMouseEvent(transformMouseEvent(event.event(), m_frame->view(), targetWidget));
    ASSERT(transformedMouseEvent.eventType() == MouseEventPressed);
    widgetImpl->handleMousePressEvent(transformedMouseEvent);
    return true;
}

bool EventHandler::passWidgetMouseDownEventToWidget(RenderWidget*)
{
    return false;
}

bool EventHandler::passMouseDownEventToWidget(Widget*)
{
    return false;
}

bool EventHandler::passWheelEventToWidget(PlatformWheelEvent& wheelEvent, Widget* widget)
{
    // Added this code to fix bug [1582796] HTML: mouse wheel doesn't scroll when over an iframe
    // This is a stop gap fix added during the end of M5. We should investigate if there 
    // is more to be done.
    //
    if ( widget->isFrameView() )
    {
        FrameView* frameView = static_cast< FrameView* >( widget );
        WTF::PassRefPtr<IScrollViewApolloImpl> impl = frameView->getApolloImpl( );
        if ( impl )
        {
            return impl->handleWheelEvent( wheelEvent );
        }
    }
    return false;
}

Clipboard* EventHandler::createDraggingClipboard() const
{
    FrameLoaderClientApollo* const loaderClient = FrameLoaderClientApollo::clientApollo(m_frame);
    ASSERT(loaderClient);
    ClipboardApolloHelper* clipboard = loaderClient->createDraggingClipboard();

    return new ClipboardApollo(true, clipboard, ClipboardWritable, m_frame);
}

bool EventHandler::eventActivatedView(const PlatformMouseEvent&) const
{
    return false;
}

void EventHandler::focusDocumentView()
{
    m_frame->view()->setFocus();
}

#if PLATFORM(WIN_OS)
unsigned int EventHandler::s_accessKeyModifiers = PlatformKeyboardEvent::AltKey;
#else
unsigned int EventHandler::s_accessKeyModifiers = PlatformKeyboardEvent::CtrlKey;
#endif

}
