/*
Copyright (C) 2006,2009 Electronic Arts, Inc.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1.  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
2.  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
3.  Neither the name of Electronic Arts, Inc. ("EA") nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY ELECTRONIC ARTS AND ITS CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL ELECTRONIC ARTS OR ITS CONTRIBUTORS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
//-----------------------------------------------------------------------------
// GenerateBulkBuildFiles.cs
//
// NAnt custom task which creates a bulkbuild '.cpp' file for every given
// input fileset.
//
// Required:
//	FileSetName    - Name of fileset containing files to be added to the bulk 'unit'.
//  OutputFilename - Filename for generated CPP.
//
// Optional
//	OutputDir	- This is where the BuildBuild files get generated to.
//
//-----------------------------------------------------------------------------

using NAnt.Core;
using NAnt.Core.Attributes;

using System.Collections;
using System.IO;


namespace EA.GenerateBulkBuildFiles
{
	[TaskName("GenerateBulkBuildFiles")]
	public class generatebulkbuildfiles : Task
	{
		string _fileSetName	=	null;
        string _outputFilename = null;
		string _outputdir = null;

		[TaskAttribute("FileSetName", Required=true)]
		public string FileSetName
		{
			get { return _fileSetName; }
			set { _fileSetName = value; }
		}

        [TaskAttribute("OutputFilename", Required=true)]
        public string OutputFilename
        {
            get { return _outputFilename; }
            set { _outputFilename = value; }
        }

		[TaskAttribute("OutputDir", Required=false)]
		public string OutputDir
		{
			get { return _outputdir; }
			set { _outputdir = value; }
		}

		/// <summary>Execute the task.</summary>
		protected override void ExecuteTask()
		{
			// This is where the output BulkBuild files get generated to.
			if (_outputdir == string.Empty && Project.Properties["package.builddir"] != null)
			{
				_outputdir = Project.Properties["package.builddir"];
			}

			if (_outputdir == null || _outputdir == string.Empty ) {
				_outputdir = System.Environment.CurrentDirectory;
			}
			
			// Create the output directory if it doesn't exist.
			if (!System.IO.Directory.Exists(_outputdir)) 
			{
				System.IO.Directory.CreateDirectory(_outputdir);
			}

			FileSet currentfileset = Project.NamedFileSets[FileSetName];
		
			// Write out the file contents to a string. 
			string fileoutput = "//-- AutoGenerated BulkBuild File.\n";

			foreach(FileItem fi in currentfileset.FileItems)
			{
				fileoutput += "#include \""+ fi.FileName  +"\"\n";
			}
			fileoutput += "//-- AutoGenerated BulkBuild File.\n";

			string outputfilenameFullPath = _outputdir + "\\" + OutputFilename;

			// Do not want to touch the Generated BulkBuild file unless the fileset changes.
			string oldfilecontents = "";
			if (System.IO.File.Exists(outputfilenameFullPath)) {
				System.IO.StreamReader sr = new System.IO.StreamReader(outputfilenameFullPath);
				oldfilecontents = sr.ReadToEnd();
				sr.Close();
			}

			if (oldfilecontents != fileoutput) {
				System.IO.StreamWriter sw = new System.IO.StreamWriter(outputfilenameFullPath);
				sw.Write(fileoutput);
				sw.Close();
				System.Console.WriteLine("{0}Updating '{1}.cpp'", LogPrefix, System.IO.Path.GetFileNameWithoutExtension(OutputFilename));
			}
		}
	}
}
